package com.bizunited.platform.rbac.server.service;

import com.bizunited.platform.rbac.server.vo.UserVo;

/**
 * 这是一个监听器，用于在本地系统的用户信息，发生变化时向开发人员发布这些通知。<br>
 * 包括用户创建时、用户信息修改时（包括用户被禁用或被启用时），以便业务端开发人员根据用户变化的事件，做相关附加逻辑的处理</p>
 * 
 * 例如proxima产品中就有相关场景，在本地系统创建了用户账号后，对proxima_user表中的内容进行添加。</p>
 * 
 * 另外需要注意的是：</br>
 * 1、一旦当实现了该接口的具体类，在处理业务逻辑过程中出现错误，包括用户基本信息添加/修改在内的整个操作都会失败</br>
 * 2、该监听器只对通过UserService调用而发生的用户信息写操作有监听作用，也就是说如果业务开发人员使用了不推荐的调用UserRepositroy的方式产生的用户写操作，将不受该监听器监控
 * @author yinwenjie
 *
 */
public interface NebulaUserVoEventListener {
  /**
   * 当基于nebula业务构建平台构建的系统中，本地用户发生新增时，该监听事件被触发
   * @param newUser 本次新增的用户信息，将以参数的方式被带入。注意，这个参数对象并不包括任何关联信息的记录。
   */
  public void onUserCreate(final UserVo newUser);
  
  /**
   * 当基于nebula业务构建平台构建的系统中，本地用户发生修改时（无论是基本信息修改，还是类似禁用、启用这样的操作），该监听事件被触发</p>
   * 
   * 注意：</br>
   * 1、该事件方法，专门为了对附属信息修改而存在，如要进行的附加处理是类似分析用户信息变化详情这样的操作，则应该使用nebula 责任链功能进行</br>
   * 2、该事件目前不会在用户最近登录事件发生变化时被触发</br>
   * 3、用户和角色、岗位、组织机构的绑定关系发生变化时，该事件不会触发</br>
   * @param currentUser 修改后的用户信息将以参数的形式传入。注意，这个参数对象并不包括任何关联信息的记录。
   */
  public void onUserUpdate(final UserVo currentUser);
}