package com.bizunited.platform.rbac.server.crypto.password;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.crypto.codec.Hex;
import org.springframework.security.crypto.password.PasswordEncoder;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;

/**
 * AES加密方式的密码加密,由于spring提供的密码加密不支持解密，但是在开发过程中（如devops中的与第三方系统对接的账户存储）需要同时用到加密和解密，
 * 特此提供此类来完成对明文密码的加密和解密，采用的AES的CBC加密方式
 * @Author: Paul Chan
 * @Date: 2019-09-11 11:24
 */
public class Aes2PasswordEncoder implements PasswordEncoder {

  private static final Logger LOGGER = LoggerFactory.getLogger(Aes2PasswordEncoder.class);
  /**
   * 加密密码
   */
  private String secret = "abcdefgh12345678";
  /**
   * 加密方式为CBC
   */
  private final String encodeType = "AES/CBC/PKCS5Padding";

  /**
   * 输出是否采用base64
   */
  private boolean encodeHashAsBase64;

  public Aes2PasswordEncoder() {

  }

  /**
   * 有自定义密码的加密
   * @param secret 必须为16位密码
   */
  public Aes2PasswordEncoder(String secret) {
    Validate.notBlank(secret, "加密密码不能为空");
    Validate.isTrue(secret.length() == 16, "密码长度必须为16");
    this.secret = secret;
  }

  @Override
  public String encode(CharSequence rawPassword) {
    Validate.notBlank(rawPassword, "密码不能为空");
    byte[] bytes = rawPassword.toString().getBytes(StandardCharsets.UTF_8);
    byte[] key = secret.getBytes(StandardCharsets.UTF_8);
    SecretKeySpec keySpec = new SecretKeySpec(key, "AES");
    IvParameterSpec iv = new IvParameterSpec(key);
    try {
      Cipher cipher = Cipher.getInstance(encodeType);
      cipher.init(Cipher.ENCRYPT_MODE, keySpec, iv);
      byte[] enBytes = cipher.doFinal(bytes);
      return this.encode(enBytes);
    } catch (NoSuchAlgorithmException | InvalidKeyException | InvalidAlgorithmParameterException
        | NoSuchPaddingException | BadPaddingException | IllegalBlockSizeException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  /**
   * 对byte数据进行base64或Hex编码输出
   * @param bytes
   * @return
   */
  private String encode(byte[] bytes) {
    if(encodeHashAsBase64) {
      return Base64.getEncoder().encodeToString(bytes);
    }
    return String.valueOf(Hex.encode(bytes));
  }

  /**
   * 解析base64编码和Hex编码
   * @param encodedPassword
   * @return
   */
  private byte[] decodeHB(String encodedPassword) {
    if(encodeHashAsBase64) {
      return Base64.getDecoder().decode(encodedPassword);
    }
    return Hex.decode(encodedPassword);
  }

  @Override
  public boolean matches(CharSequence rawPassword, String encodedPassword) {
    if(StringUtils.isBlank(rawPassword) || StringUtils.isBlank(encodedPassword)) {
      return false;
    }
    String encode = encode(rawPassword);
    return encode.equals(encodedPassword);
  }

  /**
   * 加密字符解密成明文
   * @param encodedPassword
   * @return
   */
  public String decode(String encodedPassword) {
    Validate.notBlank(encodedPassword, "密码不能为空");
    byte[] encodedBytes = this.decodeHB(encodedPassword);
    byte[] key = secret.getBytes(StandardCharsets.UTF_8);
    SecretKeySpec keySpec = new SecretKeySpec(key, "AES");
    try {
      Cipher cipher = Cipher.getInstance(encodeType);
      IvParameterSpec iv = new IvParameterSpec(key);
      cipher.init(Cipher.DECRYPT_MODE, keySpec, iv);
      byte[] bytes = cipher.doFinal(encodedBytes);
      return new String(bytes, StandardCharsets.UTF_8);
    } catch (NoSuchAlgorithmException|InvalidKeyException|InvalidAlgorithmParameterException|NoSuchPaddingException
        |BadPaddingException|IllegalBlockSizeException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  /**
   * 设置hash编码是否是base64
   * @param encodeHashAsBase64
   */
  public void setEncodeHashAsBase64(boolean encodeHashAsBase64) {
    this.encodeHashAsBase64 = encodeHashAsBase64;
  }

}
