package com.bizunited.platform.rbac.server.util;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import org.apache.commons.lang3.Validate;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import java.security.Principal;

/**
 * 本地权限认证的相关工具，目前提供当前登录人信息的获取
 * @Author: Paul Chan
 * @Date: 2020-04-17 14:13
 */
public final class SecurityUtils {

  /**
   * 不允许实例化工具类
   */
  private SecurityUtils() {
    throw new IllegalStateException("静态工具类不允许实例化");
  }
  
  /**
   * 获取当前系统登录人的代理对象
   * @return
   */
  public static Principal getPrincipal() {
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    Validate.notNull(authentication , "未获取到当前系统的登录人");
    return authentication;
  }

  /**
   * 获取当前系统的登录人账号，返回的是Principal的name属性
   * @return
   */
  public static String getUserAccount() {
    Principal principal = getPrincipal();
    String account = principal.getName();
    Validate.notBlank("未获取当前系统的登录账号");
    return account;
  }

  /**
   * 获取当前系统的登录人信息，同时会做当前登录信息的验证
   * @return
   */
  public static UserVo getCurrentUser() {
    String account = getUserAccount();
    UserService userService = ApplicationContextUtils.getApplicationContext().getBean(UserService.class);
    UserVo userVo = userService.findByAccount(account);
    Validate.notNull(userVo, "未获取到用户信息：%s", account);
    return userVo;
  }
}
