package com.bizunited.platform.rbac.server.service.init;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.service.init.InitProcessService;
import com.bizunited.platform.rbac.server.service.ButtonService;
import com.bizunited.platform.rbac.server.service.CompetenceService;
import com.bizunited.platform.rbac.server.vo.ButtonVo;
import com.bizunited.platform.rbac.server.vo.CompetenceVo;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import static org.springframework.core.io.support.ResourcePatternResolver.CLASSPATH_ALL_URL_PREFIX;


/**
 * 初始化菜单
 * 此初始化类会在系统启动时运行，此任务会扫描资源competences文件夹下面所有的.json文件，
 * 并将.json的内容作为初始化数据来初始化菜单、按钮、接口等信息
 * @Author: Paul Chan
 * @Date: 2019-10-15 16:00
 */
@Component("CompetenceInitProcess")
public class CompetenceInitProcess implements InitProcessService {

  private static final Logger LOGGER = LoggerFactory.getLogger(CompetenceInitProcess.class);

  @Autowired(required = false)
  private ButtonService buttonService;
  @Autowired(required = false)
  private CompetenceService competenceService;
  @Autowired
  private ApplicationContext applicationContext;

  @Override
  public int sort() {
    return 0;
  }

  @Override
  public boolean doProcess() {
    // 目前的逻辑是，只有数据库
    long count = competenceService.countByViewItem(1);
    return count == 0;
  }

  @Override
  public void init() {
    try {
      Resource[] resources = applicationContext.getResources(CLASSPATH_ALL_URL_PREFIX + "/competences/**.json");
      if(ArrayUtils.isEmpty(resources)) {
        return;
      }
      for (Resource resource : resources) {
        try (InputStream is = resource.getInputStream()) {
          this.initCompetences(is);
        }
      }
    } catch (IOException e) {
      LOGGER.warn("读取菜单文件错误，请检查菜单文件");
    }
  }

  /**
   * 初始化核心菜单,inputStream是一个菜单数组的文件输入流内容的格式见当前包下的nebula-competences.json资源文件
   * @param inputStream
   */
  private void initCompetences(InputStream inputStream) throws IOException {
    if(inputStream == null) {
      return;
    }
    if(competenceService == null) {
      return;
    }
    StringBuilder sb = new StringBuilder();
    try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))) {
      String line = bufferedReader.readLine();
      while (line != null){
        sb.append(line);
        line = bufferedReader.readLine();
      }
    }
    String json = sb.toString();
    if(StringUtils.isBlank(json)) {
      LOGGER.warn("初始化菜单json为空！！");
      return;
    }
    JSONArray competences = JSON.parseArray(json);
    this.initCompetences(null, competences);
  }

  /**
   * 初始化核心菜单,parent是菜单的上级，如果菜单没有上级，则parent为null。
   * competences是CompetenceVo对应的json数组，具体内容格式见nebula-competences.json资源文件，
   * 菜单地址根据菜单名称更新，若菜单不存在，则初始化菜单，且菜单默认启用。
   * 请保证json对象的字段跟nebula-competences.json资源文件一样
   * @param parent
   * @param competences
   */
  private void initCompetences(CompetenceVo parent, JSONArray competences) {
    if(CollectionUtils.isEmpty(competences)) {
      return;
    }
    Iterator<Object> iterator = competences.iterator();
    while (iterator.hasNext()) {
      Object object = iterator.next();
      JSONObject competenceJson = JSON.parseObject(object.toString());
      CompetenceVo competence = JSON.parseObject(competenceJson.toJSONString(), CompetenceVo.class);
      CompetenceVo oldCompetence;
      if(parent != null) {
        oldCompetence = competenceService.findByCommentAndViewItemAndParent(competence.getComment(), 1, parent.getId());
      } else {
        oldCompetence = competenceService.findByCommentAndViewItemAndNullParent(competence.getComment(), 1);
      }
      if(oldCompetence == null) {
        competence.setCreateDate(new Date());
        competence.setModifyDate(new Date());
        competence.setTstatus(1);
        competence.setParent(parent);
        oldCompetence = competenceService.create(competence);
      }
      this.initButtons(oldCompetence, competence.getButtons());
      if(competenceJson.containsKey("competences")) {
        this.initCompetences(oldCompetence, competenceJson.getJSONArray("competences"));
      }
    }
  }

  /**
   * 初始化菜单下的按钮
   * 该初始化只做按钮初始化新增的处理，不做更新处理
   * @param competence
   * @param buttons
   */
  private void initButtons(CompetenceVo competence, Set<ButtonVo> buttons) {
    if(buttonService == null || CollectionUtils.isEmpty(buttons)) {
      return;
    }
    for (ButtonVo button : buttons) {
      Set<CompetenceVo> competencesJson = button.getCompetences();
      ButtonVo oldButton = buttonService.findDetailsByCode(button.getCode());
      if(oldButton == null) {
        button.setCompetence(competence);
        button.setSystem(true);
        button.setCompetences(null);
        oldButton = buttonService.create(button);
      } else {
        Validate.isTrue(competence.getId().equals(oldButton.getCompetence().getId()), "重复的按钮比编码【%s】,请检查", button.getCode());
      }
      this.initButtonCompetences(oldButton, competencesJson);
    }
  }

  /**
   * 初始化按钮下的接口
   * @param button
   * @param competencesJson
   */
  private void initButtonCompetences(ButtonVo button, Set<CompetenceVo> competencesJson){
    if(CollectionUtils.isEmpty(competencesJson)) {
      return;
    }
    if(button == null){
      return;
    }
    Map<String, CompetenceVo> competencesMap = new HashMap<>();
    Set<CompetenceVo> buttonCompetences = competenceService.findByButtonId(button.getId());
    if(buttonCompetences == null){
      buttonCompetences = new HashSet<>();
      button.setCompetences(buttonCompetences);
    }else {
      for (CompetenceVo entity : buttonCompetences) {
        competencesMap.put(entity.getId(), entity);
      }
    }
    for(CompetenceVo competence : competencesJson){
      CompetenceVo oldCompetence = competenceService.findByResourceAndMethods(competence.getResource(), competence.getMethods());
      if(oldCompetence == null) {
        LOGGER.warn("未找到接口路径【{}】,请求方法【{}】，请检查!", competence.getResource(), competence.getMethods());
        continue;
      }
      if(competencesMap.containsKey(oldCompetence.getId())){
        continue;
      }
      buttonCompetences.add(oldCompetence);
      competencesMap.put(oldCompetence.getId(), oldCompetence);
    }
    buttonService.reBindCompetences(button, buttonCompetences);
  }
}
