package com.bizunited.platform.rbac.server.service.init;

import com.bizunited.platform.common.service.init.InitProcessService;
import com.bizunited.platform.rbac.server.service.CompetenceService;
import com.bizunited.platform.rbac.server.vo.CompetenceVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;
import springfox.documentation.spring.web.ControllerNamingUtils;

import javax.transaction.Transactional;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 该初始化过程完成Spring MVC中关于controller接口方法的扫描和持久化记录（每次启动都要刷新记录）
 * @author yinwenjie
 */
@Component("frameworkEndpointProcess")
public class FrameworkEndpointInitProcess implements InitProcessService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(FrameworkEndpointInitProcess.class);
  @Autowired
  private RequestMappingHandlerMapping frameworkEndpointHandler;
  @Autowired
  private CompetenceService competenceService;
  
  @Override
  public int sort() {
    // 紧接着权限初始化完成，就开始进行http接口初始化
    return -1;
  }

  @Override
  public boolean doProcess() {
    // 每次进程启动都要处理
    return true;
  }

  @Override
  @Transactional
  public void init() {
    this.initFrameworkEndpoint();
  }

  /**
   * 获取Api标注的标签
   * @param handlerMethod
   * @return
   */
  private List<String> getTags(HandlerMethod handlerMethod) {
    Class<?> clazz = handlerMethod.getBeanType();
    Api api = clazz.getAnnotation(Api.class);
    if(api == null) {
      return Lists.newArrayList();
    }
    String[] controllerTags = api.tags();
    return Arrays.stream(controllerTags).filter(tag -> StringUtils.isNotBlank(tag)).collect(Collectors.toList());
  }

  /**
   * 获取当前接口的tag分类，如果有多个分类，则取用第一个tag
   * @param handlerMethod
   * @return
   */
  private String getTag(HandlerMethod handlerMethod) {
    List<String> tags = this.getTags(handlerMethod);
    String tag;
    if(CollectionUtils.isEmpty(tags)) {
      tag = ControllerNamingUtils.controllerNameAsGroup(handlerMethod);
    } else {
      tag = tags.get(0);
    }
    return tag;
  }

  /**
   * 检测当前在spring mvc中完成注册的各个requestMapping是否已经被写入到表单引擎的功能表中<br>
   * 如果没有完成写入，则在这里进行判定后写入
   */
  private void initFrameworkEndpoint() {    
    /*
     * 操作步骤为
     * 1：遍历目前已经注册的requestMapping信息，并生成url$method这种形式的list
     * 2：取得当前competence数据表中所有的功能信息 并生成url$method这种形式的list
     * 3：比较两个list，并从第一个集合中取出那些不存在于数据库中的信息，最后进行添加操作
     * 4：根据第2步获取的dbCompetenes信息，结合当前新扫描得到的中文注释说明信息，进行更新操作
     * 
     * 注意：不进行删除操作，是因为之前那些不存在了的URL信息，可能因被设定到了诸如按钮绑定这样的功能里面
     * 这种情况开发人员自行进行手动修正才是最保险的。
     * */
    LOGGER.info("正在检测和调整requestMapping和数据库中Competence的对应关系======");
    Map<RequestMappingInfo , HandlerMethod> pathMapping = frameworkEndpointHandler.getHandlerMethods();
    // 1、=====
    Set<String> currentRequstUrls = Sets.newHashSet();
    Set<RequestMappingInfo> requestMappings = pathMapping.keySet();
    // urlCommentMapping 用于记录url和中文注释信息的关联关系
    Map<String, String> urlCommentMapping = Maps.newHashMap();
    // 用于存储url与标签的关联信息
    Map<String, String> urlTags = new HashMap<>(128);
    for (RequestMappingInfo requestMappingInfo : requestMappings) {
      // 一次方法的处理，就可能涉及到多个url和多个http method类型
      Set<String> perCurrentRequstUrls = Sets.newHashSet();
      HandlerMethod handlerMethod = pathMapping.get(requestMappingInfo);
      String tag = this.getTag(handlerMethod);
      Set<RequestMethod> requestMethods = requestMappingInfo.getMethodsCondition().getMethods();
      Set<String> patterns = requestMappingInfo.getPatternsCondition().getPatterns();
      // 注意，有可能开发人员在同一个requestMapping中定义了多个Mapping地址（但至少会有一个）
      String[] methodValues = requestMethods.stream().map(Object::toString).collect(Collectors.toList()).toArray(new String[]{});
      for (int index = 0 ; methodValues != null && index < methodValues.length ; index++) {
        final String methodValue = methodValues[index];
        patterns.forEach(item -> {
          perCurrentRequstUrls.add(item + "+" + methodValue);
          urlTags.put(item, tag);
        });
      }
      // 试图获取当前web接口所对应的java方法的中文文档信息——如果当前java使用了swagger注解
      if(handlerMethod == null) {
        continue;
      }
      ApiOperation apiOperation = handlerMethod.getMethodAnnotation(ApiOperation.class);
      if(apiOperation != null) {
         String comment = apiOperation.value();
         if(StringUtils.isNotBlank(comment)) {
           perCurrentRequstUrls.forEach(item -> urlCommentMapping.put(item, comment));
         }
      }
      currentRequstUrls.addAll(perCurrentRequstUrls);
    }
    
    // 2、====
    List<CompetenceVo> dbCompetenes = this.competenceService.findAll();
    Set<String> dbRequstUrls = null;
    if(dbCompetenes != null) {
      dbRequstUrls = dbCompetenes.stream().map(item -> item.getResource() + "+" + item.getMethods()).collect(Collectors.toSet());
    } else {
      dbRequstUrls = Sets.newHashSet();
    }
    SetView<String> difference = Sets.difference(currentRequstUrls, dbRequstUrls);
    
    // 3、====
    Pattern pattern = Pattern.compile("\\{[\\w]+\\}");
    if(difference != null) {
      List<CompetenceVo> newCompetences = difference.stream().map(item -> {
        String[] urlAndMethod = item.split("\\+");
        String url = urlAndMethod[0];
        String method = urlAndMethod[1];
        CompetenceVo currentCompetence = new CompetenceVo();
        String comment = urlCommentMapping.get(url);
        if(StringUtils.isNotBlank(comment)) {
          currentCompetence.setComment(comment);
        } else {
          currentCompetence.setComment(url);
        }
        currentCompetence.setMethods(method);
        currentCompetence.setResource(url);
        currentCompetence.setSortIndex(100);
        currentCompetence.setViewItem(0);
        currentCompetence.setTag(urlTags.get(url));
        // 是否是一个动态路径
        if(pattern.matcher(url).find()) {
          currentCompetence.setExtractUri(1);
        } else {
          currentCompetence.setExtractUri(0);
        }
        return currentCompetence;
      }).collect(Collectors.toList());
      // 批量添加
      if(newCompetences !=null && !newCompetences.isEmpty()) {
        this.competenceService.createAll(newCompetences);
      }
    }
    
 // 4、===
    if(dbCompetenes != null) {
      for (CompetenceVo dbCompetene : dbCompetenes) {
        //菜单不做更新
        if(dbCompetene.getViewItem()==1){
          continue;
        }
        String dbResource = dbCompetene.getResource();
        String dbMethod = dbCompetene.getMethods();
        if(StringUtils.isBlank(dbResource) || StringUtils.isBlank(dbMethod)) {
          continue;
        }
        String dbFullUrl = dbResource + "+" + dbMethod;
        String comment = urlCommentMapping.get(dbFullUrl);
        String dbcomment = dbCompetene.getComment();
        String tag = urlTags.get(dbResource);
        boolean update = false;
        if(StringUtils.isBlank(dbCompetene.getTag())) {
          dbCompetene.setTag(tag);
          update = true;
        }
        // 如果说明信息发生了变化，就进行更新
        if(StringUtils.isNotBlank(comment) && !StringUtils.equals(comment, dbcomment)
            && (StringUtils.equals(dbcomment, dbFullUrl) || StringUtils.equals(dbcomment, dbResource))) {
          dbCompetene.setComment(comment);
          update = true;
        }
        if(update) {
          this.competenceService.update(dbCompetene);
        }
      }
    }
  }
}