package com.bizunited.platform.rbac.server.util;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import java.security.Principal;

import static com.bizunited.platform.common.constant.Constants.ANONYMOUS_USER;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PASSWORD;

/**
 * 本地权限认证的相关工具，目前提供当前登录人信息的获取
 * @Author: Paul Chan
 * @Date: 2020-04-17 14:13
 */
public final class SecurityUtils {

  /**
   * 不允许实例化工具类
   */
  private SecurityUtils() {
    throw new IllegalStateException("静态工具类不允许实例化");
  }

  /**
   * 获取当前系统登录人的代理对象
   * @return
   */
  public static Principal getPrincipal() {
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    Validate.notNull(authentication , "当前系统未登录用户！");
    return authentication;
  }

  /**
   * 设置当前登录人，允许更换当前登录人的账号信息，相当于切换了登录账号
   * @param account
   */
  public static void setPrincipal(String account) {
    Validate.notBlank(account, "登录账号不能为空");
    UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(account, DEFAULT_PASSWORD, Lists.newArrayList());
    SecurityContextHolder.getContext().setAuthentication(authentication);
  }

  /**
   * 匿名用户设置, 当前系统未登录信息或是匿名用户登录时，允许自定义设置一个账号来替换匿名用户
   * @param account
   */
  public static void setAnonymousUser(String account) {
    Validate.notBlank(account, "登录账号不能为空");
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    if(authentication == null || ANONYMOUS_USER.equals(authentication.getName())) {
      setPrincipal(account);
    }
  }

  /**
   * 获取当前系统的登录人账号，返回的是Principal的name属性
   * @return
   */
  public static String getUserAccount() {
    Principal principal = getPrincipal();
    String account = principal.getName();
    Validate.notBlank("当前系统未登录用户！");
    Validate.isTrue(!ANONYMOUS_USER.equals(account), "当前系统未登录用户！");
    return account;
  }

  /**
   * 获取当前系统的登录人信息，同时会做当前登录信息的验证
   * @return
   */
  public static UserVo getCurrentUser() {
    String account = getUserAccount();
    UserService userService = ApplicationContextUtils.getApplicationContext().getBean(UserService.class);
    UserVo userVo = userService.findByAccount(account);
    Validate.notNull(userVo, "未获取到用户信息：%s", account);
    return userVo;
  }

}
