package com.bizunited.platform.rbac.server.service.vcode;

import java.util.Random;
import java.util.regex.Pattern;

import com.bizunited.platform.common.service.redis.RedisMutexService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import com.bizunited.platform.rbac.server.service.sms.SmsService;
import com.bizunited.platform.rbac.server.service.sms.SmsTypeEnums;

/**
 * 由技术中台底层提供的验证法生成服务实现</br>
 * @author yinwenjie
 */
public class ValidateCodeServiceDefaultImpl implements ValidateCodeService {
  /**
   * 10进制情况下8位数的最大值
   */
  private static final int MAX_DEC = 99999999;
  /**
   * 进行全部字母匹配的正则式
   */
  private static final String MATCH_LETTERS = "^[a-z|A-Z]+$";
  /**
   * 验证码防止高频重复生成的锁的key前缀
   */
  private static final String VCODE_LOCK_CYCLE_PREFIX = "VCODE_LOCK_CYCLE_";
  /**
   * 验证码防止高频发送的周期判定信息的key前缀
   */
  private static final String VCODE_SEND_CYCLE_PREFIX = "VCODE_SEND_CYCLE_";
  /**
   * 验证码防止前缀
   */
  private static final String VCODE_PREFIX = "_VCODE_";
  /**
   * .字母（纯大写）+数字性质的验证码的字符范围，这样定义主要是因为不能有混淆不清的字符出现，例如0和O
   */
  private static final char[] validate_chars = {'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',  
         'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W',  
         'X', 'Y', 'Z', '1', '2', '3', '4', '5', '6', '7', '8', '9'};
  
  /**
   * .字母（包括大小写）+数字性质的验证码的字符范围，这样定义主要是因为不能有混淆不清的字符出现，例如0和O，l和1
   */
  private static final char[] validate_ignore_chars = {'A','a', 'B','b', 'C','c', 'D','d', 'E','e', 'F','f', 'G','g', 'H','g', 'I','i', 'J','j',  
                                 'K','k', 'L', 'M','m', 'N','n', 'P','p', 'Q','q', 'R','r', 'S','s', 'T','t', 'U','u', 'V','v', 'W','w',  
                                 'X','x', 'Y','y', 'Z','z', '2', '3', '4', '5', '6', '7', '8', '9'};
  
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private SmsService smsService;

  @Override
  public String generate(String userid, String subSystem, int size, ValidateCodeType codeType, int timeout) {
    this.validate(userid, size, codeType, timeout);
    String currentsSubSystem = this.validateSubSystem(subSystem);
    /*
     * 1、首先根据指定的数制，生成符合长度要求的一个随机数
     * 2、将redis上当前用户、当前业务名下的验证码更新成新值，并返回。这里不存在线程安全性的问题
     * .如果有并发更新冲突，则以最后一次更新的数据为准。
     * */
    // 1、======
    Random currentRandom = new Random();
    Integer currentValue;
    String currentString;
    if(codeType == ValidateCodeType.NUMERICAL) {
      currentValue = currentRandom.nextInt(MAX_DEC);
      currentString = currentValue.toString();
      currentString = StringUtils.substring(currentString, 0, size);
    } else if(codeType == ValidateCodeType.CHAR) {
      currentString = this.generateChars(size);
    } else if(codeType == ValidateCodeType.IGNORE_CHAR) {
      currentString = this.generateIgnoreChars(size);
    } else {
      throw new NotSupportedCodeTypeException("目前暂不支持当前选择的验证码生成样式！！");
    }
    
    // 2、======
    // 验证码在redis保存时，必须有一个前缀。否则很容易和其它redis key重复
    String mapName = StringUtils.join(VCODE_PREFIX , userid);
    this.redisMutexService.setMCode(mapName, currentsSubSystem, currentString, timeout);
    return currentString;
  }
  
  /**
   * 该私有方法，基于字母（全大写） + 数字，生成指定长度的验证码
   * @param size
   * @return
   */
  private String generateChars(int size) {
    StringBuilder result = new StringBuilder();
    Random random = new Random();
    for (int i = 0; i < size; i++) {
      result.append(validate_chars[random.nextInt(validate_chars.length)]);
    }
    return result.toString();
  }
  
  /**
   *  该私有方法，基于字母（包括大小写） + 数字，生成指定长度的验证码
   * @param size
   * @return
   */
  private String generateIgnoreChars(int size) {
    StringBuilder result = new StringBuilder();
    Random random = new Random();
    for (int i = 0; i < size; i++) {
      result.append(validate_ignore_chars[random.nextInt(validate_ignore_chars.length)]);
    }
    return result.toString();
  }
  
  /**
   * 验证方法处理时的基本信息入参
   * @param userid 用户标识
   * @param size 验证码要求的长度
   * @param hexType 验证码进制
   * @param timeout 验证码有效期
   */
  private void validate(String userid , int size, ValidateCodeType codeType, int timeout) {
    Validate.notBlank(userid , "使用验证码生成服务时，用户标识必须传入！！");
    Validate.isTrue(size >= 4 && size <=8 , "验证码位数必须介于4位——8位之间！！");
    Validate.notNull(codeType , "使用验证码生成服务时，必须指定验证码样式效果（数字、字母+数字）！！");
    Validate.isTrue(timeout > 0 , "使用验证码生成服务时，必须为验证码指定正确的有效期（大于零的毫秒数）！！");
  }
  
  /**
   * 验证入参中的“子系统”信息，如果没有指定任何子系统名称，则默认使用root
   * @param subSystem 
   * @return
   */
  private String validateSubSystem(String subSystem) {
    String currentsSubSystem;
    if(StringUtils.isBlank(subSystem)) {
      currentsSubSystem = "root";
    } else {
      currentsSubSystem = subSystem;
    }
    Pattern pattern = Pattern.compile(MATCH_LETTERS);
    Validate.isTrue(pattern.matcher(currentsSubSystem).matches() , "业务模块名只能包含字母！！");
    return currentsSubSystem;
  }

  @Override
  public String generate(String userid, String subSystem, int size, int timeout) {
    return this.generate(userid, subSystem, size, ValidateCodeType.NUMERICAL, timeout);
  }

  @Override
  public String generate(String userid, String subSystem, int timeout) {
    return this.generate(userid, subSystem, 6, ValidateCodeType.NUMERICAL, timeout);
  }

  @Override
  public boolean match(String userid, String subSystem, String original) {
    if(StringUtils.isBlank(userid)) {
      return false;
    }
    String currentsSubSystem = this.validateSubSystem(subSystem);
    if(StringUtils.isBlank(original)) {
      return false;
    }
    
    // 完成入参边界校验后，从redis取得数据并进行判定
    String target = this.findByUserAndSubSystem(userid, currentsSubSystem);
    return StringUtils.equalsIgnoreCase(original, target);
  }
  
  @Override
  public String findByUserAndSubSystem(String userid, String subSystem) {
    Validate.notBlank(userid , "使用验证码生成服务时，用户标识必须传入！！");
    String currentsSubSystem = this.validateSubSystem(subSystem);
    
    // 查询redis
    String mapName = StringUtils.join(VCODE_PREFIX , userid);
    return this.redisMutexService.getMCode(mapName, currentsSubSystem);
  }

  @Override
  public String send(String phoneNumber , String userid , String subSystem , int size , ValidateCodeType codeType , int timeout) {
    // 电话号码格式的判定由实现者在SmsService中进行处理
    Validate.notBlank(phoneNumber , "电话号码必须填写！！");
    this.validate(userid, size, codeType , timeout);
    String currentsSubSystem = this.validateSubSystem(subSystem);
    int currentTimeout;
    if(timeout < 300000) {
      currentTimeout = 300000;
    } else {
      currentTimeout = timeout;
    }
    
    /*
     * 首先需要注意的是：
     * a、由于验证码的发送频率间隔固定为60秒，为了避免计时初期的多次提交问题，这里该方法需要获取基于redis的分布式锁
     * ,这个锁的key是vcode_send_cycle_用户名_currentsSubSystem
     * b、由于要记录指定用户标识下、指定业务名下，是否满足发送间隔（60秒），所以需要通过缓存记录上一个发送时间
     * ,那么这里使用的mapName是 "vcode_send_用户标识"，mapKey是“currentsSubSystem”，值是“用户名_currentsSubSystem”，设定的超时时间是60秒
     * 
     * .这样一来，整个处理的处理步骤描述如下：
     * 1、首先确定当前发送请求符合60秒的固定发送间隔频率中，也就是以上说明b所标识虚拟map已经过期。如果不满足发送频率则报错
     * 2、为了保证一瞬间多次请求下，短信也只发送一次，所以需要通过以上说明a的方式，进行分布式锁的获取，只有获取到了锁，才能进行下一步
     * 3、无论何种情况下，重新发送验证码并不意味着重新生成验证码，只要验证码没有超过其有效期（本方法中强制规定了至少5分钟），则重新发送一次之前的验证码
     * 4、从第2步开始的过程，无论是否操作成功，都需要在最后释放锁
     * */
    // 1、=========
    String cycleMapName = StringUtils.join(VCODE_SEND_CYCLE_PREFIX , userid);
    String cycleMapKey = currentsSubSystem;
    String cycleValue = StringUtils.join(userid , "_" , currentsSubSystem);
    String currentCycleValue = this.redisMutexService.getMCode(cycleMapName, cycleMapKey);
    Validate.isTrue(StringUtils.isBlank(currentCycleValue) , "距离上次发送不足60秒，不允许进行验证码信息重发，请稍后！！");
    
    // 2、========
    String lockKey = StringUtils.join(VCODE_LOCK_CYCLE_PREFIX , userid , "_" , currentsSubSystem);
    String currentVCode = null;
    try {
      this.redisMutexService.lock(lockKey);
      currentCycleValue = this.redisMutexService.getMCode(cycleMapName, cycleMapKey);
      Validate.isTrue(StringUtils.isBlank(currentCycleValue) , "距离上次发送不足60秒，不允许进行验证码信息重发，请稍后！！");
      
      // 3、=======
      currentVCode = this.findByUserAndSubSystem(userid, currentsSubSystem);
      // 如果条件成立，说明需要重新生成验证码
      if(StringUtils.isBlank(currentVCode)) {
        currentVCode = this.generate(userid, currentsSubSystem, size, codeType, currentTimeout);
      }
      Validate.notBlank(currentVCode , "验证码生成过程错误，请检查！！");
      // 重发，并重新记录发送周期间隔
      this.redisMutexService.setMCode(cycleMapName, cycleMapKey, cycleValue, 60000);
      this.smsService.sendSms(phoneNumber, currentVCode, SmsTypeEnums.INFO);
    } finally {
      // 4、=======
      this.redisMutexService.unlock(lockKey);
    }
    return currentVCode;
  }
}