package com.bizunited.platform.rbac.server.service.event;

import com.bizunited.platform.rbac.server.service.RolePositionMappingService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.position.PositionEventListener;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserRelationVo;
import com.google.common.collect.Sets;
import java.util.Arrays;
import java.util.Set;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.transaction.Transactional;
import java.util.List;
import org.springframework.util.CollectionUtils;

/**
 * 角色绑定职位监听处理
 * @Author: zengxingwang
 * @Date: 2020/9/24 11:23
 */
@Component("RolePositionEventListenerImpl")
public class RolePositionBindListener implements PositionEventListener {

  @Autowired
  private RoleService roleService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private RolePositionMappingService rolePositionMappingService;

  /**
   * 当本地系统被通知在本地人区创建了一个新的用户信息时，职位和角色关系发生变化，重新绑定职位和角色关系
   */
  @Override
  @Transactional
  public void onPositionCreate(UserRelationVo userRelation) {
    Validate.notNull(userRelation, "参数对象不能为空");
    PositionVo position = positionService.findByCode(userRelation.getPositionCode());
    Validate.notNull(position, "编码[%s]的职位未找到，请检查", userRelation.getPositionCode());
    List<String> roleIds = buildBindRoleIds(position, userRelation);
    //如果角色id列表为空，不做任何处理
    if(CollectionUtils.isEmpty(roleIds)){
      return;
    }
    //绑定职位-角色关系
    rolePositionMappingService.rebindPositionRoles(position.getCode(), roleIds.toArray(new String[roleIds.size()]));
  }

  @Override
  @Transactional
  public void onUnbindUser(UserRelationVo userRelation){
    Validate.notNull(userRelation, "参数对象不能为空");
    PositionVo position = positionService.findByCode(userRelation.getPositionCode());
    Validate.notNull(position, "编码[%s]的职位未找到，请检查", userRelation.getPositionCode());
    List<String> roleIds = this.buildBindRoleIds(position, userRelation);
    //如果角色id列表为空，不做任何处理
    if(CollectionUtils.isEmpty(roleIds)){
      return;
    }
    //解绑职位-角色关系
    rolePositionMappingService.unbindPositionRoles(position.getCode(), roleIds.toArray(new String[roleIds.size()]));
  }

  /**
   * 通过职位对象和用户关系对象中的角色编码数组，构建需要绑定到该职位的角色id列表
   * @param position
   * @param userRelation
   * @return
   */
  private List<String> buildBindRoleIds(PositionVo position, UserRelationVo userRelation) {
    Validate.notNull(position, "职位参数对象不能为空");
    Validate.notNull(userRelation, "职位-角色-用户关系对象不能为空");
    //如果角色编码为空，则不存在需要绑定的角色，所以返回空集合
    if(StringUtils.isBlank(userRelation.getAuthorityRoleCode())){
      return Lists.newArrayList();
    }
    List<String> roleIds = Lists.newArrayList();
    Set<String> roleCodes = Sets.newHashSet(Arrays.asList(userRelation.getAuthorityRoleCode().split(",")));
    Set<RoleVo> authorityRoles = roleService.findByRoleCodes(roleCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(authorityRoles) && authorityRoles.size() == roleCodes.size(),
            "本次绑定的这些角色【%s】中，有些角色不存在，请检查", userRelation.getAuthorityRoleCode());
    authorityRoles.forEach(role -> roleIds.add(role.getId()));
    return roleIds;
  }
}
