package com.bizunited.platform.rbac.cas.starter.configuration;

import java.util.Arrays;

import org.jasig.cas.client.session.SingleSignOutFilter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.ProviderManager;
import org.springframework.security.cas.ServiceProperties;
import org.springframework.security.cas.web.CasAuthenticationFilter;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.crypto.password.Pbkdf2PasswordEncoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.authentication.logout.LogoutFilter;

/**
 * 基于web的权限验证配置信息
 * @author yinwenjie
 */
@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true, jsr250Enabled = true)
public class WebSecurityConfig extends WebSecurityConfigurerAdapter {
  @Autowired
  private AuthenticationEntryPoint authenticationEntryPoint;
  @Autowired
  private AuthenticationProvider authenticationProvider;
  @Autowired
  private SingleSignOutFilter singleSignOutFilter;
  @Autowired
  private LogoutFilter logoutFilter;
  @Autowired
  private AuthenticationManager authenticationManager;
  
  /**
   * 忽略权限判断的url
   */
  @Value("${author.ignoreUrls}")
  private String[] ignoreUrls;
  @Autowired
  private CasProperties casProperties;
  @Value("${kuiper.loginUrl}")
  private String loginUrl;

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    
    http
    // 允许iframe嵌入
    .headers().frameOptions().disable().and()
    // 由于设置了验证filter访问为，/login/cas，所以必须通过验证，否则出现死循环
    .authorizeRequests().antMatchers(this.ignoreUrls).permitAll().and()
    .authorizeRequests().anyRequest().authenticated().and()
    .httpBasic().authenticationEntryPoint(authenticationEntryPoint).and()
    .logout().logoutSuccessUrl("/logout").and()
    .csrf().disable()
    .addFilterBefore(singleSignOutFilter, CasAuthenticationFilter.class)
    .addFilterBefore(logoutFilter, LogoutFilter.class);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
   * #configure(org.springframework.security.config.annotation.authentication.builders.
   * AuthenticationManagerBuilder)
   */
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    auth.authenticationProvider(authenticationProvider);
  }
  
  /*
   * (non-Javadoc)
   * 
   * @see
   * org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
   * #authenticationManager()
   */
  @Override
  protected AuthenticationManager authenticationManager() throws Exception {
    // 设置cas认证提供
    return new ProviderManager(Arrays.asList(authenticationProvider));
  }

  /* (non-Javadoc)
   * @see org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter#configure(org.springframework.security.config.annotation.web.builders.WebSecurity)
   */
  @Override
  public void configure(WebSecurity web) throws Exception {
    super.configure(web);
  }
  
  /**
  * CAS认证过滤器，主要实现票据认证和认证成功后的跳转
  * @param authenticationManager
  * @param serviceProperties
  * @return
  */
  @Bean
  public CasAuthenticationFilter casAuthenticationFilter(ServiceProperties sp) throws Exception {
    // cas认证过滤器，当触发本filter时，对ticket进行认证
    CasAuthenticationFilter filter = new CasAuthenticationFilter();
    filter.setServiceProperties(sp);
    filter.setAuthenticationManager(this.authenticationManager);
    filter.setFilterProcessesUrl(casProperties.getAppLoginUrl());
    filter.setContinueChainBeforeSuccessfulAuthentication(false);
    return filter;
  }
  
  /**
   * 用户密码的默认加密方式为PBKDF2加密
   * @return
   */
  @Bean
  public PasswordEncoder passwordEncoder() {
    PasswordEncoder passwordEncoder = new Pbkdf2PasswordEncoder();
    return passwordEncoder;
  }
}
