package com.bizunited.platform.rbac.cas.starter.controller;

import com.bizunited.platform.rbac.cas.starter.service.security.CasUserService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import java.io.IOException;
import java.security.Principal;
import java.util.Collection;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import io.swagger.annotations.ApiOperation;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.savedrequest.DefaultSavedRequest;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import com.bizunited.platform.rbac.cas.starter.configuration.CasProperties;
import com.bizunited.platform.rbac.server.service.UserService;

@Controller
@RequestMapping("/v1/rbac")
public class CasSecurityController {
  @Autowired
  private UserService userService;
  @Autowired
  private CasUserService casUserService;
  @Autowired
  private CasProperties casProperties;
  /**
   * 由spring boot 2.X支持的可能存在的context-path信息
   */
  @Value("${server.context-path:}")
  private String contextPath = ""; 
  /**
   * 登出成功后的重定向地址
   */
  @Value("${rabc.logoutUrl}")
  private String logoutUrl;
  
  private static final Logger LOGGER = LoggerFactory.getLogger(CasSecurityController.class);
  /**
   * 当登陆成功后，默认跳转到这个URL，并且返回登录成功后的用户基本信息
   * @return
   */
  @ApiOperation(value="当登陆成功后，默认跳转到这个URL，并且返回登录成功后的用户基本信息"
      + "一旦登录成功，服务端将会向客户端返回两个重要属性：<br>"
      + "1、SESSION属性：该属性可保证从最后一次服务器请求开始的30分钟内登录信息有效(前提是服务节点没有重启)<br>"
      + "2、persistence属性：该属性可保证从最后一次登录操作开始的100天内登录信息有效(前提是服务节点没有重启)<br>"
      + "客户端应该至少保证在进行HTTP请求时，向服务器传递persistence属性。但为了保证服务端业务处理性能，应该两个属性都进行传递。<br>"
      + "<b>请注意：正常情况下SESSION属性可能发生变化，一旦变化客户端应该向服务端传递最新的SESSION属性</b>"
      + "</p>"
      + "一旦登录成功或者通过persistence自动重登录成功后，服务端将通过以下两种方式向客户端返回新的SESSION属性和persistence属性：<br>"
      + "1、http response的header信息中，将携带。类似于：persistence =YWRtaW46MTUxNDUxNzA4MjYzNjplYzI0OTFlYWEyNDhkZmIyZWIyNjNjODc3YzM2M2Q0MA 和 SESSION =54fd02c7-4067-43c9-94f8-5f6e474cd858<br>"
      + "2、http response的cookies信息中，将携带。（此种方式是推荐使用的方式）<br>"
      + "<b>注意：以上描述只限于登录成功后的返回信息，并不是说每一次业务请求操作，服务端都会向客户端这样返回SESSION属性和persistence属性</b>"
      + "</p>"
      + "为了保证服务端能够正确识别到客户端已登录的用户权限信息，在正常的前端请求过程中，每一次客户端的请求都需要向服务端发送SESSION属性（非必要，但推荐）和persistence属性<br>"
      + "客户端可以使用以下方式，向服务端发送SESSION属性和persistence属性：<br>"
      + "1、直接使用http request的cookies发送。（此种方式是推荐使用的方式）<br>")
  @RequestMapping(value="/casSuccess" , method={RequestMethod.GET , RequestMethod.POST})
  public void loginSuccess(HttpServletRequest request , HttpServletResponse response) {
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    Validate.notNull(authentication , "为获取到当前系统的登录人");
    Object userPrincipalObject = authentication.getPrincipal();
    Validate.isTrue(userPrincipalObject instanceof Principal , "错误的权限组件!!");
    Principal logUser = (Principal)userPrincipalObject;
    
    // 由于前段模块的要求，在用户登录成功，或者通过persistence自动重登录成功后，
    // 后端服务都需要将cookies中重新设定的persistence和JSESSIONID以自定义属性的形式写入到head中。
    Collection<String> setCookies = response.getHeaders("Set-Cookie");
    String jsession = null;
    if(setCookies != null) {
      for (String setCookie : setCookies) {
        if(StringUtils.indexOf(setCookie, "persistence=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          response.setHeader("persistence", value);
        } else if(StringUtils.indexOf(setCookie, "JSESSIONID=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          jsession = value;
        }
      }
      response.setHeader("Access-Control-Expose-Headers", "JSESSIONID,persistence,Cookie");
    }

    //查询本地数据库是否有该用户，若没有，则保存至本地
    UserVo localUser = userService.findByAccount(logUser.getName());
    if(null == localUser) {
      UserVo userVo = casUserService.findByThird(logUser.getName());
      if(userVo!=null){
        userService.create(userVo);
      }
    }

    // 确认前端最终跳转页面(优先从session中确认)
    String defaultRedirectUrl = casProperties.getAppWelcomeUrl();
    HttpSession session = request.getSession();
    DefaultSavedRequest defaultSavedRequest = (DefaultSavedRequest)session.getAttribute("SPRING_SECURITY_SAVED_REQUEST");
    String currentRedirectUrl = "";
    if(defaultSavedRequest == null) {
      currentRedirectUrl = defaultRedirectUrl;
    } else {
      String query = defaultSavedRequest.getQueryString();
      String requestURI = defaultSavedRequest.getRequestURI();
      if(StringUtils.isBlank(query)) {
        currentRedirectUrl = requestURI;
      } else {
        currentRedirectUrl = StringUtils.join(requestURI , "?" , query);
      }
    }
    
    Cookie redirectCookie = new Cookie("redirectUrl", currentRedirectUrl);
    redirectCookie.setPath("/");
    response.addCookie(redirectCookie);
    // 确认当前用户账户名
    Cookie accountCookie = new Cookie("account", logUser.getName());
    accountCookie.setPath("/");
    response.addCookie(accountCookie);
    
    // 从spring session取出相关信息——如果Security没有取到
    if(StringUtils.isBlank(jsession)) {
      jsession = session.getId();
      Cookie cookie = new Cookie("JSESSIONID", jsession);
      cookie.setPath("/");
      response.addCookie(cookie);
      cookie = new Cookie("persistence", jsession);
      cookie.setPath("/");
      response.addCookie(cookie);
    }
    response.setHeader("JSESSIONID", jsession);
    
    // 去取可能的Spring boot 1.x支持的context-path位置，以便在存在nginx代理的情况下，前端页面能够正确的请求。
    Cookie contextPathCookie = new Cookie("context-path", contextPath);
    contextPathCookie.setPath("/");
    response.addCookie(contextPathCookie);
    
    // 直接构造跳转页面内容
    try {
      response.setHeader("content-type", "text/html;charset=UTF-8");
      response.getWriter().write(writeCasPage().toString());
    } catch (IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new RuntimeException(e);
    }
  }
  
  /**
   * 此接口用于临时解决cas已登陆用户访问本系统的鉴权问题
   * 构建一个cookie，内容包含persistence、JSESSIONID、projectPath(项目路径)
   * 然后携带一个成功与否的状态值，通过页面的形式返回给前端(不得已为之，除了页面形式，其他都会报错)，再由前端去判断跳转到具体哪个页面
   * @param request
   * @param response
   */
  @RequestMapping(value="/casToken" , method={RequestMethod.GET , RequestMethod.POST})
  public void invokeTransition(HttpServletRequest request , HttpServletResponse response) {
    //根据前端的要求，返回携带信息成功与否
    boolean flag = false;
    //前端只需要persistence，JSESSIONID和cooikes
    Collection<String> setCookies = response.getHeaders("Set-Cookie");
    String jsession = null;
    //第一次可能没有值，一旦成功访问过了我们的页面，都会又cookies
    if(setCookies != null) {
      for (String setCookie : setCookies) {
        if(StringUtils.indexOf(setCookie, "persistence=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          response.setHeader("persistence", value);
        } else if(StringUtils.indexOf(setCookie, "JSESSIONID=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          jsession = value;
        }
      }
      response.setHeader("Access-Control-Expose-Headers", "JSESSIONID,persistence,Cookie");
    }
    
    
    //第一次访问肯定没有值，这时会从request session中获取
    if(StringUtils.isBlank(jsession)) {
      HttpSession session = request.getSession();
      jsession = session.getId();
      Cookie cookie = new Cookie("JSESSIONID", jsession);
      cookie.setPath("/");
      response.addCookie(cookie);
      cookie = new Cookie("persistence", jsession);
      cookie.setPath("/");
      response.addCookie(cookie);
    }
    response.setHeader("JSESSIONID", jsession);
    
    
    //逻辑进行到这里时，判断是否成功的拿取或产生了sessionId
    //如果jsession有值，那么我们就认定cas的其他用户请求时成功的，合法的
    if(jsession != null ) {
      flag = true;
    }
    //应前端要求增加一个项目地址的cooike信息
    Cookie projectPathCookie = new Cookie("projectPath", request.getContextPath());
    response.addCookie(projectPathCookie);
    
    //由于洋河环境问题，只能返回页面信息才不会报错，所以这里返回的时一个携带了请求最终结果(true或者false)的页面
    try {
      response.setHeader("content-type", "text/html;charset=UTF-8");
      response.getWriter().write("<!DOCTYPE html><html><body>" + flag + "</body></html>");
    } catch (IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new RuntimeException(e);
    }
  }
  
  private StringBuffer writeCasPage() {
    StringBuffer pageContext = new StringBuffer();
    
    pageContext.append("<!DOCTYPE html>").append(System.lineSeparator());
    pageContext.append("<html>").append(System.lineSeparator());
    pageContext.append("<head>").append(System.lineSeparator());
    pageContext.append("<title>鉴权跳转</title>").append(System.lineSeparator());
    pageContext.append("</head>").append(System.lineSeparator());
    pageContext.append("<body>").append(System.lineSeparator());
    pageContext.append("<div id='authMsg'>正在鉴权,请等待...</div>").append(System.lineSeparator());
    pageContext.append("<script>").append(System.lineSeparator());
    pageContext.append("//设置消息").append(System.lineSeparator());
    pageContext.append("function setAuthMsg(msg){").append(System.lineSeparator());
    pageContext.append("document.getElementById(\"authMsg\").innerHTML=msg;").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("//设置storage").append(System.lineSeparator());
    pageContext.append("function setStorage(session,persistence,account){").append(System.lineSeparator());
    pageContext.append("var curStorage = window.localStorage;").append(System.lineSeparator());
    pageContext.append("var formEngineData = {").append(System.lineSeparator());
    pageContext.append("session:session,").append(System.lineSeparator());
    pageContext.append("persistence:persistence").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("curStorage.setItem(\"formEngineData\", JSON.stringify(formEngineData));").append(System.lineSeparator());
    pageContext.append("var loginInfo = {").append(System.lineSeparator());
    pageContext.append("id:account,").append(System.lineSeparator());
    pageContext.append("account:account").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("curStorage.setItem(\"formEngineLoginInfo\",JSON.stringify(loginInfo));").append(System.lineSeparator());
    pageContext.append("//写入主机头到localStorage 供API调用").append(System.lineSeparator());
    pageContext.append("let domain = getCookie(\"context-path\");").append(System.lineSeparator());
    pageContext.append("if(domain){").append(System.lineSeparator());
    pageContext.append("curStorage.setItem(\"formEngineDomain\",JSON.stringify({Domain:domain}));").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("//localstorage注入表单引擎数据").append(System.lineSeparator());
    pageContext.append("//读取当前cookie信息").append(System.lineSeparator());
    pageContext.append("function getCookie(name){").append(System.lineSeparator());
    pageContext.append("var arr,reg=new RegExp(\"(^| )\"+name+\"=([^;]*)(;|$)\"); //正则匹配").append(System.lineSeparator()); 
    pageContext.append("if(arr=document.cookie.match(reg)){").append(System.lineSeparator());
    pageContext.append("return unescape(arr[2]);").append(System.lineSeparator());
    pageContext.append("}else{").append(System.lineSeparator());
    pageContext.append("return \"\";").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("} ").append(System.lineSeparator());
    pageContext.append("(function(){").append(System.lineSeparator());
    pageContext.append("//读取 sessionid").append(System.lineSeparator());
    pageContext.append("var sessionId = getCookie(\"JSESSIONID\")").append(System.lineSeparator());
    pageContext.append("//读取persistence ").append(System.lineSeparator());
    pageContext.append("var persistence = getCookie(\"persistence\");").append(System.lineSeparator());
    pageContext.append("//读取 登录账户信息").append(System.lineSeparator());
    pageContext.append("var account = getCookie(\"account\");").append(System.lineSeparator());
    pageContext.append("if(!sessionId || !persistence ||!account){").append(System.lineSeparator());
    pageContext.append("setAuthMsg('鉴权失败! 请<a href=\"javascript:back();\">返回重新登录</a>');").append(System.lineSeparator());
    pageContext.append("return;").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("//写入当前数据到localStorage").append(System.lineSeparator());
    pageContext.append("setStorage(sessionId,persistence,account);").append(System.lineSeparator());
    pageContext.append("//获得跳转回的路径").append(System.lineSeparator());
    pageContext.append("var redirectUrl = getCookie(\"redirectUrl\");").append(System.lineSeparator());
    pageContext.append("if(!redirectUrl){").append(System.lineSeparator());
    pageContext.append("setAuthMsg(\"鉴权成功! 但未指定需要跳转的页面。\");").append(System.lineSeparator());
    pageContext.append("return;").append(System.lineSeparator());
    pageContext.append("}").append(System.lineSeparator());
    pageContext.append("//跳转回表单引擎或者指定页面").append(System.lineSeparator());
    pageContext.append("window.location.href=redirectUrl;").append(System.lineSeparator());
    pageContext.append("})();").append(System.lineSeparator());
    pageContext.append("</script>").append(System.lineSeparator());
    pageContext.append("</body>").append(System.lineSeparator());
    pageContext.append("</html>").append(System.lineSeparator());
    
    return pageContext;
  }
  
  /**
   * 该http方法用于登出系统，并按照设定信息，跳转到指定的登陆后页面。所谓登录实际上就是清楚当前系统的cookies信息
   */
  @ApiOperation(value="该http方法用于登出系统，并按照设定信息，跳转到指定的登陆后页面。所谓登录实际上就是清楚当前系统的cookies信息")
  @RequestMapping(value="/logoutSuccess" , method={RequestMethod.GET , RequestMethod.POST})
  public void logoutSuccess(HttpServletRequest request , HttpServletResponse response) {
    Cookie[] allCookies = request.getCookies();
    if(allCookies != null) {
      for(int index = 0 ; index < allCookies.length ; index++){
        Cookie currentCookie = allCookies[index];
        currentCookie.setMaxAge(0);
        response.addCookie(currentCookie);
      }
    }
    
    // 跳转到指定的页面
    try {
      response.sendRedirect(this.logoutUrl);
    } catch (IOException e) {
      LOGGER.error(e.getMessage() , e);
    }
  }
}
