package com.bizunited.platform.rbac.cas.starter.configuration;

import java.util.Arrays;

import org.jasig.cas.client.session.SingleSignOutFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.ProviderManager;
import org.springframework.security.cas.ServiceProperties;
import org.springframework.security.cas.web.CasAuthenticationFilter;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.crypto.password.Pbkdf2PasswordEncoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.CorsConfigurationSource;
import org.springframework.web.cors.UrlBasedCorsConfigurationSource;

/**
 * 基于web的权限验证配置信息
 * @author yinwenjie
 */
@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true, jsr250Enabled = true)
public class WebSecurityConfig extends WebSecurityConfigurerAdapter {
  @Autowired
  private AuthenticationEntryPoint authenticationEntryPoint;
  @Autowired
  private AuthenticationProvider authenticationProvider;
  @Autowired
  private SingleSignOutFilter singleSignOutFilter;
//  @Autowired
//  private AuthenticationManagerBuilder auth;
  private static final Logger LOGGER = LoggerFactory.getLogger(WebSecurityConfig.class);

  /**
   * 忽略权限判断的url
   */
  @Value("${rbac.ignoreUrls}")
  private String[] ignoreUrls;
  @Autowired
  private CasProperties casProperties;
  @Value("${rbac.loginUrl}")
  private String loginUrl;
  /**
   * 登出请求地址
   */
  @Value("${rbac.logoutUrl:/v1/rbac/logout}")
  private String logoutUrl;

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    CorsConfigurationSource configurationSource = this.corsConfigurationSource();
    http
    // 显示允许登录操作时跨域
    .cors().configurationSource(configurationSource).and()
    // 允许iframe嵌入
    .headers().frameOptions().disable().and()
    // 由于设置了验证filter访问为，/login/cas，所以必须通过验证，否则出现死循环
    .authorizeRequests().antMatchers(this.ignoreUrls).permitAll().and()
    .authorizeRequests().anyRequest().authenticated().and()
    .httpBasic().authenticationEntryPoint(authenticationEntryPoint).and()
    .logout()
    // 登出页面
    .logoutUrl(logoutUrl)
    // 登出成功后
    .logoutSuccessUrl("/v1/rbac/logoutSuccess").permitAll().and()
    .csrf().disable()
    .addFilterBefore(singleSignOutFilter, CasAuthenticationFilter.class);
  }
  
  /**
   * 配置权限的跨域操作
   * @return
   */
  private CorsConfigurationSource corsConfigurationSource() {
    CorsConfiguration configuration = new CorsConfiguration();
    configuration.setAllowedOrigins(Arrays.asList("*"));
    configuration.setAllowedMethods(Arrays.asList("*"));
    configuration.setAllowedHeaders(Arrays.asList("*"));
    configuration.setAllowCredentials(true);
    UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
    source.registerCorsConfiguration("/**", configuration);
    return source;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
   * #configure(org.springframework.security.config.annotation.authentication.builders.
   * AuthenticationManagerBuilder)
   */
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    super.configure(auth);
    auth.authenticationProvider(authenticationProvider);
  }
  
  /*
   * (non-Javadoc)
   * 
   * @see
   * org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
   * #authenticationManager()
   */
  @Override
  protected AuthenticationManager authenticationManager() throws Exception {
    // 设置cas认证提供
    return new ProviderManager(Arrays.asList(authenticationProvider));
  }

  /* (non-Javadoc)
   * @see org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter#configure(org.springframework.security.config.annotation.web.builders.WebSecurity)
   */
  @Override
  public void configure(WebSecurity web) throws Exception {
    super.configure(web);
  }
  
  /**
  * CAS认证过滤器，主要实现票据认证和认证成功后的跳转
  * @param authenticationManager
  * @param serviceProperties
  * @return
  */
  @Bean
  public CasAuthenticationFilter casAuthenticationFilter(ServiceProperties sp){
    // cas认证过滤器，当触发本filter时，对ticket进行认证
    CasAuthenticationFilter filter = new CasAuthenticationFilter();
    AuthenticationManager authenticationManager = null;
    try {
      authenticationManager = this.authenticationManager();
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
    }
    
    filter.setServiceProperties(sp);
    filter.setAuthenticationManager(authenticationManager);
    filter.setFilterProcessesUrl(casProperties.getAppLoginUrl());
    filter.setContinueChainBeforeSuccessfulAuthentication(false);
    return filter;
  }
  
  /**
   * 用户密码的默认加密方式为PBKDF2加密
   * @return
   */
  @Bean(name="passwordEncoder")
  public PasswordEncoder passwordEncoder() {
    return new Pbkdf2PasswordEncoder();
  }
}
