package com.bizunited.platform.rbac.cas.starter.configuration;

import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.ProviderManager;
import org.springframework.security.cas.ServiceProperties;
import org.springframework.security.cas.web.CasAuthenticationFilter;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.web.cors.CorsConfigurationSource;

import com.bizunited.platform.rbac.cas.starter.handle.SimpleAccessDeniedHandler;

/**
 * 基于web的权限验证配置信息
 * @author yinwenjie
 */
@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true, jsr250Enabled = true)
public class WebSecurityConfig extends WebSecurityConfigurerAdapter {
  @Autowired
  @Qualifier("AuthenticationEntryPoint")
  private AuthenticationEntryPoint authenticationEntryPoint;
  @Autowired
  @Qualifier("AuthenticationProvider")
  private AuthenticationProvider authenticationProvider;
  @Autowired
  private UserDetailsService userDetailsService; 
  /**
   * 登出成功后的处理器
   */
  @Autowired
  private LogoutSuccessHandler logoutSuccessHandler;
  /**
   * 鉴权失败或者不具备权限时的处理器
   */
  @Autowired
  @Qualifier("AccessDeniedHandler")
  private SimpleAccessDeniedHandler accessDeniedHandler;
  @Autowired
  private PasswordEncoder passwordEncoder;
  @Autowired
  private CorsConfigurationSource corsConfigurationSource;

  private static final Logger LOGGER = LoggerFactory.getLogger(WebSecurityConfig.class);

  /**
   * 忽略权限判断的url
   */
  @Value("${rbac.ignoreUrls}")
  private String[] ignoreUrls;
  @Autowired
  private CasProperties casProperties;
  /**
   * 当本服务的登录信息失败后，只需要返回601或者602编码给请求端，无需跳转到cas-server登录页面
   */
  @Value("${rbac.cas.showHttpCode}")
  private boolean showHttpCode;
  @Autowired
  private ServiceProperties serviceProperties;

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    http
    // 允许跨域操作
    .cors().configurationSource(this.corsConfigurationSource).and()
    // 允许iframe嵌入
    .headers().frameOptions().disable().and()
    // 由于设置了验证filter访问为，/login/cas，所以必须通过验证，否则出现死循环
    .authorizeRequests().antMatchers(this.ignoreUrls).permitAll().and()
    .authorizeRequests().anyRequest().authenticated().and()
    .httpBasic().authenticationEntryPoint(authenticationEntryPoint)
    .and()
    // 设定cas协议登出成功后的处理器
    .logout()
      .logoutUrl(casProperties.getAppLogoutUrl())
      .logoutSuccessHandler(logoutSuccessHandler).permitAll()
    .and()
      // 关闭csrf
      .csrf().disable()
    // 持久化登录信息，登录时间为100天 
    .rememberMe()
      .tokenValiditySeconds(100 * 24 * 60 * 60)
      .rememberMeCookieName("persistence")
      .alwaysRemember(true)
    .and()
      .addFilterBefore(casAuthenticationFilter(this.serviceProperties) , CasAuthenticationFilter.class);

    // 如果条件成立，说明集成当前当前cas的服务是一个后端微服务
    if(showHttpCode) {
      http
      // 设定单点登录异常的处理器
      .exceptionHandling()
      .authenticationEntryPoint(WebSecurityConfig.this.accessDeniedHandler)
      .accessDeniedHandler(WebSecurityConfig.this.accessDeniedHandler);
    }
  }
  
  @Autowired
  public void configureGlobal(AuthenticationManagerBuilder auth) throws Exception {
    // 设置密码加密模式
    auth.userDetailsService(userDetailsService).passwordEncoder(this.passwordEncoder);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
   * #configure(org.springframework.security.config.annotation.authentication.builders.
   * AuthenticationManagerBuilder)
   */
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    super.configure(auth);
    auth.authenticationProvider(authenticationProvider);
  }
  
  /*
   * (non-Javadoc)
   * 
   * @see
   * org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
   * #authenticationManager()
   */
  @Override
  protected AuthenticationManager authenticationManager() throws Exception {
    // 设置cas认证提供
    return new ProviderManager(Arrays.asList(authenticationProvider));
  }

  /* (non-Javadoc)
   * @see org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter#configure(org.springframework.security.config.annotation.web.builders.WebSecurity)
   */
  @Override
  public void configure(WebSecurity web) throws Exception {
    super.configure(web);
  }
  
  /**
  * CAS认证过滤器，主要实现票据认证和认证成功后的跳转
  * @param authenticationManager
  * @param serviceProperties
  * @return
  */
  private CasAuthenticationFilter casAuthenticationFilter(ServiceProperties sp){
    // cas认证过滤器，当触发本filter时，对ticket进行认证
    CasAuthenticationFilter filter = new CasAuthenticationFilter();
    AuthenticationManager authenticationManager = null;
    try {
      authenticationManager = this.authenticationManager();
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
    }

    filter.setServiceProperties(sp);
    filter.setAuthenticationManager(authenticationManager);
    filter.setFilterProcessesUrl(casProperties.getAppLoginUrl());
    filter.setContinueChainBeforeSuccessfulAuthentication(false);
    return filter;
  }
}
