package com.bizunited.platform.rbac.okta.starter.configuration;

import com.bizunited.platform.rbac.okta.starter.handle.SimpleAccessDeniedHandler;
import com.bizunited.platform.rbac.security.starter.service.security.CustomFilterSecurityInterceptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.AccessDecisionManager;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.crypto.password.Pbkdf2PasswordEncoder;
import org.springframework.security.oauth2.client.oidc.userinfo.OidcUserRequest;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.client.userinfo.OAuth2UserService;
import org.springframework.security.oauth2.core.oidc.user.OidcUser;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.access.intercept.FilterSecurityInterceptor;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.CorsConfigurationSource;
import org.springframework.web.cors.UrlBasedCorsConfigurationSource;

import java.util.Arrays;

/**
 * 针对okta单点登录的web权限配置
 */
@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true, jsr250Enabled = true)
public class OktaWebSecurityConfigurerAdapter extends WebSecurityConfigurerAdapter {

  /**
   * 忽略权限判断的url
   */
  @Value("${rbac.ignoreUrls}")
  private String[] ignoreUrls;
  @Value("${rbac.loginUrl}")
  private String loginUrl;
  /**
   * 登出操作地址
   */
  @Value("${rbac.logoutUrl}")
  private String logoutUrl;

  @Autowired
  private OAuth2UserService<OidcUserRequest, OidcUser> oidcUserService;
  @Autowired
  private UserDetailsService userDetailsService;
  @Autowired
  private AccessDecisionManager accessDecisionManager;
  @Lazy
  @Autowired
  private FilterInvocationSecurityMetadataSource securityMetadataSource;
  @Autowired(required = false)
  private ClientRegistrationRepository clientRegistrationRepository;


  @Override
  protected void configure(HttpSecurity http) throws Exception {
    CustomFilterSecurityInterceptor filterSecurityInterceptor = new CustomFilterSecurityInterceptor(securityMetadataSource, accessDecisionManager, super.authenticationManager());
    CorsConfigurationSource configurationSource = this.corsConfigurationSource();
    SimpleAccessDeniedHandler simpleAccessDeniedHandler = new SimpleAccessDeniedHandler(clientRegistrationRepository);
    http
        .addFilterAt(filterSecurityInterceptor, FilterSecurityInterceptor.class)
        .cors().configurationSource(configurationSource).and()
        .headers().frameOptions().disable()
        .and().authorizeRequests().antMatchers(this.ignoreUrls).permitAll()
        .and().authorizeRequests().anyRequest().authenticated()
        .and().exceptionHandling()
        .authenticationEntryPoint(simpleAccessDeniedHandler)
        .accessDeniedHandler(simpleAccessDeniedHandler)
        // enable OAuth2/OIDC
        .and().oauth2Client()
        .and().oauth2Login()
        .userInfoEndpoint().oidcUserService(oidcUserService);
    http.csrf().disable();
  }


  @Override
  public void configure(WebSecurity web) throws Exception {
    super.configure(web);
  }

  @Autowired
  public void globalUserDetails(AuthenticationManagerBuilder auth) throws Exception {
    auth.userDetailsService(userDetailsService)
        .passwordEncoder(passwordEncoder());
    auth.parentAuthenticationManager(authenticationManagerBean());
  }

  /**
   * 配置权限的跨域操作
   * @return
   */
  private CorsConfigurationSource corsConfigurationSource() {
    CorsConfiguration configuration = new CorsConfiguration();
    configuration.setAllowedOrigins(Arrays.asList("*"));
    configuration.setAllowedMethods(Arrays.asList("*"));
    configuration.setAllowedHeaders(Arrays.asList("*"));
    configuration.setAllowCredentials(true);
    UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
    source.registerCorsConfiguration("/**", configuration);
    return source;
  }

  /**
   * 用户密码的默认加密方式为PBKDF2加密
   * @return
   */
  @Bean(name="passwordEncoder")
  @ConditionalOnMissingBean
  public PasswordEncoder passwordEncoder() {
    return new Pbkdf2PasswordEncoder();
  }

}
