package com.bizunited.platform.rbac.security.starter.configuration;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.annotation.web.configurers.ExpressionUrlAuthorizationConfigurer;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.crypto.password.Pbkdf2PasswordEncoder;
import org.springframework.web.cors.CorsUtils;

import com.bizunited.platform.rbac.security.starter.service.security.CustomAccessDecisionManager;
import com.bizunited.platform.rbac.security.starter.service.security.CustomFilterInvocationSecurityMetadataSource;
import com.bizunited.platform.rbac.security.starter.service.security.CustomFilterSecurityInterceptor;
import com.bizunited.platform.rbac.security.starter.service.security.CustomUserSecurityDetailsService;

/**
 * 和访问权限有关的配置信息在这里
 * @author yinwenjie
 */
@Configuration
@EnableAutoConfiguration
@EnableWebSecurity
@ComponentScan(basePackages= {"com.bizunited.platform.rbac.security.starter"})
public class SecurityConfig extends WebSecurityConfigurerAdapter {
  /**
   * 忽略权限判断的url
   */
  @Value("${author.ignoreUrls}")
  private String[] ignoreUrls;
  
  @Value("${kuiper.loginUrl}")
  private String loginUrl;
  
  @Override
  protected void configure(HttpSecurity http) throws Exception {
    http
      // 允许iframe嵌入
      .headers().frameOptions().disable().and()
      // 设定显示jsession设定信息
      .sessionManagement()
        .enableSessionUrlRewriting(true).and()
      .authorizeRequests()
        //对preflight放行
        .requestMatchers(CorsUtils::isPreFlightRequest).permitAll()
        // 系统中的“登录页面”在被访问时，不进行权限控制
        .antMatchers(ignoreUrls).permitAll()
        // 其它访问都要验证权限
        .anyRequest().authenticated().and()
        // ==================== 设定登录页的url地址，它不进行权限控制
      .formLogin()
        // 由于后端提供的都是restful接口，并没有直接跳转的页面
        // 所以只要访问的url没有通过权限认证，就跳到这个请求上，并直接排除权限异常
        .loginPage("/v1/formengine/security/loginFail")
        // 登录请求点
        .loginProcessingUrl(loginUrl)
        // 登录失败，返回到这里
        .failureForwardUrl("/v1/formengine/security/loginFail")
        // 登录成功后，默认到这个URL，返回登录成功后的信息
        .successForwardUrl("/v1/formengine/security/loginSuccess").permitAll().and()
        // ===================== 设定登出后的url地址
      .logout()
        // 登出页面
        .logoutUrl("/v1/formengine/security/logout")
        // 登录成功后
        .logoutSuccessUrl("/v1/formengine/security/logoutSuccess").permitAll().and()
        // ===================== 关闭csrf
      .csrf()
        .disable()
      .rememberMe()
        // 持久化登录信息，登录时间为100天 
        .tokenValiditySeconds(100*24*60*60)
        .rememberMeCookieName("persistence")
        .alwaysRemember(true);
    
    // 允许preflight request
    ExpressionUrlAuthorizationConfigurer<HttpSecurity>.ExpressionInterceptUrlRegistry registry = http.authorizeRequests();
    // 让Spring security放行所有preflight request
    registry.requestMatchers(CorsUtils::isPreFlightRequest).permitAll();
  }
  
  @Autowired
  public void configureGlobal(AuthenticationManagerBuilder auth) throws Exception {
    // 设置密码加密模式
    auth.userDetailsService(customUserDetailsService()).passwordEncoder(passwordEncoder());
  }
  
  /**
   * 用户密码的默认加密方式为PBKDF2加密
   * @return
   */
  @Bean
  @ConditionalOnMissingBean(PasswordEncoder.class)
  public PasswordEncoder passwordEncoder() {
    PasswordEncoder passwordEncoder = new Pbkdf2PasswordEncoder();
    return passwordEncoder;
  }

  /**
   * 自定义UserDetailsService，从数据库中读取用户信息
   * @return
   */
  @Bean(name="CustomUserSecurityDetailsService")
  @ConditionalOnMissingBean
  public CustomUserSecurityDetailsService customUserDetailsService() {
    return new CustomUserSecurityDetailsService();
  }
  @Bean(name="CustomAccessDecisionManager")
  @ConditionalOnMissingBean
  public CustomAccessDecisionManager getCustomAccessDecisionManager() {
    return new CustomAccessDecisionManager();
  }
  @Bean(name="CustomFilterInvocationSecurityMetadataSource")
  @ConditionalOnMissingBean
  public CustomFilterInvocationSecurityMetadataSource getCustomFilterInvocationSecurityMetadataSource() {
    return new CustomFilterInvocationSecurityMetadataSource();
  }
  @Bean(name="CustomFilterSecurityInterceptor")
  @ConditionalOnMissingBean
  public CustomFilterSecurityInterceptor getCustomFilterSecurityInterceptor() {
    return new CustomFilterSecurityInterceptor();
  }
}