package com.bizunited.platform.rbac.security.starter.controller;

import io.swagger.annotations.ApiOperation;

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import com.bizunited.platform.rbac.security.starter.controller.model.ResponseCode;
import com.bizunited.platform.rbac.security.starter.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserVo;

/**
 * 和权限安全相关的接口，在这里暴露<br>
 * @author yinwenjie 
 */
@RestController
@RequestMapping("/v1/rbac/security")
public class SecurityController {
  
  @Autowired
  private UserService userService;
  @Autowired
  private RoleService roleService;
  
  /**
   * 当登陆成功后，默认跳转到这个URL，并且返回登录成功后的用户基本信息
   * @return
   */
  @ApiOperation(value="当登陆成功后，默认跳转到这个URL，并且返回登录成功后的用户基本信息"
      + "一旦登录成功，服务端将会向客户端返回两个重要属性：<br>"
      + "1、SESSION属性：该属性可保证从最后一次服务器请求开始的30分钟内登录信息有效(前提是服务节点没有重启)<br>"
      + "2、persistence属性：该属性可保证从最后一次登录操作开始的100天内登录信息有效(前提是服务节点没有重启)<br>"
      + "客户端应该至少保证在进行HTTP请求时，向服务器传递persistence属性。但为了保证服务端业务处理性能，应该两个属性都进行传递。<br>"
      + "<b>请注意：正常情况下SESSION属性可能发生变化，一旦变化客户端应该向服务端传递最新的SESSION属性</b>"
      + "</p>"
      + "一旦登录成功或者通过persistence自动重登录成功后，服务端将通过以下两种方式向客户端返回新的SESSION属性和persistence属性：<br>"
      + "1、http response的header信息中，将携带。类似于：persistence =YWRtaW46MTUxNDUxNzA4MjYzNjplYzI0OTFlYWEyNDhkZmIyZWIyNjNjODc3YzM2M2Q0MA 和 SESSION =54fd02c7-4067-43c9-94f8-5f6e474cd858<br>"
      + "2、http response的cookies信息中，将携带。（此种方式是推荐使用的方式）<br>"
      + "<b>注意：以上描述只限于登录成功后的返回信息，并不是说每一次业务请求操作，服务端都会向客户端这样返回SESSION属性和persistence属性</b>"
      + "</p>"
      + "为了保证服务端能够正确识别到客户端已登录的用户权限信息，在正常的前端请求过程中，每一次客户端的请求都需要向服务端发送SESSION属性（非必要，但推荐）和persistence属性<br>"
      + "客户端可以使用以下方式，向服务端发送SESSION属性和persistence属性：<br>"
      + "1、直接使用http request的cookies发送。（此种方式是推荐使用的方式）<br>")
  @RequestMapping(value="/loginSuccess" , method=RequestMethod.POST)
  public ResponseModel loginSuccess(HttpServletRequest request , HttpServletResponse response) {
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    String account = authentication.getName();
    if(StringUtils.isEmpty(account)) {
      ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode._501, new IllegalAccessException("not found op user!"));
      return result;
    }
    
    // 查询用户基本信息
    UserVo user = this.userService.findByAccount(account);
    // 如果条件成立，说明这个用户存在数据问题。抛出异常
    if(user == null) {
      ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode._501, new IllegalAccessException("not found op user!"));
      return result;
    }
    // 查询这用用户目前拥有的角色，包括自身绑定的、用户组绑定的、组织机构绑定的等等
    List<RoleVo> roles = this.roleService.findAllByUserId(user.getId());
    user.setRoles(new HashSet<>(roles));
    
    // 由于前段模块的要求，在用户登录成功，或者通过persistence自动重登录成功后，
    // 后端服务都需要将cookies中重新设定的persistence和JSESSIONID以自定义属性的形式写入到head中。
    Collection<String> setCookies = response.getHeaders("Set-Cookie");
    String jsession = null;
    if(setCookies != null) {
      for (String setCookie : setCookies) {
        if(StringUtils.indexOf(setCookie, "persistence=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          response.setHeader("persistence", value);
        } else if(StringUtils.indexOf(setCookie, "JSESSIONID=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          jsession = value;
        }
      }
      response.setHeader("Access-Control-Expose-Headers", "JSESSIONID,persistence,Cookie");
    }
    
    // 从spring session取出相关信息——如果Security没有取到
    if(StringUtils.isBlank(jsession)) {
      HttpSession session = request.getSession();
      jsession = session.getId();
      Cookie cookie = new Cookie("JSESSIONID", jsession);
      response.addCookie(cookie);
    }
    response.setHeader("JSESSIONID", jsession);
    
    // 隔断用户关联信息后返回
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode._0, null);
    result.setData(user);
    return result;
  }
  
  /**
   * 由于后端提供的都是restful接口，并没有直接跳转的页面<br>
   * 所以只要访问的url没有通过权限认证，就跳到这个请求上，并直接排除权限异常
   */
  @ApiOperation(value="由于后端提供的都是restful接口，并没有直接跳转的页面<br>"
      + "所以只要访问的url没有通过权限认证，就跳到这个请求上，并直接排除权限异常")
  @RequestMapping(value="/loginFail" , method={RequestMethod.GET , RequestMethod.POST})
  public ResponseModel loginFail() throws IllegalAccessException {
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode._501,new IllegalAccessException("用户已失效或用户名/密码错误，请检查!"));
    return result;
  }
  
  /**
   * 成功登出
   */
  @ApiOperation(value="成功登出")
  @RequestMapping(value="/logoutSuccess" , method=RequestMethod.GET)
  public ResponseModel logoutSuccess() {
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode._0, null);
    return result;
  }
}