package com.bizunited.platform.rbac.security.starter.handle;

import java.io.IOException;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;
import org.springframework.stereotype.Component;

import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserVo;

/**
 * 鉴权成功（无论哪种鉴权模式），将执行这个处理器：</p>
 * 当登陆成功后，默认跳转到这个URL，并且返回登录成功后的用户基本信息一旦登录成功，服务端将会向客户端返回两个重要属性：</br>
 * 1、SESSION属性：该属性可保证从最后一次服务器请求开始的30分钟内登录信息有效(前提是服务节点没有重启)</br>
 * 2、persistence属性：该属性可保证从最后一次登录操作开始的100天内登录信息有效(前提是服务节点没有重启)</br>
 * 客户端应该至少保证在进行HTTP请求时，向服务器传递persistence属性。但为了保证服务端业务处理性能，应该两个属性都进行传递。</br></br>
 * 请注意：正常情况下SESSION属性可能发生变化，一旦变化客户端应该向服务端传递最新的SESSION属性</br>
 * 一旦登录成功或者通过persistence自动重登录成功后，服务端将通过以下两种方式向客户端返回新的SESSION属性和persistence属性：</br>
 * 1、http response的header信息中，将携带。类似于：persistence =YWRtaW46MTUxNDUxNzA4MjYzNjplYzI0OTFlYWEyNDhkZmIyZWIyNjNjODc3YzM2M2Q0MA 和 SESSION =54fd02c7-4067-43c9-94f8-5f6e474cd858</br>
 * 2、http response的cookies信息中，将携带。（此种方式是推荐使用的方式）</br></br>
 * 注意：以上描述只限于登录成功后的返回信息，并不是说每一次业务请求操作，服务端都会向客户端这样返回SESSION属性和persistence属性</br>
 * 为了保证服务端能够正确识别到客户端已登录的用户权限信息，在正常的前端请求过程中，每一次客户端的请求都需要向服务端发送SESSION属性（非必要，但推荐）和persistence属性</br>
 * 客户端可以使用以下方式，向服务端发送SESSION属性和persistence属性：</br>
 * 1、直接使用http request的cookies发送。（此种方式是推荐使用的方式）</br></br>
 * TODO 没有必要再去查数据库了，用户基本信息，具有的角色有已经有了
 * @author yinwenjie
 */
@Component("simpleAuthenticationSuccessHandler")
public class SimpleAuthenticationSuccessHandler extends SavedRequestAwareAuthenticationSuccessHandler implements HandleOutPut {
  @Autowired
  private UserService userService;
  @Autowired
  private RoleService roleService;
  
  @Override
  @Transactional
  public void onAuthenticationSuccess(HttpServletRequest request, HttpServletResponse response, Authentication authentication) throws IOException, ServletException {
    //什么都不做的话，那就直接调用父类的方法
    //super.onAuthenticationSuccess(request, response, authentication);
    //如果是要跳转到某个页面的，比如我们的那个whoim的则
    //new DefaultRedirectStrategy().sendRedirect(request, response, "/whoim");
    String account = authentication.getName();
    if(StringUtils.isBlank(account)) {
      ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E501, new IllegalAccessException("not found op user!"));
      this.writeResponse(response, result);
      return;
    }
    // 最后更新当前用户的最后更新时间
    this.userService.updateLastloginTime(account);
    
    // 查询用户基本信息
    UserVo user = this.userService.findByAccount(account);
    // 如果条件成立，说明这个用户存在数据问题。抛出异常
    if(user == null) {
      ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E501, new IllegalAccessException("not found op user!"));
      this.writeResponse(response, result);
      return;
    }
    // 查询这用用户目前拥有的角色，包括自身绑定的、用户组绑定的、组织机构绑定的等等
    List<RoleVo> roles = this.roleService.findAllByUserId(user.getId());
    user.setRoles(new HashSet<>(roles));
    
    // 由于前端模块的要求，在用户登录成功，或者通过persistence自动重登录成功后，
    // 后端服务都需要将cookies中重新设定的persistence和JSESSIONID以自定义属性的形式写入到head中。
    Collection<String> setCookies = response.getHeaders("Set-Cookie");
    String jsession = null;
    if(setCookies != null) {
      for (String setCookie : setCookies) {
        if(StringUtils.indexOf(setCookie, "persistence=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          response.setHeader("persistence", value);
        } else if(StringUtils.indexOf(setCookie, "JSESSIONID=") != -1) {
          int indexd = setCookie.indexOf("=");
          String value = setCookie.substring(indexd+1);
          jsession = value;
        }
      }
      response.setHeader("Access-Control-Expose-Headers", "JSESSIONID,persistence,Cookie");
    }
    // 从spring session取出相关信息——如果Security没有取到
    if(StringUtils.isBlank(jsession)) {
      HttpSession session = request.getSession();
      jsession = session.getId();
      Cookie cookie = new Cookie("JSESSIONID", jsession);
      cookie.setSecure(true);
      response.addCookie(cookie);
    }
    response.setHeader("JSESSIONID", jsession);
    
    // 隔断用户关联信息后返回
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E0, null);
    result.setData(user);
    this.writeResponse(response, result);
  }
}
