package com.bizunited.platform.rbac.security.starter.service.security.authenticate;

import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.crypto.password.PasswordEncoder;

import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.service.ccode.CheckCodeService;
import com.bizunited.platform.rbac.server.service.vcode.ValidateCodeService;
import com.bizunited.platform.rbac.server.vo.UserVo;

/**
 * 鉴权决策器的抽象父类，涵盖一些在鉴权决策过程中，多个具体决策过程一定会使用到的公共方法
 * @author yinwenjie
 */
public abstract class AbstractAuthenticationDecision implements AuthenticationDecision {
  /**
   * 表达登录操作在验证码功能中设定的子系统名
   */
  protected static final String LOG_SUBSYSTEM = "login";
  @Autowired
  @Qualifier("passwordEncoder")
  @Lazy
  private PasswordEncoder passwordEncoder;
  @Autowired
  private UserService userService;
  @Autowired
  private ValidateCodeService validateCodeService;
  @Autowired
  private CheckCodeService checkCodeService;
  
  /**
   * .该上层方法用于验证传入的用户登录账号是否能够在数据层找到，并检查密码是否满足要求
   * @param userName 用户登录时输入的用户名
   * @param inputPassword 用户登录时输出的密码
   * @return
   */
  protected UserVo validateUserNameAndPassword(String userName , String inputPassword) {
    Validate.notBlank(inputPassword , "密码信息必须填写！！");
    UserVo currentUser = this.validateUserName(userName);
    String currentPassword = currentUser.getPassword();
    Validate.isTrue(passwordEncoder.matches(inputPassword, currentPassword) , ERROR_USER);
    return currentUser;
  }
  
  /**
   * .该上层方法用于验证传入的用户登录手机号是否能够在数据层找到，并检查密码是否满足要求
   * @param userName 用户登录时输入的用户名
   * @param inputPassword 用户登录时输出的密码
   * @return
   */
  protected UserVo validatePhoneAndPassword(String phone , String inputPassword) {
    Validate.notBlank(inputPassword , "密码信息必须填写！！");
    /*
     * .操作过程主要为：
     * 1、查询这个用户是否存在
     * 2、查询其密码是否匹配
     * */
    UserVo currentUser = this.validatePhone(phone);
    String currentPassword = currentUser.getPassword();
    Validate.isTrue(passwordEncoder.matches(inputPassword, currentPassword) , ERROR_USER);
    return currentUser;
  }
  
  /**
   * .该上层方法用于验证传入的用户登录账号是否能够在数据层找到，
   * @param userName 用户登录时输入的用户名
   * @return
   */
  protected UserVo validateUserName(String userName) {
    Validate.notBlank(userName , "用户名信息必须填写！！");
    UserVo currentUser = this.userService.findByAccount(userName);
    Validate.notNull(currentUser , ERROR_USER);
    Validate.isTrue(currentUser.getUseStatus() == 1 , ERROR_USER);
    return currentUser;
  }
  
  /**
   * 该上层方法用于验证传入的用户手机号是否能够在数据层找到。
   * @param phone
   * @return
   */
  protected UserVo validatePhone(String phone) {
    Validate.notBlank(phone , ERROR_USER);
    UserVo currentUser = this.userService.findByPhone(phone);
    Validate.notNull(currentUser , ERROR_USER);
    Validate.isTrue(currentUser.getUseStatus() == 1 , ERROR_USER);
    return currentUser;
  }
  
  /**
   * 验证指定的用户标识下，登录业务的验证码是否能够匹配，如果不能匹配将直接抛出异常
   * @param sessionId 指定的用户标识
   * @param smsVerificationCode 用户输入的验证码
   */
  protected void validateSmsVerificationCode(String sessionId , String smsVerificationCode) {
    Validate.notBlank(sessionId , ERROR_USER);
    Validate.notBlank(smsVerificationCode , "用户必须输入收到的短信验证码信息！！");
    boolean isMatch = this.validateCodeService.match(sessionId, LOG_SUBSYSTEM, smsVerificationCode);
    Validate.isTrue(isMatch , ERROR_SMS_CODE);
  }
  
  /**
   * 验证指定的用户标识下，登录业务的校验码是否能够匹配，如果不能匹配将直接抛出异常。</br>
   * 一定要注意，校验码一般只用在登录业务中，一般就是用来防止机器人操作和用户重复提交使用的。页面上的介质一般是图片
   * @param sessionId 指定的用户标识
   * @param checkCode 用户输入的校验码
   */
  protected void validateCheckCode(String sessionId , String checkCode) {
    Validate.notBlank(sessionId , ERROR_USER);
    Validate.notBlank(checkCode , "用户必须输入校验码，一般它以图片的方式呈现给用户！！");
    boolean isMatch = this.checkCodeService.match(sessionId, checkCode);
    Validate.isTrue(isMatch , ERROR_CHECK_CODE);
  }
}
