package com.bizunited.platform.rbac.security.starter.service.security;

import java.util.LinkedList;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserVo;

/**
 * 这个自定义的用户信息查询服务（权限模块使用），用来查询指定的用户详情（包括角色信息）
 * @author yinwenjie
 */
public class CustomUserSecurityDetailsService implements UserDetailsService {
  
  @Autowired
  private UserService userService;
  
  @Autowired
  private RoleService roleService;
  
  /* (non-Javadoc)
   * @see org.springframework.security.core.userdetails.UserDetailsService#loadUserByUsername(java.lang.String)
   */
  @Override
  public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
    /*
     * 查找用户的处理过程描述如下：
     * 1、首先查找用户信息，没有找到，就抛出异常
     * 2、查找这个账号对应的角色信息（当然是目前还可以用的角色信息），角色的来源有几部分
     *   a、当前用户直接绑定的角色
     *   b、当前用户所属的一个或者多个用户组绑定的角色
     *   c、当前用户所属组织机构、组织机构的各父级组织结构所绑定的角色
     * 注意：以上步骤中，如果找到了ADMIN这样存在于ignoreMethodCheckRoles配置中的角色时，就不需要再继续找了。
     * 因为一旦有了这个角色，就不再受角色-功能的绑定限制了，任何功能都可以访问
     * 3、构造UserDetails对象，并返回
     * */
    // 查询用户基本信息
    UserVo currentUser = this.userService.findByAccount(username);
    if(currentUser == null || currentUser.getUseStatus() != 1) {
      throw new UsernameNotFoundException("没有发现指定的账号，或者账号状态不正确！");
    }
    
    // 查询用户角色信息
    List<RoleVo> roles = null;
    roles = this.roleService.findAllByUserId(currentUser.getId());
    if(roles == null || roles.isEmpty()) {
      throw new UsernameNotFoundException("用户权限状态错误，请联系客服人员！");
    }
    List<SimpleGrantedAuthority> authorities = new LinkedList<>();
    for (RoleVo role : roles) {
      SimpleGrantedAuthority authoritie = new SimpleGrantedAuthority(role.getRoleName());
      authorities.add(authoritie);
    }
    
    // 角色信息形成authorities集合对象
    return new User(username, currentUser.getPassword(), authorities);
  }
}
