package com.bizunited.platform.rbac.security.starter.service.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.access.SecurityConfig;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import com.bizunited.platform.rbac.server.service.CompetenceService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.CompetenceVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.google.common.collect.Lists;

/**
 * @author yinwenjie
 */
public class CustomFilterInvocationSecurityMetadataSource implements FilterInvocationSecurityMetadataSource {
  /**
   * 匿名用户鉴权体系
   */
  private static final SecurityConfig ANONYMOUS_CONFIG = new SecurityConfig("ANONYMOUS");
  @Autowired
  private CompetenceService competenceService;
  @Autowired
  private RoleService roleService;
  /**
   * 忽略权限判断的url
   */
  @Value("${rbac.ignoreUrls}")
  private String[] ignoreUrls;
  /**
   * 那些不需要进行资源鉴权的具有超级管理员特性的角色
   */
  @Value("${rbac.ignoreMethodCheckRoles}")
  private String[] ignoreMethodCheckRoles;
  @Autowired
  private RequestMappingHandlerMapping frameworkEndpointHandler;
  
  /* (non-Javadoc)
   * @see org.springframework.security.access.SecurityMetadataSource#getAttributes(java.lang.Object)
   */
  @Override
  public Collection<ConfigAttribute> getAttributes(Object object) throws IllegalArgumentException {
    /*
     * 这个方法的核心含义是，一个指定的URL地址，需要什么角色（可以是多个）才能访问。
     * 如果没有指定，说明“任何角色”都不能访问。确定功能路径绑定的角色过程如下：
     * 1、结合frameworkEndpointHandler工具后，首先从工具中找出当前请求在MVC中对应的处理器
     * (注意，如果当前url处于当前忽略配置中，则不需要进行后续处理)
     * 2、如果存在至少一个匹配路径，那么从数据库中找出这些对应的资源信息（可能不止一个）
     * 3、这里进行资源访问角色的过滤，将能够访问这些资源的角色写入结果集合
     * */
    // 1、====================
    List<ConfigAttribute> configs = new ArrayList<>();
    FilterInvocation filterInvocation = (FilterInvocation) object;
    HttpServletRequest httpRequest = filterInvocation.getHttpRequest();
    // 在忽略权限判断的设置中，增加几个系统默认的路径
    ArrayList<String> currentIgnoreUrls = new ArrayList<>();
    if(ignoreUrls != null && ignoreUrls.length > 0) {
      currentIgnoreUrls.addAll(Lists.newArrayList(ignoreUrls));
    }
    currentIgnoreUrls.addAll(Lists.newArrayList(com.bizunited.platform.rbac.security.starter.configuration.SecurityConfig.DEFAULT_IGNOREURLS));
    
    for (String ignoreUrl : currentIgnoreUrls) {
      AntPathRequestMatcher requestMatcher = new AntPathRequestMatcher(ignoreUrl);
      if(requestMatcher.matches(httpRequest)) {
        configs.add(ANONYMOUS_CONFIG);
        return configs;
      }
    }
    // 另外，frameworkEndpointHandler中还有urlPathHelper、pathMatcher可以使用
    Map<RequestMappingInfo , HandlerMethod> pathMapping = frameworkEndpointHandler.getHandlerMethods();
    Set<RequestMappingInfo> requestMappings = pathMapping.keySet();
    List<RequestMappingInfo> matchingResults = Lists.newArrayList();
    for (RequestMappingInfo requestMappingInfo : requestMappings) {
      RequestMappingInfo current = requestMappingInfo.getMatchingCondition(httpRequest);
      if(current != null) {
        matchingResults.add(current);
      }
    }
    // 如果没有找到任何处理器，说明这个URL地址并没有加入到权限控制体系，则不做权限过滤
    // 原则就是“未明确拒绝则是允许”
    if(matchingResults.isEmpty()) {
      configs.add(ANONYMOUS_CONFIG);
      return configs;
    }
    
    // 4、========如果当前登录者具有超级管理员性质的设定，那么也为当前访问的资源写入ANONYMOUS_CONFIG标识
    // TODO 注释要重写
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    if(authentication == null) {
      throw new AccessDeniedException("not author!");
    }
    Collection<? extends GrantedAuthority> currentRoles = authentication.getAuthorities();
    if(ignoreMethodCheckRoles != null && currentRoles != null && !currentRoles.isEmpty()) {
      Set<String> currentRoleNames = currentRoles.stream().map(GrantedAuthority::toString).collect(Collectors.toSet());
      for (String currentRoleName : currentRoleNames) {
        // 如果条件成立，就说明这个用户携带了不需要进行资源鉴权的超级管理员性质的角色
        if(StringUtils.equalsAnyIgnoreCase(currentRoleName, ignoreMethodCheckRoles)) {
          configs.add(ANONYMOUS_CONFIG);
          return configs;
        }
      }
    }
    
    // 2、=======
    List<CompetenceVo> currentCompetences = Lists.newArrayList();
    for (RequestMappingInfo requestMappingInfo : matchingResults) {
      Set<String> resources = requestMappingInfo.getPatternsCondition().getPatterns();
      if(resources == null || resources.isEmpty()) {
        continue;
      }
      for (String resource : resources) {
        List<CompetenceVo> dbCompetences = this.competenceService.findByResource(resource , 1);
        if(dbCompetences != null && !dbCompetences.isEmpty()) {
          currentCompetences.addAll(dbCompetences);
        }
      }
    }
    // 如果没有任何数据库中存储功能信息，则不再做权限过滤（但正常情况下，不会出现这样的问题）
    if(currentCompetences.isEmpty()) {
      configs.add(ANONYMOUS_CONFIG);
      return configs;
    }
    
    // 3、===================
    String currentMethod = httpRequest.getMethod();
    for (CompetenceVo competence : currentCompetences) {
      String methods = competence.getMethods();
      // 注意，方法类型也必须满足
      if(methods.indexOf(currentMethod) != -1) {
        // 取得当前功能和角色的绑定关系
        List<RoleVo> roles = roleService.findByCompetenceId(competence.getId());
        if(roles == null || roles.isEmpty()) {
          continue;
        }
        // 将角色添加到返回值中，证明当前系统中已设定当前角色可以访该功能
        for (RoleVo role : roles) {
          SecurityConfig securityConfig = new SecurityConfig(role.getRoleName());
          configs.add(securityConfig);
        }
      }
    }
    
    // 执行到来这里，如果还没有确定当前资源能够被访问的角色定义，说明当前资源不能被访问
    if(configs.isEmpty()) {
      throw new AccessDeniedException("not author!");
    }
    return configs;
  }

  /* (non-Javadoc)
   * @see org.springframework.security.access.SecurityMetadataSource#getAllConfigAttributes()
   */
  @Override
  public Collection<ConfigAttribute> getAllConfigAttributes() {
    return Collections.emptyList();
  }

  /* (non-Javadoc)
   * @see org.springframework.security.access.SecurityMetadataSource#supports(java.lang.Class)
   */
  @Override
  public boolean supports(Class<?> clazz) {
    return true;
  }
}