package com.bizunited.platform.titan.starter.configuration.flowable;

import com.bizunited.platform.titan.starter.configuration.flowable.process.FlowableProcessProperties;
import com.bizunited.platform.titan.starter.configuration.flowable.process.Process;
import com.bizunited.platform.titan.starter.configuration.flowable.process.ProcessAsync;
import com.bizunited.platform.titan.starter.configuration.flowable.process.ProcessAsyncHistory;
import org.flowable.common.engine.impl.cfg.IdGenerator;
import org.flowable.common.engine.impl.persistence.StrongUuidGenerator;
import org.flowable.engine.*;
import org.flowable.job.service.impl.asyncexecutor.AsyncExecutor;
import org.flowable.spring.ProcessEngineFactoryBean;
import org.flowable.spring.SpringProcessEngineConfiguration;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.io.Resource;
import org.springframework.transaction.PlatformTransactionManager;

import javax.sql.DataSource;
import java.io.IOException;
import java.util.List;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-22 10:13
 * @Description: Flowable流程引擎自动配置
 */
@Configuration
@EnableConfigurationProperties({
    FlowableProperties.class,
    FlowableMailProperties.class,
    FlowableProcessProperties.class
})
@Import({
    FlowableJobConfiguration.class
})
public class ProcessEngineAutoConfiguration extends AbstractSpringEngineAutoConfiguration {

  /**
   * 流程的配置信息
   */
  protected final FlowableProcessProperties processProperties;
  /**
   * 邮件的配置信息
   */
  protected final FlowableMailProperties mailProperties;

  public ProcessEngineAutoConfiguration(FlowableProperties flowableProperties, FlowableProcessProperties processProperties,
                                        FlowableMailProperties mailProperties) {
    super(flowableProperties);
    this.processProperties = processProperties;
    this.mailProperties = mailProperties;

  }

  /**
   * If there is no app engine configuration, then trigger a creation of the process engine.
   */
  @Configuration
  static class StandaloneEngineConfiguration extends BaseEngineConfigurationWithConfigurers<SpringProcessEngineConfiguration> {

    @Bean
    public ProcessEngineFactoryBean processEngine(SpringProcessEngineConfiguration configuration) throws Exception {
      ProcessEngineFactoryBean processEngineFactoryBean = new ProcessEngineFactoryBean();
      processEngineFactoryBean.setProcessEngineConfiguration(configuration);

      invokeConfigurers(configuration);

      return processEngineFactoryBean;
    }
  }

  /**
   * RuntimeService服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public RuntimeService runtimeServiceBean(ProcessEngine processEngine) {
    return processEngine.getRuntimeService();
  }

  /**
   * 流程资源服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public RepositoryService repositoryServiceBean(ProcessEngine processEngine) {
    return processEngine.getRepositoryService();
  }

  /**
   * 任务服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public TaskService taskServiceBean(ProcessEngine processEngine) {
    return processEngine.getTaskService();
  }

  /**
   * 历史数据服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public HistoryService historyServiceBean(ProcessEngine processEngine) {
    return processEngine.getHistoryService();
  }

  /**
   * 管理服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public ManagementService managementServiceBean(ProcessEngine processEngine) {
    return processEngine.getManagementService();
  }

  /**
   * 动态流程文件服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public DynamicBpmnService dynamicBpmnServiceBean(ProcessEngine processEngine) {
    return processEngine.getDynamicBpmnService();
  }

  /**
   * 表单服务的初始化，这个暂时用不着
   * @param processEngine
   * @return
   */
  @Bean
  public FormService formServiceBean(ProcessEngine processEngine) {
    return processEngine.getFormService();
  }

  /**
   * 用户认证服务的初始化
   * @param processEngine
   * @return
   */
  @Bean
  public IdentityService identityServiceBean(ProcessEngine processEngine) {
    return processEngine.getIdentityService();
  }

  /**
   * The Async Executor must not be shared between the engines.
   * Therefore a dedicated one is always created.
   */
  //暂时有问题，先注释掉
  /*@Bean
  @ProcessAsync
  @ConfigurationProperties(prefix = "flowable.process.async.executor")
  @ConditionalOnMissingBean(name = "processAsyncExecutor")
  public SpringAsyncExecutor processAsyncExecutor(
      ObjectProvider<TaskExecutor> taskExecutor,
      @Process ObjectProvider<TaskExecutor> processTaskExecutor,
      ObjectProvider<SpringRejectedJobsHandler> rejectedJobsHandler,
      @Process ObjectProvider<SpringRejectedJobsHandler> processRejectedJobsHandler
  ) {
    return new SpringAsyncExecutor(
        getIfAvailable(processTaskExecutor, taskExecutor),
        getIfAvailable(processRejectedJobsHandler, rejectedJobsHandler)
    );
  }*/

  // 暂时有问题，先注释掉
  /*@Bean
  @ProcessAsyncHistory
  @ConfigurationProperties(prefix = "flowable.process.async-history.executor")
  @ConditionalOnMissingBean(name = "asyncHistoryExecutor")
  @ConditionalOnProperty(prefix = "flowable.process", name = "async-history.enable")
  public SpringAsyncHistoryExecutor asyncHistoryExecutor(
      ObjectProvider<TaskExecutor> taskExecutor,
      @Process ObjectProvider<TaskExecutor> processTaskExecutor,
      ObjectProvider<SpringRejectedJobsHandler> rejectedJobsHandler,
      @Process ObjectProvider<SpringRejectedJobsHandler> processRejectedJobsHandler
  ) {
    return new SpringAsyncHistoryExecutor(
        getIfAvailable(processTaskExecutor, taskExecutor),
        getIfAvailable(processRejectedJobsHandler, rejectedJobsHandler)
    );
  }*/

  /**
   * 初始化流程引擎的配置信息
   * @param dataSource 数据源
   * @param platformTransactionManager  事务管理器
   * @param processIdGenerator 流程ID生成器
   * @param globalIdGenerator 远程ID生成器
   * @param asyncExecutorProvider 异步执行任务
   * @param asyncHistoryExecutorProvider 异步历史任务执行器
   * @return
   * @throws IOException
   */
  @Bean
  public SpringProcessEngineConfiguration springProcessEngineConfiguration(DataSource dataSource, PlatformTransactionManager platformTransactionManager,
                                                                           @Process ObjectProvider<IdGenerator> processIdGenerator,
                                                                           ObjectProvider<IdGenerator> globalIdGenerator,
                                                                           @ProcessAsync ObjectProvider<AsyncExecutor> asyncExecutorProvider,
                                                                           @ProcessAsyncHistory ObjectProvider<AsyncExecutor> asyncHistoryExecutorProvider) throws IOException {

    SpringProcessEngineConfiguration conf = new SpringProcessEngineConfiguration();

    // 获取本地流程定义资源
    List<Resource> resources = this.discoverDeploymentResources(
        flowableProperties.getProcessDefinitionLocationPrefix(),
        flowableProperties.getProcessDefinitionLocationSuffixes(),
        flowableProperties.isCheckProcessDefinitions()
    );

    // 如果存在本地流程定义资源，则设置部署资源信息
    if (resources != null && !resources.isEmpty()) {
      conf.setDeploymentResources(resources.toArray(new Resource[0]));
      conf.setDeploymentName(flowableProperties.getDeploymentName());
    }

    // 配置异步执行器
    AsyncExecutor springAsyncExecutor = asyncExecutorProvider.getIfUnique();
    if (springAsyncExecutor != null) {
      conf.setAsyncExecutor(springAsyncExecutor);
    }

    // 配置历史数据异步处理器
    AsyncExecutor springAsyncHistoryExecutor = asyncHistoryExecutorProvider.getIfUnique();
    if (springAsyncHistoryExecutor != null) {
      conf.setAsyncHistoryEnabled(true);
      conf.setAsyncHistoryExecutor(springAsyncHistoryExecutor);
    }

    // 配置Spring引擎
    configureSpringEngine(conf, platformTransactionManager);
    // 配置数据源
    configureEngine(conf, dataSource);
    // 配置部署名称
    conf.setDeploymentName(defaultText(flowableProperties.getDeploymentName(), conf.getDeploymentName()));
    // 是否开启Idm引擎，目前不开启
    conf.setDisableIdmEngine(false);
    // 是否开启异步执行器
    conf.setAsyncExecutorActivate(flowableProperties.isAsyncExecutorActivate());
    // 是否开启历史数据异步处理器
    conf.setAsyncHistoryExecutorActivate(flowableProperties.isAsyncHistoryExecutorActivate());
    // 配置邮件地址
    conf.setMailServerHost(mailProperties.getHost());
    // 配置邮件端口
    conf.setMailServerPort(mailProperties.getPort());
    // 配置邮件用户名
    conf.setMailServerUsername(mailProperties.getUsername());
    // 配置邮件密码
    conf.setMailServerPassword(mailProperties.getPassword());
    // 配置邮件默认表单
    conf.setMailServerDefaultFrom(mailProperties.getDefaultFrom());
    // 配置发送邮件地址
    conf.setMailServerForceTo(mailProperties.getForceTo());
    // 配置邮件服务是否开启Ssl
    conf.setMailServerUseSSL(mailProperties.isUseSsl());
    // 配置邮件服务是否开启Tls
    conf.setMailServerUseTLS(mailProperties.isUseTls());

    // 是否开启流程定义的版本控制
    conf.setEnableProcessDefinitionHistoryLevel(processProperties.isEnableProcessDefinitionHistoryLevel());
    // 设置流程定义的缓存数量
    conf.setProcessDefinitionCacheLimit(processProperties.getDefinitionCacheLimit());
    // 是否开启安全的BPMN流程文件
    conf.setEnableSafeBpmnXml(processProperties.isEnableSafeXml());
    // 配置历史级别
    conf.setHistoryLevel(flowableProperties.getHistoryLevel());

    // 配置活动的字体
    conf.setActivityFontName(flowableProperties.getActivityFontName());
    // 配置备注的字体
    conf.setAnnotationFontName(flowableProperties.getAnnotationFontName());
    // 配置标签的字体
    conf.setLabelFontName(flowableProperties.getLabelFontName());

    // 配置ID生成器，如果没有配置ID生成器，则默认使用UUID生成器
    IdGenerator idGenerator = getIfAvailable(processIdGenerator, globalIdGenerator);
    if (idGenerator == null) {
      idGenerator = new StrongUuidGenerator();
    }
    conf.setIdGenerator(idGenerator);

    return conf;
  }

}
