package com.bizunited.platform.titan.starter.controller;

import com.bizunited.platform.core.controller.BaseController;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.titan.entity.ProcessInstanceEntity;
import com.bizunited.platform.titan.entity.ProcessTemplateEntity;
import com.bizunited.platform.titan.starter.service.ProcessInstanceService;
import com.bizunited.platform.titan.vo.ProcessImageVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.*;

import java.security.Principal;
import java.util.Date;
import java.util.Map;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-16 16:02
 * @Description: 流程实例功能业务模型的MVC Controller层实现，基于HTTP Restful风格
 */
@Api(tags = "流程实例功能业务模型的MVC Controller层实现")
@RestController
@RequestMapping("v1/titan/instances")
public class ProcessInstanceController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(ProcessInstanceController.class);

  @Autowired
  private ProcessInstanceService processInstanceService;

  /**
   * 根据模版ID开启流程
   * @param processTemplateId
   * @param variables
   * @return
   */
  @PostMapping("startProcess")
  @ApiOperation(value = "根据模版ID开启流程")
  public ResponseModel startProcess(@RequestParam(name = "processTemplateId") @ApiParam(value = "模版ID", required = true) String processTemplateId,
                                    @RequestBody Map<String, Object> variables){
    try {
      Principal principal = this.getPrincipal();
      ProcessInstanceEntity processInstance = processInstanceService.startProcess(processTemplateId, variables, principal);
      return buildHttpReslutW(processInstance, "processTemplate", "currentNodes", "latestNode");
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }


  /**
   * 条件分页查询流程实例
   * @param pageable
   * @param isApplicant
   * @param processKey
   * @param processName
   * @param formNo
   * @param applyDate
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "条件分页查询流程实例", notes = "如果是查我发起的流程，isApplicant参数传1 \n页数参数为page，第一页从0开始，分页大小参数为size（size默认为50）")
  public ResponseModel findByConditions(@PageableDefault(value = 50) Pageable pageable,
                                        @RequestParam(name = "isApplicant", required = false, defaultValue = "0") @ApiParam(value = "是否是发起人") Boolean isApplicant,
                                        @RequestParam(name = "processKey", required = false) @ApiParam(value = "模版key") String processKey,
                                        @RequestParam(name = "processName", required = false) @ApiParam(value = "模版名称") String processName,
                                        @RequestParam(name = "formNo", required = false) @ApiParam(value = "表单编号") String formNo,
                                        @RequestParam(name = "state", required = false) @ApiParam(value = "流程状态") Integer state,
                                        @RequestParam(name = "applyDate", required = false) @ApiParam(value = "申请日期,格式为:yyyy-MM-dd") @DateTimeFormat(pattern = "yyyy-MM-dd") Date applyDate){
    try {
      Principal principal = this.getPrincipal();
      ProcessInstanceEntity processInstance = new ProcessInstanceEntity();
      ProcessTemplateEntity processTemplate = new ProcessTemplateEntity();
      processTemplate.setProcessKey(processKey);
      processTemplate.setProcessName(processName);
      processInstance.setProcessTemplate(processTemplate);
      processInstance.setFormNo(formNo);
      processInstance.setProcessState(state);
      processInstance.setLatestSubmitTime(applyDate);
      Page<ProcessInstanceEntity> page = processInstanceService.findByConditions(pageable, processInstance, principal, isApplicant);
      return buildHttpReslutW(page, "processTemplate", "applicantUser", "latestAssignment", "currentAssignments", "latestNode", "currentNodes");
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据流程实例ID查询详情
   * @param processInstanceId
   * @return
   */
  @GetMapping("findDetailsByProcessInstanceId")
  @ApiOperation(value = "根据流程实例ID查询详情")
  public ResponseModel findDetailsByProcessInstanceId(@RequestParam(name = "processInstanceId") @ApiParam(value = "流程实例ID", required = true) String processInstanceId){
    try {
      ProcessInstanceEntity processInstance = processInstanceService.findDetailsByProcessInstanceId(processInstanceId);
      return buildHttpReslutW(processInstance, "records", "currentNodes", "latestNode", "currentAssignments",
          "latestAssignment", "applicantUser", "processTemplate", "records.operator", "records.processTemplateNode",
          "attachments", "attachments.ordinaryFile", "processTemplate.onSubmitScript");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据流程实例ID查询流程图片信息
   * @param processInstanceId
   * @return
   */
  @GetMapping("findImageInfoByProcessInstanceId")
  @ApiOperation(value = "根据流程实例ID查询流程图片信息")
  public ResponseModel findImageInfoByProcessInstanceId(@RequestParam(name = "processInstanceId") @ApiParam(value = "流程实例ID", required = true) String processInstanceId){
    try {
      ProcessImageVo processImage = processInstanceService.findImageInfoByProcessInstanceId(processInstanceId);
      return buildHttpReslutW(processImage, "imageNodes", "imageNodes.records", "imageNodes.records.operator",
          "imageNodes.node", "imageNodes.assignments", "imageNodes.assignmentNames", "imageNodes.node.processTemplateNodeMulti",
          "imageNodes.node.processTemplateNodePermission");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 设置流程变量
   * @param processInstanceId
   * @param variables
   * @return
   */
  @PatchMapping("setProcessVariables")
  @ApiOperation(value = "设置流程变量")
  public ResponseModel setProcessVariables(@RequestParam(name = "processInstanceId") @ApiParam(value = "流程实例ID", required = true) String processInstanceId,
                                           @RequestBody Map<String, Object> variables){
    try {
      variables = processInstanceService.setProcessVariables(processInstanceId, variables);
      return buildHttpReslut(variables);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 查询流程变量
   * @param processInstanceId
   * @return
   */
  @GetMapping("findVariablesByProcessInstanceId")
  @ApiOperation(value = "查询流程变量")
  public ResponseModel findVariablesByProcessInstanceId(@RequestParam(name = "processInstanceId") @ApiParam(value = "流程实例ID", required = true) String processInstanceId){
    try {
      Map<String, Object> variables = processInstanceService.findVariablesByProcessInstanceId(processInstanceId);
      return buildHttpReslut(variables);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

}
