package com.bizunited.platform.titan.starter.controller;

import com.bizunited.platform.core.controller.BaseController;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.service.ProcessTaskService;
import com.bizunited.platform.titan.starter.service.invoke.model.ProcessInputParamsModel;
import com.bizunited.platform.titan.vo.TaskVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.security.Principal;
import java.util.List;

/**
 * 流程任务功能业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @Author: Paul Chan
 * @Date: 2019-05-23 14:13
 */
@Api(tags = "流程任务功能业务模型的MVC Controller层实现")
@RestController
@RequestMapping("v1/titan/tasks")
public class ProcessTaskController extends BaseController {

  private final static Logger LOGGER = LoggerFactory.getLogger(ProcessTaskController.class);

  @Autowired
  private ProcessTaskService processTaskService;

  /**
   * 处理任务操作,通过责任链方式执行流程任务的处理和表单数据的更新
   * @param model
   * @return
   */
  @PostMapping("handleTask")
  @ApiOperation(value = "任务操作", notes = "支持操作：BTN_001(审批通过),BTN_002(驳回),BTN_003(回退),BTN_004(取回),BTN_011(提交),BTN_012(撤销),BTN_013(终止)")
  public ResponseModel handleTask(@RequestBody ProcessInputParamsModel model){
    try {
      Object result = processTaskService.handleTask(model);
      return buildHttpReslutW(result);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }


  /**
   * 处理任务转办
   * @param taskId
   * @param targetAssignment
   * @param content
   * @return
   */
  @PostMapping("handleTransferAssignee")
  @ApiOperation(value = "任务转办")
  public ResponseModel handleTransferAssignee(@RequestParam("taskId") @ApiParam(value = "任务ID", required = true)String taskId,
                                        @RequestParam("targetAssignment") @ApiParam(value = "任务转办目标用户", required = true)String targetAssignment,
                                        @RequestParam(value = "content", required = false) @ApiParam(value = "审批内容")String content){
    try {
      Principal principal = this.getPrincipal();
      processTaskService.handleTransferAssignee(taskId, targetAssignment, principal, content);
      return buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }


  /**
   * 处理任务抄送
   * @param taskId
   * @return
   */
  @PostMapping("handleCarbonCopy")
  @ApiOperation(value = "任务抄送")
  public ResponseModel handleCarbonCopy(@RequestParam("taskId") @ApiParam(value = "任务ID", required = true)String taskId,
                                  @RequestBody @ApiParam(value = "抄送的用户名组") List<String> assignments){
    try {
      Principal principal = this.getPrincipal();
      List<ProcessCarbonCopyEntity> carbonCopies = processTaskService.handleCarbonCopy(taskId, assignments, principal);
      return buildHttpReslutW(carbonCopies, "processInstance", "ccUser", "receiveAssignment");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 添加会签人员
   * @param taskId
   * @param assignments
   * @return
   */
  @PostMapping("handleAddMulti")
  @ApiOperation(value = "添加会签人员")
  public ResponseModel handleAddMulti(@RequestParam("taskId") @ApiParam(value = "任务ID", required = true)String taskId,
                                      @RequestParam(value = "content", required = false) @ApiParam(value = "意见")String content,
                                      @RequestBody @ApiParam(value = "会签人员") List<String> assignments){
    try {
      Principal principal = this.getPrincipal();
      processTaskService.handleAddMulti(taskId, assignments, principal, content);
      return buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 删除会签人员，目前只支持平行会签的减签
   * @param taskId
   * @param assignments
   * @return
   */
  @PostMapping("handleSubMulti")
  @ApiOperation(value = "删除会签人员")
  public ResponseModel handleSubMulti(@RequestParam("taskId") @ApiParam(value = "任务ID", required = true)String taskId,
                                @RequestParam(value = "content", required = false) @ApiParam(value = "意见")String content,
                                @RequestBody @ApiParam(value = "删除会签人员", required = true) List<String> assignments){
    try {
      Principal principal = this.getPrincipal();
      processTaskService.handleSubMulti(taskId, assignments, principal, content);
      return buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }


  /**
   * 查询我的待办
   * @param pageable
   * @param taskVo
   * @return
   */
  @GetMapping("findMyTasksByConditions")
  @ApiOperation(value = "查询我的待办")
  public ResponseModel findMyTasksByConditions(@PageableDefault(value = 50) Pageable pageable, TaskVo taskVo){
    try {
      Principal principal = this.getPrincipal();
      Page<TaskVo> carbonCopies = processTaskService.findMyTasksByConditions(taskVo, principal, pageable);
      return buildHttpReslutW(carbonCopies);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 查询我已办的任务
   * @param pageable
   * @param processKey
   * @param processName
   * @param formNo
   * @param state
   * @return
   */
  @GetMapping("findDoneByConditions")
  @ApiOperation(value = "查询我已办的任务", notes = "页数参数为page，第一页从0开始，分页大小参数为size（size默认为50）")
  public ResponseModel findDoneByConditions(@PageableDefault(value = 50) Pageable pageable,
                                            @RequestParam(name = "processKey", required = false) @ApiParam(value = "模版key") String processKey,
                                            @RequestParam(name = "processName", required = false) @ApiParam(value = "模版名称") String processName,
                                            @RequestParam(name = "formNo", required = false) @ApiParam(value = "表单编号") String formNo,
                                            @RequestParam(name = "state", required = false) @ApiParam(value = "流程状态") Integer state){
    try {
      Principal principal = this.getPrincipal();
      ProcessInstanceOperateRecordEntity record = new ProcessInstanceOperateRecordEntity();
      ProcessInstanceEntity processInstance = new ProcessInstanceEntity();
      ProcessTemplateEntity processTemplate = new ProcessTemplateEntity();
      processTemplate.setProcessKey(processKey);
      processTemplate.setProcessName(processName);
      processInstance.setProcessTemplate(processTemplate);
      processInstance.setFormNo(formNo);
      processInstance.setProcessState(state);
      record.setProcessInstance(processInstance);
      Page<ProcessInstanceOperateRecordEntity> page = processTaskService.findDoneByConditions(pageable, record, principal);
      return buildHttpReslutW(page, "processInstance", "processInstance.processTemplate", "processTemplateNode", "processInstance.applicantUser");
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 查询流程实例可回退的节点
   * @param processInstanceId
   * @return
   */
  @GetMapping("findBackableNodes")
  @ApiOperation(value = "查询流程实例可回退的节点")
  public ResponseModel findBackableNodes(@RequestParam("processInstanceId") @ApiParam(value = "流程实例ID") String processInstanceId){
    try {
      List<ProcessTemplateNodeEntity> nodes = processTaskService.findBackableNodes(processInstanceId);
      return buildHttpReslutW(nodes);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }


  /**
   * 根据实例ID和节点key查询任务信息
   * @param processInstanceId
   * @param taskKey
   * @return
   */
  @GetMapping("findTaskInfoByInstanceIdAndTaskKey")
  @ApiOperation(value = "根据实例ID和节点key查询任务信息")
  public ResponseModel findTaskInfoByInstanceIdAndTaskKey(@RequestParam("processInstanceId") @ApiParam(value = "流程实例ID") String processInstanceId,
                                                          @RequestParam("taskKey") @ApiParam(value = "流程节点key") String taskKey){
    try {
      TaskVo taskVO = processTaskService.findTaskInfoByInstanceIdAndTaskKey(processInstanceId, taskKey);
      return buildHttpReslutW(taskVO, "node", "node.processTemplateNodePermission", "node.processTemplateNodeMulti", "node.assignment", "node.processTemplateNodeMulti.assignments");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据节点实例ID查询任务信息
   * @param taskId
   * @return
   */
  @GetMapping("findTaskInfoByTaskId")
  @ApiOperation(value = "根据节点实例ID查询任务信息")
  public ResponseModel findTaskInfoByTaskId(@RequestParam("taskId") @ApiParam(value = "流程实例任务ID") String taskId){
    try {
      TaskVo taskVO = processTaskService.findTaskInfoByTaskId(taskId);
      return buildHttpReslutW(taskVO, "node", "node.processTemplateNodePermission", "node.processTemplateNodeMulti", "node.assignment", "node.processTemplateNodeMulti.assignments");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据节点实例ID查询可减签人员
   * @param taskId
   * @return
   */
  @GetMapping("findCanSubMultiAssignments")
  @ApiOperation(value = "根据节点实例ID查询可减签人员")
  public ResponseModel findCanSubMultiAssignments(@RequestParam("taskId") @ApiParam(value = "流程实例任务ID") String taskId){
    try {
      Principal principal = this.getPrincipal();
      List<ProcessAssignmentEntity> assignments = processTaskService.findCanSubMultiAssignments(taskId, principal);
      return buildHttpReslutW(assignments);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据节点实例ID查询会签人员
   * @param taskId
   * @return
   */
  @GetMapping("findTaskMultiAssignments")
  @ApiOperation(value = "根据节点实例ID查询会签人员")
  public ResponseModel findTaskMultiAssignments(@RequestParam("taskId") @ApiParam(value = "流程实例任务ID") String taskId){
    try {
      List<ProcessAssignmentEntity> assignments = processTaskService.findTaskMultiAssignments(taskId);
      return buildHttpReslutW(assignments);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

}
