package com.bizunited.platform.titan.starter.controller;

import com.bizunited.platform.core.controller.BaseController;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.titan.entity.ProcessTemplateEntity;
import com.bizunited.platform.titan.starter.service.ProcessTemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.security.Principal;
import java.util.List;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-13 10:57
 * @Description: 流程模版功能业务模型的MVC Controller层实现，基于HTTP Restful风格
 */
@Api(tags = "流程模版功能业务模型的MVC Controller层实现")
@RestController
@RequestMapping("v1/titan/templates")
public class ProcessTemplateController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(ProcessTemplateController.class);

  /**
   * 流程模版输入的属性白名单
   */
  private static final String[] PROCESS_TEMPLATE_PROPERTIES_W = new String[]{"createUser", "modifyUser", "processTemplateNodes", "processTemplateNodes.processTemplateNodePermission",
      "processTemplateNodes.assignment", "processTemplateNodes.assignment.processListener", "processTemplateNodes.assignment.script", "processTemplateNodes.assignment.variables",
      "permissions", "processTemplateListeners", "processTemplateListeners.processListener", "processTemplateListeners.script", "processTemplateListeners.variables", "onSubmitScript",
      "processTemplateNodes.processTemplateNodeMulti", "processTemplateNodes.processTemplateNodeMulti.assignments", "variables"};

  @Autowired
  private ProcessTemplateService processTemplateService;

  /**
   * 创建流程模版，新的流程key必须在之前的库中是不存在的
   * @param processTemplateEntity
   * @return
   */
  @PostMapping("")
  @ApiOperation(value = "创建流程模版", notes = "创建流程模版，新的流程key必须在之前的库中是不存在的")
  public ResponseModel create(@RequestBody ProcessTemplateEntity processTemplateEntity){
    try {
      Principal principal = this.getPrincipal();
      ProcessTemplateEntity processTemplate = processTemplateService.create(processTemplateEntity, principal, 1);
      return buildHttpReslutW(processTemplate, PROCESS_TEMPLATE_PROPERTIES_W);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 更新流程模版
   * @param processTemplateEntity
   * @return
   */
  @PatchMapping("")
  @ApiOperation(value = "更新流程模版")
  public ResponseModel update(@RequestBody ProcessTemplateEntity processTemplateEntity){
    try {
      Principal principal = this.getPrincipal();
      ProcessTemplateEntity processTemplate = processTemplateService.update(processTemplateEntity, principal);
      return buildHttpReslutW(processTemplate, PROCESS_TEMPLATE_PROPERTIES_W);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 分页查询流程模版
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "分页查询流程模版", notes = "页数参数为page，第一页从0开始，分页大小参数为size（size默认为50）")
  public ResponseModel findByConditions(@PageableDefault(value = 50) Pageable pageable,
                                        @ApiParam(value = "流程key") @RequestParam(value = "processKey", required = false) String processKey,
                                        @ApiParam(value = "流程名称") @RequestParam(value = "processName", required = false) String processName,
                                        @ApiParam(value = "表单名称") @RequestParam(value = "formName", required = false) String formName,
                                        @ApiParam(value = "是否是最后版本") @RequestParam(value = "isLastVersion", required = false) Boolean isLastVersion){
    try {
      ProcessTemplateEntity processTemplate = new ProcessTemplateEntity();
      processTemplate.setProcessKey(processKey);
      processTemplate.setProcessName(processName);
      processTemplate.setFormName(formName);
      processTemplate.setLastVersion(isLastVersion);
      Page<ProcessTemplateEntity> page = processTemplateService.findByConditions(pageable, processTemplate);
      return buildHttpReslutW(page);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 根据流程key查询流程模版
   * @param processKey
   * @return
   */
  @GetMapping("findByProcessKey")
  @ApiOperation(value = "根据流程key查询流程模版", notes = "根据版本号倒序排序")
  public ResponseModel findByProcessKey(@ApiParam(value = "流程key") @RequestParam(value = "processKey", required = false) String processKey){
    try {
      List<ProcessTemplateEntity> processTemplates = processTemplateService.findByProcessKey(processKey);
      return buildHttpReslutW(processTemplates);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据ID查询流程模版明细，包含模版下的所有关联信息
   * @param id
   * @return
   */
  @GetMapping("findDetailsById")
  @ApiOperation(value = "根据ID查询流程模版明细", notes = "包含模版下的所有关联信息")
  public ResponseModel findDetailsById(@ApiParam(value = "流程模版ID", required = true) @RequestParam("id") String id){
    try {
      ProcessTemplateEntity processTemplate = processTemplateService.findDetailsById(id);
      return buildHttpReslutW(processTemplate, PROCESS_TEMPLATE_PROPERTIES_W);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 升级流程版本,以最新模版为本升级版本，升级版本要求当前无正在编辑的模版
   * @param processTemplate 流程模版
   * @return
   */
  @PutMapping("upgradeVersion")
  @ApiOperation(value = "升级流程版本", notes = "以最新模版为本升级版本，升级版本要求当前无正在编辑的模版")
  public ResponseModel upgradeVersion(@RequestBody ProcessTemplateEntity processTemplate){
    try {
      Principal principal = this.getPrincipal();
      ProcessTemplateEntity processTemplateEntity = processTemplateService.upgradeVersion(processTemplate, principal);
      return buildHttpReslutW(processTemplateEntity, PROCESS_TEMPLATE_PROPERTIES_W);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 发布流程到流程引擎，发布后该流程模版将不可在进行编辑
   * @param id
   * @return
   */
  @PostMapping("deploy")
  @ApiOperation(value = "发布流程到流程引擎", notes = "发布流程到流程引擎，发布后该流程模版将不可在进行编辑")
  public ResponseModel deploy(@ApiParam(value = "流程模版ID", required = true) @RequestParam("id") String id){
    try {
      Principal principal = this.getPrincipal();
      processTemplateService.deploy(id, principal);
      return buildHttpReslut();
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 根据ID删除流程模版，只能删除未发布的流程模版
   * @param id
   * @return
   */
  @DeleteMapping("deleteById")
  @ApiOperation(value = "根据ID删除流程模版", notes = "根据ID删除流程模版，只能删除未发布的流程模版")
  public ResponseModel deleteById(@ApiParam(value = "流程模版ID", required = true) @RequestParam("id") String id){
    try {
      Principal principal = this.getPrincipal();
      processTemplateService.deleteById(id, principal);
      return buildHttpReslut();
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 分页查询可发起的流程模版
   * @param pageable
   * @param processKey
   * @param processName
   * @param formName
   * @return
   */
  @GetMapping("findStartableByConditions")
  @ApiOperation(value = "分页查询可发起的流程模版", notes = "页数参数为page，第一页从0开始，分页大小参数为size（size默认为50）")
  public ResponseModel findStartableByConditions(@PageableDefault(value = 50) Pageable pageable,
                                                 @ApiParam(value = "流程key") @RequestParam(value = "processKey", required = false) String processKey,
                                                 @ApiParam(value = "流程名称") @RequestParam(value = "processName", required = false) String processName,
                                                 @ApiParam(value = "表单名称") @RequestParam(value = "formName", required = false) String formName){
    try {
      Principal principal = this.getPrincipal();
      ProcessTemplateEntity processTemplate = new ProcessTemplateEntity();
      processTemplate.setProcessKey(processKey);
      processTemplate.setProcessName(processName);
      processTemplate.setFormName(formName);
      Page<ProcessTemplateEntity> page = processTemplateService.findStartableByConditions(pageable, processTemplate, principal);
      return buildHttpReslutW(page);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }


  /**
   * 获取流程xml
   * @param id
   * @return
   */
  @GetMapping("findProcessXmlById")
  @ApiOperation(value = "获取流程xml")
  public ResponseModel findProcessXmlById(String id){
    try {
      String processXml = processTemplateService.findProcessXmlById(id);
      return buildHttpReslutW(processXml);
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }


}
