package com.bizunited.platform.titan.starter.repository.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.bizunited.platform.titan.entity.ProcessInstanceEntity;
import com.bizunited.platform.titan.entity.ProcessInstanceOperateRecordEntity;
import com.bizunited.platform.titan.entity.ProcessTemplateEntity;
import com.bizunited.platform.titan.starter.service.TitanToolkitService;
import com.bizunited.platform.titan.vo.TaskVo;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.query.internal.NativeQueryImpl;
import org.hibernate.transform.Transformers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.bizunited.platform.titan.starter.common.Constants.DEFAULT_QUERY_COVER_CODE;
import static com.bizunited.platform.titan.starter.common.Constants.USERNAME_PREFIX;

/**
 * 流程任务持久化实现
 * @Author: Paul Chan
 * @Date: 2019-05-25 20:18
 */
@Repository("ProcessTaskRepositoryImpl")
public class ProcessTaskRepositoryImpl implements ProcessTaskRepositoryCustom {

  @Autowired
  private EntityManager entityManager;
  @Autowired
  @Qualifier("TitanToolkitService")
  private TitanToolkitService titanToolkitService;

  /**
   * 用原生sql查询我的待办任务
   * 关联表：
   *    ACT_RU_TASK：流程引擎实时任务表
   *    engine_process_instance：流程实例表
   *    engine_user：用户表
   *    engine_process_template：流程模版表
   * 具体字段信息请查阅com.bizunited.platform.titan.vo.TaskVO字段
   *
   * @param user
   * @param taskVo
   * @param pageable
   * @return
   */
  @SuppressWarnings("unchecked")
  @Override
  public Page<TaskVo> findMyTasksByConditions(UserVo user, TaskVo taskVo, Pageable pageable) {
    StringBuilder sql = new StringBuilder("SELECT art.ID_ AS taskId, art.TASK_DEF_KEY_ AS taskDefinitionKey, art.NAME_ AS taskName, ");
    sql.append(" ept.process_name AS processName, epi.form_no AS formNo, art.PROC_INST_ID_ AS processInstanceId,");
    sql.append(" epi.process_state AS processState, eu.user_name AS applicantUserName, epi.create_time AS createTime, ept.process_key AS processKey");
    sql.append(" FROM ACT_RU_TASK art ");
    sql.append(" inner JOIN engine_process_instance epi ON art.PROC_INST_ID_ = epi.process_instance_id");
    sql.append(" inner JOIN engine_user eu ON eu.id = epi.applicant_user");
    sql.append(" inner JOIN engine_process_template ept ON ept.id = epi.process_template");
    sql.append(" WHERE (art.ASSIGNEE_ = :userAccount or art.ASSIGNEE_ in :positionCodes ) ");
    StringBuilder countSql = new StringBuilder("SELECT COUNT(*) ");
    countSql.append(" FROM ACT_RU_TASK art ");
    countSql.append(" inner JOIN engine_process_instance epi ON art.PROC_INST_ID_ = epi.process_instance_id");
    countSql.append(" inner JOIN engine_user eu ON eu.id = epi.applicant_user");
    countSql.append(" inner JOIN engine_process_template ept ON ept.id = epi.process_template");
    countSql.append(" WHERE (art.ASSIGNEE_ = :userAccount or art.ASSIGNEE_ in :positionCodes) ");
    StringBuilder conditions = new StringBuilder();
    Map<String, Object> parameter = new HashMap<>(16);
    parameter.put("userAccount", StringUtils.join(USERNAME_PREFIX, user.getAccount()));
    List<String> positionCodes = titanToolkitService.getAssignmentPositionCodes(user.getPositions());
    if(CollectionUtils.isEmpty(positionCodes)) positionCodes.add(DEFAULT_QUERY_COVER_CODE);
    parameter.put("positionCodes", positionCodes);
    if(taskVo != null){
      if(StringUtils.isNotBlank(taskVo.getFormNo())){
        // 表单编号条件
        conditions.append(" and epi.form_no = :formNo");
        parameter.put("formNo", taskVo.getFormNo());
      }
      if(StringUtils.isNotBlank(taskVo.getProcessKey())){
        // 流程key条件
        conditions.append(" and ept.process_key = :processKey");
        parameter.put("processKey", taskVo.getProcessKey());
      }
      if(StringUtils.isNotBlank(taskVo.getProcessName())){
        // 流程名称条件
        conditions.append(" and ept.process_name = :processName");
        parameter.put("processName", taskVo.getProcessName());
      }
      if(StringUtils.isNotBlank(taskVo.getProcessInstanceId())){
        // 流程实例ID条件
        conditions.append(" and art.PROC_INST_ID_ = :processInstanceId");
        parameter.put("processInstanceId", taskVo.getProcessInstanceId());
      }
      if(taskVo.getProcessState() != null){
        // 流程状态条件
        conditions.append(" and epi.process_state = :processState");
        parameter.put("processState", taskVo.getProcessState());
      }
      if(StringUtils.isNotBlank(taskVo.getTaskName())){
        // 任务名称条件
        conditions.append(" and art.NAME_ = :taskName");
        parameter.put("taskName", taskVo.getTaskName());
      }
      if(StringUtils.isNotBlank(taskVo.getTaskId())){
        // 任务ID条件
        conditions.append(" and art.ID_ = :taskId");
        parameter.put("taskId", taskVo.getTaskId());
      }
      if(taskVo.getCreateTime() != null){
        // 流程实例创建时间条件
        conditions.append(" and epi.create_time >= :createTime");
        parameter.put("createTime", taskVo.getCreateTime());
      }
    }
    sql.append(conditions).append(" order by art.CREATE_TIME_ ASC");
    countSql.append(conditions);
    Query query = entityManager.createNativeQuery(sql.toString());
    query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.aliasToBean(TaskVo.class));
    Query countQuery = entityManager.createNativeQuery(countSql.toString());
    parameter.forEach((k, v) -> {
      query.setParameter(k, v);
      countQuery.setParameter(k, v);
    });
    query.setFirstResult(pageable.getPageNumber() * pageable.getPageSize());
    query.setMaxResults(pageable.getPageSize());
    List<TaskVo> result = query.getResultList();
    long count = ((BigInteger)countQuery.getResultList().get(0)).longValue();
    PageImpl<TaskVo> currentPage = new PageImpl<>(result ,pageable, count);
    return currentPage;
  }

  @SuppressWarnings("unchecked")
  @Override
  public Page<ProcessInstanceOperateRecordEntity> findDoneByConditions(Pageable pageable, ProcessInstanceOperateRecordEntity record, UserEntity user) {
    StringBuilder hql = new StringBuilder("select pior from ProcessInstanceOperateRecordEntity pior ");
    hql.append(" inner join fetch pior.processInstance pi ");
    hql.append(" inner join fetch pior.processTemplateNode ptn ");
    hql.append(" inner join fetch pi.processTemplate pt");
    hql.append(" inner join pior.operator pioro");
    hql.append(" inner join fetch pi.applicantUser piau");
    hql.append(" where pioro.id = :operator and pior.createTime >= pi.latestSubmitTime and pior.operation in ('BTN_001','BTN_002','BTN_003','BTN_012','BTN_013') ");
    StringBuilder countHql = new StringBuilder("select count(*) from ProcessInstanceOperateRecordEntity pior ");
    countHql.append(" inner join pior.processInstance pi ");
    countHql.append(" inner join pior.processTemplateNode ptn ");
    countHql.append(" inner join pi.processTemplate pt");
    countHql.append(" inner join pior.operator pioro");
    countHql.append(" inner join pi.applicantUser piau");
    countHql.append(" where pioro.id = :operator and pior.createTime >= pi.latestSubmitTime and pior.operation in ('BTN_001','BTN_002','BTN_003','BTN_012','BTN_013') ");
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameter = new HashMap<>(16);
    parameter.put("operator", user.getId());
    ProcessInstanceEntity processInstance = record.getProcessInstance();
    if(processInstance != null){
      if(processInstance.getProcessState() != null){
        condition.append(" and pi.processState = :processState ");
        parameter.put("processState", processInstance.getProcessState());
      }
      if(StringUtils.isNotBlank(processInstance.getFormNo())){
        condition.append(" and pi.formNo = :formNo ");
        parameter.put("formNo", processInstance.getFormNo());
      }
      ProcessTemplateEntity processTemplate = processInstance.getProcessTemplate();
      if(processTemplate != null){
        if(StringUtils.isNotBlank(processTemplate.getProcessKey())){
          condition.append(" and pt.processKey = :processKey ");
          parameter.put("processKey", processTemplate.getProcessKey());
        }
        if(StringUtils.isNotBlank(processTemplate.getProcessName())){
          condition.append(" and pt.processName = :processName ");
          parameter.put("processName", processTemplate.getProcessName());
        }
      }
    }
    hql.append(condition.toString());
    countHql.append(condition.toString());
    hql.append(" order by pior.createTime desc");
    Query query = entityManager.createQuery(hql.toString());
    Query countQuery = entityManager.createQuery(countHql.toString());
    parameter.forEach((k, v) -> {
      query.setParameter(k, v);
      countQuery.setParameter(k, v);
    });
    query.setFirstResult(pageable.getPageNumber() * pageable.getPageSize());
    query.setMaxResults(pageable.getPageSize());
    List<ProcessInstanceOperateRecordEntity> result = query.getResultList();
    long count = (long)countQuery.getResultList().get(0);
    PageImpl<ProcessInstanceOperateRecordEntity> currentPage = new PageImpl<>(result ,pageable, count);
    return currentPage;
  }

}

