package com.bizunited.platform.titan.starter.service;

import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.service.invoke.model.ProcessInputParamsModel;
import com.bizunited.platform.titan.vo.TaskVo;
import org.flowable.task.api.Task;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.security.Principal;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 流程任务服务的接口定义
 * @Author: Paul Chan
 * @Date: 2019-05-23 10:07
 */
public interface ProcessTaskService {

  /**
   * 提交表单
   * @param processInstanceId
   * @param principal
   * @param variables
   */
  void handleSubmitForm(String processInstanceId, Principal principal, Map<String, Object> variables);

  /**
   * 审批通过
   * @param taskId
   * @param principal
   * @param content
   * @param variables
   */
  void handleComplete(String taskId, Principal principal, String content, Map<String, Object> variables);

  /**
   * 驳回,除了需要删除当前的任务节点，同时还需要删除在当前任务节点后生成activity实例
   * @param taskId
   * @param principal
   * @param content
   * @param variables
   */
  void handleReject(String taskId, Principal principal, String content, Map<String, Object> variables);

  /**
   * 任务转办
   * @param taskId
   * @param targetAssignment
   * @param principal
   * @param content
   */
  void handleTransferAssignee(String taskId, String targetAssignment, Principal principal, String content);

  /**
   * 任务回退
   * @param taskId
   * @param targetActivitiId
   * @param principal
   * @param content
   * @param variables
   */
  void handleBackTask(String taskId, String targetActivitiId, Principal principal, String content, Map<String, Object> variables);

  /**
   * 抄送任务
   * @param taskId
   * @param assignments
   * @param principal
   * @return
   */
  List<ProcessCarbonCopyEntity> handleCarbonCopy(String taskId, List<String> assignments, Principal principal);

  /**
   * 获取流程实例当前的任务节点
   * @param instanceEntity
   * @return
   */
  Set<ProcessTemplateNodeEntity> findCurrentNodes(ProcessInstanceEntity instanceEntity);

  /**
   * 获取流程实例当前的任务人,返回的是持久化的对象
   * @param instanceEntity
   * @return
   */
  Set<ProcessAssignmentEntity> findCurrentAssignments(ProcessInstanceEntity instanceEntity);

  /**
   * 查询我的待办
   * @param taskVo
   * @param principal
   * @param pageable
   * @return
   */
  Page<TaskVo> findMyTasksByConditions(TaskVo taskVo, Principal principal, Pageable pageable);

  /**
   * 终止流程
   * @param taskId
   * @param principal
   * @param content
   */
  void handleStop(String taskId, Principal principal, String content);

  /**
   * 获取实例当前的任务节点
   * @param processInstanceId
   * @return
   */
  List<Task> findCurrentTasks(String processInstanceId);

  /**
   * 撤销流程
   * @param processInstanceId
   * @param principal
   * @param content
   * @param variables 流程变量,map的json格式
   */
  void handleCancel(String processInstanceId, Principal principal, String content, Map<String, Object> variables);

  /**
   * 取回任务，能取回的前提是后面没有已完成的任务
   * @param processInstanceId
   * @param activityId
   * @param principal
   */
  void handleRetrieve(String processInstanceId, String activityId, Principal principal);

  /**
   * 增加会签人员
   * 处理串行会签的加签,串行会签在加签时不能确认产生出来的任务，而平行会签可以，所以处理方式不一样,串行会签则需要给流程引擎预设会签的人员，设置串行会签任务的任务人需要在审批通过后设置
   * 处理平行会签的加签,平行会签在加签的同时就能确认产生出来的任务，而串行会签不行，所以处理方式不一样,平行会签加签在任务生成后直接设置审批人信息
   * @param taskId
   * @param assignments
   * @param principal
   * @param content
   */
  void handleAddMulti(String taskId, List<String> assignments, Principal principal, String content);

  /**
   * 减会签人员
   * @param taskId
   * @param assignments
   * @param principal
   * @param content
   */
  void handleSubMulti(String taskId, List<String> assignments, Principal principal, String content);

  /**
   * 操作流程任务，目前支持的操作有：BTN_001(审批通过),BTN_002(驳回),BTN_003(回退),BTN_004(取回),BTN_011(提交),BTN_012(撤销),BTN_013(终止)
   * 采用责任链处理任务，先处理流程的任务，然后再保存表单的数据
   * @param model
   * @return
   */
  Object handleTask(ProcessInputParamsModel model) throws InvokeProxyException;

  /**
   * 查询流程实例可回退的节点
   * @param processInstanceId
   * @return
   */
  List<ProcessTemplateNodeEntity> findBackableNodes(String processInstanceId);

  /**
   * 查询我已办的任务
   * @param pageable
   * @param record
   * @param principal
   * @return
   */
  Page<ProcessInstanceOperateRecordEntity> findDoneByConditions(Pageable pageable, ProcessInstanceOperateRecordEntity record, Principal principal);


  /**
   * 根据实例ID和节点key查询任务信息
   * @param processInstanceId
   * @param taskKey
   * @return
   */
  TaskVo findTaskInfoByInstanceIdAndTaskKey(String processInstanceId, String taskKey);

  /**
   * 根据节点实例ID查询任务信息
   * @param taskId
   * @return
   */
  TaskVo findTaskInfoByTaskId(String taskId);

  /**
   * 根据节点实例ID查询可减签人员
   * @param taskId
   * @param principal
   * @return
   */
  List<ProcessAssignmentEntity> findCanSubMultiAssignments(String taskId, Principal principal);

  /**
   * 根据节点实例ID查询会签人员
   * @param taskId
   * @return
   */
  List<ProcessAssignmentEntity> findTaskMultiAssignments(String taskId);
}
