package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.entity.UuidEntity;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.bizunited.platform.titan.entity.ProcessAssignmentEntity;
import com.bizunited.platform.titan.starter.repository.ProcessAssignmentRepository;
import com.bizunited.platform.titan.starter.service.ProcessAssignmentService;
import com.bizunited.platform.titan.starter.service.TitanToolkitService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;
import java.util.stream.Collectors;

import static com.bizunited.platform.titan.starter.common.Constants.POSITION_PREFIX;
import static com.bizunited.platform.titan.starter.common.Constants.USERNAME_PREFIX;

/**
 * 流程审批人服务实现
 * @Author: Paul Chan
 * @Date: 2019-05-31 10:45
 */
@Service("ProcessAssignmentServiceImpl")
public class ProcessAssignmentServiceImpl extends BaseService implements ProcessAssignmentService {

  @Autowired
  private ProcessAssignmentRepository processAssignmentRepository;

  @Autowired
  @Qualifier("TitanToolkitService")
  private TitanToolkitService titanToolkitService;

  /**
   * 验证保存的数据
   * @param assignments
   */
  private void saveValidation(Set<ProcessAssignmentEntity> assignments){
    for (ProcessAssignmentEntity assignment : assignments) {
      this.saveValidation(assignment);
    }
  }

  /**
   * 验证保存的数据
   * @param assignment
   */
  private void saveValidation(ProcessAssignmentEntity assignment){
    this.valid(assignment.getAssignment());
    Validate.notBlank(assignment.getAssignmentName(), "审批代理人名称不能为空");
  }

  @Override
  @Transactional
  public Set<ProcessAssignmentEntity> save(String resourceId, Set<ProcessAssignmentEntity> assignments) {
    if(assignments == null) assignments = new HashSet<>();
    Validate.notBlank(resourceId, "数据源ID不能为空");
    this.saveValidation(assignments);
    Map<String, ProcessAssignmentEntity> assignmentsMap = assignments.stream().collect(Collectors.toMap(UuidEntity::getId, assignment -> assignment, (a, b) -> b, () -> new HashMap<>(8)));
    Set<ProcessAssignmentEntity> saveAssignments = new HashSet<>();
    Set<ProcessAssignmentEntity> oldAssignments = processAssignmentRepository.findByResourceId(resourceId);
    Set<ProcessAssignmentEntity> addEntities = new HashSet<>();
    Set<ProcessAssignmentEntity> deleteEntities = new HashSet<>();
    Set<ProcessAssignmentEntity> updateEntities = new HashSet<>();
    titanToolkitService.collectionDiscrepancy(assignments, oldAssignments, ProcessAssignmentEntity::getId, deleteEntities, updateEntities, addEntities);
    Date date = new Date();
    // 保存新增的数据
    for (ProcessAssignmentEntity addEntity : addEntities) {
      addEntity.setCreateTime(date);
      addEntity.setModifyTime(date);
      addEntity.setResourceId(resourceId);
      addEntity.setSort(addEntity.getSort() == null ? 1 : addEntity.getSort());
      processAssignmentRepository.save(addEntity);
      saveAssignments.add(addEntity);
    }
    // 更新数据
    for (ProcessAssignmentEntity updateEntity : updateEntities) {
      ProcessAssignmentEntity assignment = assignmentsMap.get(updateEntity.getId());
      updateEntity.setModifyTime(new Date());
      updateEntity.setAssignment(assignment.getAssignment());
      updateEntity.setAssignmentName(assignment.getAssignmentName());
      updateEntity.setSort(assignment.getSort() == null? updateEntity.getSort() : assignment.getSort());
      processAssignmentRepository.save(updateEntity);
      saveAssignments.add(updateEntity);
    }
    processAssignmentRepository.deleteAll(deleteEntities);
    return saveAssignments;
  }

  @Override
  @Transactional
  public ProcessAssignmentEntity save(String resourceId, ProcessAssignmentEntity assignment){
    if(assignment == null) return null;
    this.saveValidation(assignment);
    Long count = processAssignmentRepository.countByResourceId(resourceId);
    Validate.isTrue(count <= 1, "该资源下有多个审批人，不符合一对一关系，请用ProcessAssignmentService#save(java.lang.String, java.util.Set)方法保存");
    ProcessAssignmentEntity entity = processAssignmentRepository.findOneByResourceId(resourceId);
    Date date = new Date();
    if(entity == null){
      entity = assignment;
      entity.setId(null);
      entity.setSort(entity.getSort() == null ? 1 : entity.getSort());
      entity.setResourceId(resourceId);
      entity.setCreateTime(date);
    } else {
      entity.setSort(assignment.getSort() == null? entity.getSort() : assignment.getSort());
      entity.setAssignment(assignment.getAssignment());
      entity.setAssignmentName(assignment.getAssignmentName());
    }
    entity.setModifyTime(date);
    processAssignmentRepository.save(entity);
    return entity;
  }

  @Override
  @Transactional
  public void deleteByResourceId(String resourceId) {
    processAssignmentRepository.deleteByResourceId(resourceId);
  }

  @Override
  public Set<ProcessAssignmentEntity> findByResourceId(String resourceId) {
    Sort sort = new Sort(Sort.Direction.ASC, "sort");
    return processAssignmentRepository.findByResourceId(resourceId, sort);
  }

  @Override
  public boolean equals(UserEntity user, String assignment) {
    if(StringUtils.isBlank(assignment)) return false;
    if(assignment.startsWith(USERNAME_PREFIX)){
      // 用户名
      return assignment.equals(StringUtils.join(USERNAME_PREFIX, user.getAccount()));
    }
    if(assignment.startsWith(POSITION_PREFIX)){
      // 职位
      UserVo userVo = userService.findDetailsById(user.getId());
      Set<PositionVo> positions = userVo.getPositions();
      if(!CollectionUtils.isEmpty(positions)){
        for (PositionVo position : positions) {
          if(assignment.equals(StringUtils.join(POSITION_PREFIX, position.getCode()))){
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * 解析代理人对象
   * @param assignment
   * @return
   */
  @Override
  public ProcessAssignmentEntity findAssignment(String assignment){
    if(StringUtils.isBlank(assignment)) return null;
    ProcessAssignmentEntity assignmentEntity = new ProcessAssignmentEntity();
    if(assignment.startsWith(USERNAME_PREFIX)){
      String account = assignment.replaceFirst(USERNAME_PREFIX, "");
      UserVo userVo = userService.findByAccount(account);
      Validate.notNull(userVo, "未找到用户：%s", account);
      assignmentEntity.setAssignment(assignment);
      assignmentEntity.setAssignmentName(userVo.getUserName());
    } else if(assignment.startsWith(POSITION_PREFIX)){
      String positionCode = assignment.replaceFirst(POSITION_PREFIX, "");
      PositionVo position = super.findPositionByCode(positionCode);
      Validate.notNull(position, "未找到职位信息：%s", positionCode);
      assignmentEntity.setAssignment(assignment);
      assignmentEntity.setAssignmentName(position.getName());
    } else {
      throw new IllegalArgumentException(String.format("错误的代理人：%s,必须以%s或%s开头", assignment, USERNAME_PREFIX, POSITION_PREFIX));
    }
    assignmentEntity.setSort(1);
    assignmentEntity.setCreateTime(new Date());
    assignmentEntity.setModifyTime(new Date());
    return assignmentEntity;
  }

  @Override
  @Transactional
  public ProcessAssignmentEntity save(String resourceId, String assignment) {
    ProcessAssignmentEntity assignmentEntity = this.findAssignment(assignment);
    return this.save(resourceId, assignmentEntity);
  }

  @Override
  public boolean contains(List<String> assignments, UserEntity user) {
    if(!CollectionUtils.isEmpty(assignments)){
      for (String assignment : assignments) {
        if(equals(user, assignment)){
          return true;
        }
      }
    }
    return false;
  }

  @Override
  public void valid(List<String> assignments){
    Validate.notEmpty(assignments, "代理人不能为空");
    for (String assignment : assignments) {
      this.valid(assignment);
    }
  }

  @Override
  public void valid(String assignment){
    Validate.notBlank(assignment, "代理人不能为空");
    Validate.isTrue(assignment.startsWith(USERNAME_PREFIX) || assignment.startsWith(POSITION_PREFIX), "代理人【%s】错误，代理人必须以%s或%s开头", assignment, USERNAME_PREFIX, POSITION_PREFIX);
  }

}
