package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.bizunited.platform.titan.entity.ProcessAssignmentEntity;
import com.bizunited.platform.titan.entity.ProcessCarbonCopyEntity;
import com.bizunited.platform.titan.entity.ProcessInstanceEntity;
import com.bizunited.platform.titan.starter.common.enums.TaskOperateBtn;
import com.bizunited.platform.titan.starter.repository.ProcessCarbonCopyRepository;
import com.bizunited.platform.titan.starter.service.ProcessAssignmentService;
import com.bizunited.platform.titan.starter.service.ProcessCarbonCopyService;
import com.bizunited.platform.titan.starter.service.ProcessInstanceOperateRecordService;
import org.apache.commons.lang3.Validate;
import org.flowable.task.api.Task;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.*;

/**
 * 流程任务抄送服务的接口实现
 * @Author: Paul Chan
 * @Date: 2019-05-24 15:58
 */
@Service("ProcessCarbonCopyServiceImpl")
public class ProcessCarbonCopyServiceImpl extends BaseService implements ProcessCarbonCopyService {

  @Autowired
  private ProcessCarbonCopyRepository processCarbonCopyRepository;

  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessInstanceOperateRecordService processInstanceOperateRecordService;

  @Override
  public List<ProcessCarbonCopyEntity> create(Task task, ProcessInstanceEntity processInstance, UserEntity ccUser, List<String> assignments) {
    Validate.notEmpty(assignments, "任务抄送接收人不能为空");
    processAssignmentService.valid(assignments);
    List<ProcessCarbonCopyEntity> carbonCopies = new ArrayList<>(assignments.size());
    StringBuilder content =  new StringBuilder("抄送流程任务给用户【");
    Set<String> assignmentsSet = new HashSet<>();
    for (int i = 0; i < assignments.size(); i++) {
      String assignment = assignments.get(i);
      Validate.notBlank(assignment, "任务抄送接收人对象不能为空");
      Validate.isTrue(!assignmentsSet.contains(assignment), "有重复的抄送人:%s", assignment);
      Validate.isTrue(!processAssignmentService.equals(ccUser, assignment), "不能抄送给任务人自己");
      assignmentsSet.add(assignment);
      Long count = processCarbonCopyRepository.countByTaskIdAndAssignment(task.getId(), assignment);
      Validate.isTrue(count == null || count < 1, "用户【%s】已抄送过，请勿重复抄送", assignment);
      ProcessCarbonCopyEntity carbonCopy = new ProcessCarbonCopyEntity();
      carbonCopy.setCcUser(ccUser);
      carbonCopy.setCreateTime(new Date());
      carbonCopy.setProcessInstance(processInstance);
      carbonCopy.setState(0);
      carbonCopy.setTaskDefId(task.getTaskDefinitionKey());
      carbonCopy.setTaskId(task.getId());
      processCarbonCopyRepository.save(carbonCopy);
      ProcessAssignmentEntity assignmentEntity = processAssignmentService.save(carbonCopy.getId(), assignment);
      carbonCopy.setReceiveAssignment(assignmentEntity);
      carbonCopies.add(carbonCopy);
      if(i > 0) content.append("，");
      content.append(assignmentEntity.getAssignmentName());
    }
    content.append("】");
    processCarbonCopyRepository.saveAll(carbonCopies);
    processInstanceOperateRecordService.create(task, processInstance, ccUser, TaskOperateBtn.BTN_006, content.toString());
    return carbonCopies;
  }

  @Override
  public Page<ProcessCarbonCopyEntity> findMyByConditions(Pageable pageable, Principal principal, ProcessCarbonCopyEntity carbonCopy) {
    UserEntity user = super.getLoginUser(principal);
    UserVo userVo = userService.findDetailsById(user.getId());
    return processCarbonCopyRepository.findMyByConditions(pageable, carbonCopy, userVo);
  }

  @Override
  @Transactional
  public void read(String id, Principal principal) {
    Validate.notBlank(id, "抄送ID不能为空");
    UserEntity user = getLoginUser(principal);
    ProcessCarbonCopyEntity carbonCopy = processCarbonCopyRepository.findDetailById(id);
    Validate.notNull(carbonCopy, "未找到抄送信息对象");
    Validate.isTrue(processAssignmentService.equals(user, carbonCopy.getReceiveAssignment().getAssignment()), "阅读失败，当前抄送任务不属于当前用户");
    if(carbonCopy.getState().equals(0)){
      carbonCopy.setState(1);
      carbonCopy.setReadTime(new Date());
      processCarbonCopyRepository.save(carbonCopy);
    }
  }

}
