package com.bizunited.platform.titan.starter.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.repository.ProcessCheckRecordRepository;
import com.bizunited.platform.titan.starter.repository.ProcessTemplateRepository;
import com.bizunited.platform.titan.starter.service.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.eclipse.jdt.internal.compiler.problem.AbortCompilation;
import org.flowable.bpmn.model.BpmnModel;
import org.flowable.bpmn.model.Process;
import org.flowable.bpmn.model.UserTask;
import org.flowable.engine.HistoryService;
import org.flowable.engine.RepositoryService;
import org.flowable.engine.RuntimeService;
import org.flowable.engine.TaskService;
import org.flowable.engine.history.HistoricProcessInstance;
import org.flowable.engine.repository.Deployment;
import org.flowable.engine.repository.ProcessDefinition;
import org.flowable.engine.runtime.ProcessInstance;
import org.flowable.task.api.Task;
import org.flowable.task.api.TaskQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.security.Principal;
import java.util.*;
import java.util.stream.Collectors;

import static com.bizunited.platform.titan.starter.common.Constants.TASK_DEFAULT_START_ASSIGNMENT_NAME;
import static java.math.BigDecimal.ROUND_HALF_DOWN;

/**
 * 流程检测服务接口实现
 * @Author: Paul Chan
 * @Date: 2019-06-17 15:05
 */
@Service("ProcessCheckRecordServiceImpl")
public class ProcessCheckRecordServiceImpl extends BaseService implements ProcessCheckRecordService {

  private final static Logger LOGGER = LoggerFactory.getLogger(ProcessCheckRecordServiceImpl.class);

  @Autowired
  private ProcessTemplateRepository processTemplateRepository;
  @Autowired
  private ProcessCheckRecordRepository processCheckRecordRepository;

  private ProcessCheckRecordService proxySelf;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private TaskService taskService;
  @Autowired
  private RuntimeService runtimeService;
  @Autowired
  private HistoryService historyService;
  @Autowired
  private RepositoryService repositoryService;
  @Autowired
  private ProcessTemplateService processTemplateService;
  @Autowired
  private ProcessInstanceService processInstanceService;
  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessTemplateNodeService processTemplateNodeService;
  @Autowired
  private ProcessCheckRecordItemService processCheckRecordItemService;

  /**
   * 获取当前类的ioc代理类
   */
  @PostConstruct
  public void setProxySelf() {
    this.proxySelf = applicationContext.getBean(ProcessCheckRecordService.class);
  }

  @Override
  @Transactional
  public ProcessCheckRecordEntity checkProcess(String templateId, String assignment, Map<String, Object> variables, Principal principal) {
    Validate.notBlank(templateId, "模版ID不能为空");
    Validate.notNull(variables, "变量对象不能为空");
    processAssignmentService.valid(assignment);
    variables.put(TASK_DEFAULT_START_ASSIGNMENT_NAME, assignment);
    UserEntity user = getLoginUser(principal);
    ProcessTemplateEntity template = processTemplateService.findById(templateId);
    Validate.notNull(template, "未找到流程模版对象");
    ProcessCheckRecordEntity record = new ProcessCheckRecordEntity();
    String processDefinitionId = template.getProcessDefinitionId();
    try {
      proxySelf.checkProcess(template, record, variables);
    } catch (Exception e){
      LOGGER.warn(e.getMessage(), e);
    }
    template.setProcessDefinitionId(processDefinitionId);
    record.setCreateUser(user);
    record.setCreateTime(new Date());
    record.setProcessTemplate(template);
    processCheckRecordRepository.save(record);
    Set<ProcessCheckRecordItemEntity> items = processCheckRecordItemService.create(record, record.getProcessCheckRecordItems());
    record.setProcessCheckRecordItems(items);
    return record;
  }

  @Override
  @Transactional(Transactional.TxType.REQUIRES_NEW)
  public void checkProcess(ProcessTemplateEntity template, ProcessCheckRecordEntity record, Map<String, Object> variables){
    BpmnModel bpmn = processTemplateService.findBpmnById(template.getId());
    Validate.notNull(bpmn, "未找到流程定义对象");
    Process process = bpmn.getProcesses().get(0);
    String processId = process.getId();
    ProcessInstance processInstance;
    record.setParams(JSONObject.toJSONString(variables));
    record.setProcessDefId(processId);
    record.setProcessTemplate(template);
    record.setTestProcessDefId(processId);
    record.setTitle(StringUtils.join(template.getProcessName(), "_检测报告"));
    try {
      ProcessDefinition processDefinition = this.deploy(bpmn, template);
      processInstance = this.startProcess(template, processDefinition.getId(), variables);
    } catch (Exception e){
      record.setState(2);
      record.setResult(String.format("模版格式错误：%s", e.getMessage()));
      return;
    }
    boolean haveException = false;
    long startTime = System.nanoTime();
    try {
      this.complete(template, processInstance, record);
    } catch (Exception e){
      haveException = true;
    }
    long endTime = System.nanoTime();
    boolean processEnd = this.isProcessEnd(processInstance.getProcessInstanceId());
    // 如果有异常，则检测失败
    if(haveException) {
      record.setState(2);
    } else if(processEnd) {
      // 流程正常结束，检测通过
      record.setState(1);
    } else {
      // 流程未正常结束，检测未完成
      record.setState(0);
    }
    record.setProcessInstanceId(processInstance.getId());
    this.initRecordResult(process, record, (endTime - startTime) / 1000000L);
    throw new AbortCompilation();
  }

  @Override
  public List<ProcessCheckRecordEntity> findByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) return null;
    return processCheckRecordRepository.findByTemplateId(templateId);
  }

  @Override
  public ProcessCheckRecordEntity findDetailById(String id) {
    if(StringUtils.isBlank(id)) return null;
    return processCheckRecordRepository.findDetailById(id);
  }

  /**
   * 初始化检测记录结果
   * @param process
   * @param record
   * @param time 耗时（毫秒）
   */
  private void initRecordResult(Process process, ProcessCheckRecordEntity record, long time){
    StringBuilder result = new StringBuilder("本次检测共耗时<span style=\"color:green;font-weight: bolder;\">");
    result.append(time / 1000D).append("</span>秒,完成度<span style=\"color:green;font-weight: bolder;\">");
    int[] completeness = this.calculateCompleteness(process, record);
    result.append(completeness[0]).append("%</span>，成功<span style=\"color:green;font-weight: bolder;\">");
    result.append(completeness[1]).append("</span>项，失败<span style=\"color:red;font-weight: bolder;\">");
    int failed = completeness[2] - completeness[1];
    if(record.getState() == 1){
      failed = 0;
    }
    result.append(failed).append("</span>项");
    record.setResult(result.toString());
  }


  /**
   * 计算成功率，以数组方式返回，第0个是完成百分比，第1个完成任务数，第2个是任务总数
   * @param process
   * @param record
   * @return
   */
  private int[] calculateCompleteness(Process process, ProcessCheckRecordEntity record){
    int competedTasks = 0;
    Set<String> taskDefKeys = new HashSet<>();
    int totalTasks = process.getFlowElements().stream().filter(e -> e instanceof UserTask).collect(Collectors.toList()).size();
    Set<ProcessCheckRecordItemEntity> items = record.getProcessCheckRecordItems();
    if(!CollectionUtils.isEmpty(items)){
      for (ProcessCheckRecordItemEntity item : items) {
        if(!taskDefKeys.contains(item.getNodeKey())){
          if(item.getState().equals(1)){
            competedTasks++;
          }
          taskDefKeys.add(item.getNodeKey());
        }
      }
    }
    int completeness = new BigDecimal(competedTasks).divide(new BigDecimal(totalTasks), 2, ROUND_HALF_DOWN).multiply(new BigDecimal(100)).intValue();
    if(record.getState() == 1){
      completeness = 100;
    }
    return new int[]{completeness, competedTasks, totalTasks};
  }

  /**
   * 审批流程
   * @param processInstance
   * @param record
   */
  private void complete(ProcessTemplateEntity template, ProcessInstance processInstance, ProcessCheckRecordEntity record){
    if(this.isProcessEnd(processInstance.getProcessInstanceId())) return;
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    if(CollectionUtils.isEmpty(tasks)) return;
    for (Task task : tasks) {
      TaskQuery taskQuery = taskService.createTaskQuery();
      task = taskQuery.taskId(task.getId()).singleResult();
      if(task == null) {
        continue;
      }
      StringBuilder result = new StringBuilder("<p>开始审批流程节点：");
      result.append(task.getName()).append("</p>");
      result.append("<p>节点审批人：").append(task.getAssignee()).append("</p>");
      ProcessTemplateNodeEntity node = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(template.getId(), task.getTaskDefinitionKey());
      if(node == null){
        result.append("<p>未找到流程节点的配置信息</p>");
        this.addItem(record, initRecordItem(task, result.toString(), 2));
        throw new IllegalArgumentException("未找到流程节点的配置信息");
      }
      if(StringUtils.isBlank(task.getAssignee())){
        if(!node.getNullSkip()){
          result.append("<p>流程节点未找到审批人</p>");
          this.addItem(record, initRecordItem(task, result.toString(), 2));
          throw new IllegalArgumentException("流程节点未找到审批人");
        } else {
          result.append("<p>审批人为空，根据节点配置自动跳过</p>");
        }
      } else {
        ProcessAssignmentEntity assignment;
        try {
          assignment = processAssignmentService.findAssignment(task.getAssignee());
        } catch (Exception e){
          result.append("<p>流程节点审批人错误：").append(e.getMessage()).append("</p>");
          this.addItem(record, initRecordItem(task, result.toString(), 2));
          throw new IllegalArgumentException(String.format("流程节点审批人错误：%s", e.getMessage()));
        }
        if(assignment == null){
          result.append("<p>流程节点审批人错误,未找到对应的用户或岗位：").append(task.getAssignee()).append("</p>");
          this.addItem(record, initRecordItem(task, result.toString(), 2));
          throw new IllegalArgumentException(String.format("流程节点审批人错误,未找到对应的用户或岗位：%s", task.getAssignee()));
        }
      }
      try {
        taskService.complete(task.getId());
      } catch (Exception e){
        result.append("<p>流程节点审批失败，失败原因：").append(e.getMessage()).append("</p>");
        this.addItem(record, initRecordItem(task, result.toString(), 2));
        throw e;
      }
      this.addItem(record, initRecordItem(task, result.toString(), 1));
    }
    this.complete(template, processInstance, record);
  }

  /**
   * 添加明细
   * @param record
   * @param item
   */
  private void addItem(ProcessCheckRecordEntity record, ProcessCheckRecordItemEntity item){
    Set<ProcessCheckRecordItemEntity> items = record.getProcessCheckRecordItems();
    if(items == null){
      items = new LinkedHashSet<>();
    }
    items.add(item);
    record.setProcessCheckRecordItems(items);
  }

  /**
   * 初始化明细
   * @param task
   * @return
   */
  private ProcessCheckRecordItemEntity initRecordItem(Task task, String result, Integer state){
    ProcessCheckRecordItemEntity item = new ProcessCheckRecordItemEntity();
    item.setCheckType("PASS");
    item.setCreateTime(new Date());
    item.setNodeKey(task.getTaskDefinitionKey());
    item.setNodeName(task.getName());
    item.setResult(result);
    item.setState(state);
    item.setTaskId(task.getId());
    return item;
  }

  /**
   * 判断流程是否已经结束
   * @param processInstanceId
   * @return
   */
  private boolean isProcessEnd(String processInstanceId){
    HistoricProcessInstance historicProcessInstance = historyService.createHistoricProcessInstanceQuery()
        .processInstanceId(processInstanceId)
        .finished()
        .singleResult();
    if(historicProcessInstance != null){
      return true;
    }
    return false;
  }

  /**
   * 部署流程
   * @param bpmn
   * @param template
   * @return
   */
  private ProcessDefinition deploy(BpmnModel bpmn, ProcessTemplateEntity template){
    Process process = bpmn.getProcesses().get(0);
    String deployName = process.getId() + ".bpmn";
    Deployment deployment = repositoryService.createDeployment()
        .addBpmnModel(deployName, bpmn)
        .name(process.getId())
        .key(process.getId())
        .deploy();
    ProcessDefinition processDefinition = repositoryService.createProcessDefinitionQuery().deploymentId(deployment.getId()).singleResult();
    template.setProcessDefinitionId(processDefinition.getId());
    processTemplateRepository.save(template);
    return processDefinition;
  }

  /**
   * 开启流程
   *
   * @param template
   * @param processDefinitionId
   * @param variables
   * @return
   */
  private ProcessInstance startProcess(ProcessTemplateEntity template, String processDefinitionId, Map<String, Object> variables){
    variables = processInstanceService.initInstanceMultiVariable(template, variables);
    ProcessInstance processInstance = runtimeService.startProcessInstanceById(processDefinitionId, variables);
    return processInstance;
  }

}
