package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.bizunited.platform.core.service.image.FileUpdateService;
import com.bizunited.platform.titan.entity.ProcessInstanceAttachmentEntity;
import com.bizunited.platform.titan.entity.ProcessInstanceEntity;
import com.bizunited.platform.titan.starter.repository.ProcessInstanceAttachmentRepository;
import com.bizunited.platform.titan.starter.service.ProcessInstanceAttachmentService;
import com.bizunited.platform.titan.starter.service.ProcessInstanceService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

/**
 * 流程实例附件管理的服务接口实现
 * @Author: Paul Chan
 * @Date: 2019-06-10 14:11
 */
@Service("ProcessInstanceAttachmentServiceImpl")
public class ProcessInstanceAttachmentServiceImpl extends BaseService implements ProcessInstanceAttachmentService {

  @Autowired
  private ProcessInstanceAttachmentRepository processInstanceAttachmentRepository;

  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private NebulaFileService nebulaFileService;
  @Autowired
  private ProcessInstanceService processInstanceService;

  /**
   * 保存前验证
   * @param attachments
   */
  private void createValidation(Set<ProcessInstanceAttachmentEntity> attachments){
    Validate.notEmpty(attachments, "保存对象不能为空");
    Set<String> fileNames = new HashSet<>();
    for (ProcessInstanceAttachmentEntity attachment : attachments) {
      Validate.isTrue(attachment.getId() == null, "创建对象不能有ID");
      Validate.notNull(attachment.getFileName(), "源文件名不能为空");
      Validate.notNull(attachment.getFileRename(), "文件名不能为空");
      Validate.notNull(attachment.getFileCode(), "文件路径不能为空");
      Validate.isTrue(!fileNames.contains(attachment.getFileName()), "重复的文件：%s", attachment.getFileName());
      fileNames.add(attachment.getFileName());
    }
  }

  @Override
  @Transactional
  public Set<ProcessInstanceAttachmentEntity> create(String processInstanceId, Set<ProcessInstanceAttachmentEntity> attachments, Principal principal) {
    Validate.notBlank(processInstanceId, "流程实例ID不能为空");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(processInstanceId);
    Validate.notNull(processInstance, "未找到流程实例对象");
    UserEntity user = getLoginUser(principal);
    if(attachments == null) attachments = new HashSet<>();
    createValidation(attachments);
    Set<ProcessInstanceAttachmentEntity> oldAttachments = processInstanceAttachmentRepository.findDetailsByProcessInstanceId(processInstanceId);
    String[] fileRenames = new String[attachments.size()];
    int index = 0;
    for (ProcessInstanceAttachmentEntity attachment : attachments) {
      fileRenames[index] = StringUtils.join(attachment.getFileCode(), attachment.getFileRename());
      attachment.setCreateUser(user);
      attachment.setProcessInstance(processInstance);
      attachment.setCreateTime(new Date());
      processInstanceAttachmentRepository.save(attachment);
      oldAttachments.add(attachment);
      index++;
    }
    fileUpdateService.updateEffective(fileRenames);
    return oldAttachments;
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notBlank(id, "ID不能为空");
    ProcessInstanceAttachmentEntity attachment = processInstanceAttachmentRepository.findDetailById(id);
    if(attachment != null){
      processInstanceAttachmentRepository.deleteById(id);
      processInstanceAttachmentRepository.flush();
      nebulaFileService.deleteFile(attachment.getFileCode(), attachment.getFileName(), attachment.getFileRename());
    }
  }

  @Override
  public Set<ProcessInstanceAttachmentEntity> findDetailsByProcessInstanceId(String processInstanceId) {
    return processInstanceAttachmentRepository.findDetailsByProcessInstanceId(processInstanceId);
  }

}
