package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.common.enums.TaskOperateBtn;
import com.bizunited.platform.titan.starter.repository.ProcessInstanceOperateRecordRepository;
import com.bizunited.platform.titan.starter.service.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.flowable.task.api.Task;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 流程操作记录服务的接口实现
 * @Author: Paul Chan
 * @Date: 2019-05-23 14:47
 */
@Service("ProcessInstanceOperateRecordServiceImpl")
public class ProcessInstanceOperateRecordServiceImpl extends BaseService implements ProcessInstanceOperateRecordService {

  private final static Logger LOGGER = LoggerFactory.getLogger(ProcessInstanceOperateRecordServiceImpl.class);

  @Autowired
  private ProcessInstanceOperateRecordRepository processInstanceOperateRecordRepository;

  @Autowired
  private ProcessTaskService processTaskService;
  @Autowired
  private ProcessInstanceService processInstanceService;
  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessInstanceMsgService processInstanceMsgService;
  @Autowired
  private ProcessTemplateNodeService processTemplateNodeService;

  @Override
  @Transactional
  public ProcessInstanceOperateRecordEntity create(Task task, ProcessInstanceEntity processInstance, UserEntity user, TaskOperateBtn operateBtn, String content) {
    ProcessTemplateNodeEntity node = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    Validate.notNull(node, "数据错误，未找到流程节点");
    ProcessInstanceOperateRecordEntity record = new ProcessInstanceOperateRecordEntity();
    record.setCreateTime(new Date());
    record.setContent(content);
    record.setOperation(operateBtn.getBtn());
    record.setOperationDesc(operateBtn.getDesc());
    record.setProcessInstance(processInstance);
    record.setOperator(user);
    record.setProcessTemplateNode(node);
    record.setTaskId(task.getId());
    processInstanceOperateRecordRepository.save(record);
    if(StringUtils.isNotBlank(task.getAssignee())) {
      // 任务审批人不为空时保存操作记录的同时保存一份消息
      processInstanceMsgService.create(user, processInstance, record.getContent(), operateBtn);
    }
    return record;
  }

  @Override
  public List<ProcessInstanceOperateRecordEntity> findByProcessInstanceIdAndLatestSubmitTimeAndBtns(String processInstanceId, Date latestSubmitTime, String... btns) {
    return processInstanceOperateRecordRepository.findByProcessInstanceIdAndLatestSubmitTimeAndBtns(processInstanceId, latestSubmitTime, btns);
  }

  @Override
  public Set<ProcessInstanceOperateRecordEntity> findDetailsByProcessInstanceId(String processInstanceId) {
    Set<ProcessInstanceOperateRecordEntity> records = processInstanceOperateRecordRepository.findDetailsByProcessInstanceId(processInstanceId);
    return addCurrentTaskRecord(processInstanceId, records);
  }

  @Override
  public Set<ProcessInstanceOperateRecordEntity> findByProcessInstanceIdAndProcessNodeId(String processInstanceId, String processNodeId) {
    return processInstanceOperateRecordRepository.findByProcessInstanceIdAndProcessNodeId(processInstanceId, processNodeId);
  }

  @Override
  public Long countByTaskIdAndUserIdAndBtns(String taskId, String userId, String... btns) {
    return processInstanceOperateRecordRepository.countByTaskIdAndUserIdAndBtns(taskId, userId, btns);
  }

  @Override
  public Set<ProcessInstanceOperateRecordEntity> findByTaskIdAndBtns(String taskId, String... btns) {
    return processInstanceOperateRecordRepository.findByTaskIdAndBtns(taskId, btns);
  }

  /**
   * 添加流程实例当前的任务信息到操作记录，状态为：0=未执行
   * @param processInstanceId
   * @param records
   * @return
   */
  private Set<ProcessInstanceOperateRecordEntity> addCurrentTaskRecord(String processInstanceId, Set<ProcessInstanceOperateRecordEntity> records){
    if(records == null) records = new HashSet<>();
    List<Task> tasks = processTaskService.findCurrentTasks(processInstanceId);
    if(!CollectionUtils.isEmpty(tasks)){
      ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(processInstanceId);
      if(processInstance == null) return records;
      ProcessTemplateEntity processTemplate = processInstance.getProcessTemplate();
      if(processTemplate == null) return records;
      for (Task task : tasks) {
        ProcessInstanceOperateRecordEntity record = new ProcessInstanceOperateRecordEntity();
        ProcessTemplateNodeEntity node = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processTemplate.getId(), task.getTaskDefinitionKey());
        if(StringUtils.isNotBlank(task.getAssignee())){
          ProcessAssignmentEntity processAssignment = null;
          try {
            processAssignment = processAssignmentService.findAssignment(task.getAssignee());
          } catch (Exception e){
            LOGGER.warn(e.getMessage(), e);
          }
          if(processAssignment != null){
            UserEntity user = new UserEntity();
            user.setUserName(processAssignment.getAssignmentName());
            record.setOperator(user);
          }
        }
        record.setTaskId(task.getId());
        record.setProcessTemplateNode(node);
        record.setState(0);
        record.setCreateTime(new Date());
        records.add(record);
      }
    }
    return records;
  }

}
