package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.common.Constants;
import com.bizunited.platform.titan.starter.common.enums.MultiType;
import com.bizunited.platform.titan.starter.common.enums.ProcessInstanceState;
import com.bizunited.platform.titan.starter.common.enums.ProcessTemplateState;
import com.bizunited.platform.titan.starter.repository.ProcessInstanceRepository;
import com.bizunited.platform.titan.starter.service.*;
import com.bizunited.platform.titan.vo.ProcessImageNodeVo;
import com.bizunited.platform.titan.vo.ProcessImageVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.flowable.bpmn.model.Process;
import org.flowable.bpmn.model.*;
import org.flowable.engine.HistoryService;
import org.flowable.engine.RepositoryService;
import org.flowable.engine.RuntimeService;
import org.flowable.engine.TaskService;
import org.flowable.engine.history.HistoricActivityInstance;
import org.flowable.engine.runtime.Execution;
import org.flowable.engine.runtime.ExecutionQuery;
import org.flowable.engine.runtime.ProcessInstance;
import org.flowable.task.api.Task;
import org.flowable.task.api.TaskInfo;
import org.flowable.task.api.history.HistoricTaskInstance;
import org.flowable.task.api.history.HistoricTaskInstanceQuery;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.*;
import java.util.stream.Collectors;

import static com.bizunited.platform.titan.starter.common.Constants.*;
import static org.flowable.bpmn.constants.BpmnXMLConstants.ELEMENT_SEQUENCE_FLOW;

/**
 * 流程实例服务的接口实现
 * @Author: Paul Chan
 * @Date: 2019-05-22 19:17
 */
@Service("ProcessInstanceServiceImpl")
public class ProcessInstanceServiceImpl extends BaseService implements ProcessInstanceService {

  @Autowired
  private HistoryService historyService;
  @Autowired
  private RepositoryService repositoryService;
  @Autowired
  private ProcessInstanceRepository processInstanceRepository;

  @Autowired
  private TaskService taskService;
  @Autowired
  private RuntimeService runtimeService;
  @Autowired(required = false)
  private InstanceService instanceService;
  @Autowired
  private PlatformContext platformContext;
  @Autowired
  private ProcessTaskService processTaskService;
  @Autowired
  private ProcessVariableService processVariableService;
  @Autowired
  private ProcessTemplateService processTemplateService;
  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessTemplateNodeService processTemplateNodeService;
  @Autowired
  private ProcessInstanceAttachmentService processInstanceAttachmentService;
  @Autowired
  private ProcessInstanceOperateRecordService processInstanceOperateRecordService;

  /**
   * 根据流程模版ID开启流程实例
   * 1、获取登录用户
   * 2、验证模版ID参数
   * 3、根据模版ID获取模版信息并验证模版是否存在以及模版是否已经发布
   * 4、判断如参变量是否为空，如果是则初始化
   * 5、将当前登录人加入到流程变量中
   * 6、获取流程模版的预置变量并加入到流程变量中
   * 7、初始化会签节点的会签人员变量
   * 8、调用流程引擎runtimeService#startProcessInstanceById启动流程实例
   * 9、初始化流程实例对象，获取当前流程实例的当前任务放到流程实例对象中
   * 10、保存流程实例对象
   * 11、获取流程实例的当前任务人并设置到流程实例中
   * 12、根据流程表单类型处理流程的发起节点：
   *    1、如果是无表单的流程，则直接调processTaskService#handleSubmitForm方法自动跳过发起节点
   *    2、如果是表单模版的流程：
   *        1、判断当前系统是否开启表单引擎模块
   *        2、初始化表单实例对象
   *        3、调用instanceService#create方法创建表单实例
   *        3、更新流程实例的表单实例信息
   *    3、如果是自定义表单的流程：
   *        暂时未做处理
   * @param processTemplateId
   * @param variables
   * @param principal
   * @return
   */
  @Override
  @Transactional
  public ProcessInstanceEntity startProcess(String processTemplateId, Map<String, Object> variables, Principal principal) {
    UserEntity user = super.getLoginUser(principal);
    Validate.notBlank(processTemplateId, "模版ID不能为空");
    ProcessTemplateEntity processTemplate = processTemplateService.findById(processTemplateId);
    Validate.notNull(processTemplate, "流程模版对象不存在");
    Validate.isTrue(processTemplate.getProcessState().equals(ProcessTemplateState.PUBLISHED.getState()), "流程模版未发布，不能发起流程");
    if(variables == null) variables = new HashMap<>(16);
    variables.put(Constants.TASK_DEFAULT_START_ASSIGNMENT_NAME, StringUtils.join(USERNAME_PREFIX, principal.getName()));
    Map<String, Object> templateVariables = processVariableService.getVariablesByTargetId(processTemplateId);
    variables.putAll(templateVariables);
    this.initInstanceMultiVariable(processTemplate, variables);
    ProcessInstance processInstance = runtimeService.startProcessInstanceById(processTemplate.getProcessDefinitionId(), variables);
    Date nowTime = new Date();
    ProcessInstanceEntity instanceEntity = new ProcessInstanceEntity();
    instanceEntity.setApplicantUser(user);
    instanceEntity.setCreateTime(nowTime);
    instanceEntity.setModifyTime(nowTime);
    instanceEntity.setLatestSubmitTime(processInstance.getStartTime());
    instanceEntity.setProcessInstanceId(processInstance.getProcessInstanceId());
    instanceEntity.setProcessState(ProcessInstanceState.DRAFT.getState());
    instanceEntity.setProcessTemplate(processTemplate);
    Set<ProcessTemplateNodeEntity> nodes = processTaskService.findCurrentNodes(instanceEntity);
    instanceEntity.setCurrentNodes(nodes);
    instanceEntity.setLatestOperateTime(nowTime);
    processInstanceRepository.save(instanceEntity);
    Set<ProcessAssignmentEntity> currentAssignments = processTaskService.findCurrentAssignments(instanceEntity);
    instanceEntity.setCurrentAssignments(currentAssignments);
    switch (processTemplate.getFormType()) {
      case 0:
        // 无表单则系统自动提交第一个任务
        processTaskService.handleSubmitForm(processInstance.getProcessInstanceId(), principal, null);
        break;
      case 1:
        // 调用表单模版服务创建表单实例
        Validate.isTrue(platformContext.isEnableKuiper(), "应用未启用表单引擎模块，不能创建表单模版的流程");
        InstanceEntity instance = new InstanceEntity();
        TemplateEntity template = new TemplateEntity();
        template.setId(processTemplate.getFormTemplateId());
        instance.setTemplate(template);
        String taskCode = StringUtils.join(instanceEntity.getProcessInstanceId(), NODE_ID_DEFAULT_START);
        instance = instanceService.create(instance, taskCode, principal);
        instanceEntity.setFormInstanceId(instance.getId());
        instanceEntity.setFormNo(instance.getId());
        instanceEntity.setFormCreateActivityId(instance.getActivities().get(0).getId());
        processInstanceRepository.save(instanceEntity);
        break;
      case 2:
        // TODO: 2019-05-22 调用表单地址创建表单实例或者其他方案
        break;
      default:
        break;
    }
    return instanceEntity;
  }


  /**
   * 初始化会签节点的会签人员变量
   * @param processTemplate
   * @param variables
   * @return
   */
  @Override
  public Map<String, Object> initInstanceMultiVariable(ProcessTemplateEntity processTemplate, Map<String, Object> variables){
    Validate.notNull(processTemplate, "流程模版不能为空");
    if(variables == null) variables = new HashMap<>(16);
    Set<ProcessTemplateNodeEntity> nodes = processTemplateNodeService.findDetailsByProcessTemplateId(processTemplate.getId());
    for (ProcessTemplateNodeEntity node : nodes) {
      ProcessTemplateNodeMultiEntity multi = node.getProcessTemplateNodeMulti();
      if(multi == null || multi.getMiType().equals(MultiType.NONE_MULTI.getType())) continue;
      String collection = StringUtils.join(node.getProcessNodeId(), TASK_MULTI_COLLECTION_SUFFIX);
      if(multi.getPresetMiAssignments()){
        List<String> assignments = multi.getAssignments().stream().map(ProcessAssignmentEntity::getAssignment).collect(Collectors.toList());
        variables.put(collection, assignments);
      } else {
        variables.put(collection, new ArrayList<>());
      }
    }
    return variables;
  }


  @Override
  @Transactional
  public void initInstanceMulti(ProcessInstanceEntity processInstance){
    Execution processExecution = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).onlyProcessInstanceExecutions().singleResult();
    Validate.notNull(processExecution, "未找到流程实例的执行实例对象");
    Map<String, Object> variables = runtimeService.getVariables(processExecution.getId());
    this.initInstanceMultiVariable(processInstance.getProcessTemplate(), variables);
    runtimeService.setVariables(processExecution.getId(), variables);
  }

  @Override
  public ProcessInstanceEntity findByProcessInstanceId(String processInstanceId) {
    return processInstanceRepository.findByProcessInstanceId(processInstanceId);
  }

  @Override
  @Transactional
  public ProcessInstanceEntity save(ProcessInstanceEntity processInstance) {
    return processInstanceRepository.save(processInstance);
  }

  @Override
  @Transactional
  public ProcessInstanceEntity update(ProcessInstanceEntity processInstance, UserEntity user, ProcessTemplateNodeEntity currentNode, Set<ProcessAssignmentEntity> nextAssignments, Set<ProcessTemplateNodeEntity> nextNodes, Integer state) {
    Date nowTime = new Date();
    processInstance.setModifyTime(nowTime);
    processInstance.setLatestOperateTime(nowTime);
    processInstance.setLatestAssignment(user);
    processInstance.setLatestNode(currentNode);
    processInstance.setCurrentAssignments(nextAssignments);
    processInstance.setCurrentNodes(nextNodes);
    processInstance.setProcessState(state);
    processInstanceRepository.save(processInstance);
    return processInstance;
  }

  @Override
  public Page<ProcessInstanceEntity> findByConditions(Pageable pageable, ProcessInstanceEntity processInstance, Principal principal, Boolean isApplicant) {
    if(isApplicant){
      UserEntity user = getLoginUser(principal);
      processInstance.setApplicantUser(user);
    }
    Page<ProcessInstanceEntity> page = processInstanceRepository.findByConditions(pageable, processInstance);
    if(!CollectionUtils.isEmpty(page.getContent())){
      for (ProcessInstanceEntity processInstanceEntity : page.getContent()) {
        Set<ProcessAssignmentEntity> currentAssignments = processAssignmentService.findByResourceId(processInstanceEntity.getId());
        processInstanceEntity.setCurrentAssignments(currentAssignments);
      }
    }
    return page;
  }

  @Override
  public ProcessInstanceEntity findDetailsByProcessInstanceId(String processInstanceId) {
    if(StringUtils.isBlank(processInstanceId)) return null;
    ProcessTemplateEntity processTemplate = processTemplateService.findDetailsByProcessInstanceId(processInstanceId);
    ProcessInstanceEntity processInstance = processInstanceRepository.findDetailsByProcessInstanceId(processInstanceId);
    if(processInstance == null) return null;
    Set<ProcessTemplateNodeEntity> currentNodes = processInstanceRepository.findCurrentNodesByProcessInstanceId(processInstanceId);
    Set<ProcessInstanceOperateRecordEntity> records = processInstanceOperateRecordService.findDetailsByProcessInstanceId(processInstanceId);
    Set<ProcessInstanceAttachmentEntity> attachments =  processInstanceAttachmentService.findDetailsByProcessInstanceId(processInstanceId);
    Set<ProcessAssignmentEntity> currentAssignments = processAssignmentService.findByResourceId(processInstance.getId());
    processInstance.setCurrentAssignments(currentAssignments);
    processInstance.setCurrentNodes(currentNodes);
    processInstance.setRecords(records);
    processInstance.setAttachments(attachments);
    processInstance.setProcessTemplate(processTemplate);
    return processInstance;
  }

  @Override
  public ProcessImageVo findImageInfoByProcessInstanceId(String processInstanceId) {
    ProcessTemplateEntity processTemplate = processTemplateService.findDetailsByProcessInstanceId(processInstanceId);
    Validate.notNull(processTemplate, "未找到流程模版对象");
    ProcessInstanceEntity processInstance = this.findDetailsByProcessInstanceId(processInstanceId);
    Validate.notNull(processInstance, "未找到流程实例对象");
    BpmnModel bpmnModel = repositoryService.getBpmnModel(processTemplate.getProcessDefinitionId());
    Validate.notNull(bpmnModel, "未找到流程设计图资源对象");
    // 高亮的线条
    List<String> highLightedFlows = new ArrayList<>();
    // 高亮的历史活动
    List<String> highLightedHisActivities = new ArrayList<>();
    // 查询在最后一次提交后完成的活动实例
    List<HistoricActivityInstance> list = historyService.createHistoricActivityInstanceQuery()
        .processInstanceId(processInstanceId)
        .finishedAfter(processInstance.getLatestSubmitTime())
        .finished()
        .list();
    List<Task> tasks = processTaskService.findCurrentTasks(processInstanceId);
    // 高亮的当前待办任务活动
    List<String> highLightedActivities = tasks.stream().map(TaskInfo::getTaskDefinitionKey).collect(Collectors.toList());
    list.forEach(activityInstance -> {
      if (activityInstance.getActivityType().equals(ELEMENT_SEQUENCE_FLOW)) {
        highLightedFlows.add(activityInstance.getActivityId());
      } else {
        highLightedHisActivities.add(activityInstance.getActivityId());
      }
    });
    // 获取流程开始活动和出线的活动ID
    List<String> activities = processTemplateService.getStartEventAndOutLine(bpmnModel.getProcesses().get(0));
    highLightedFlows.addAll(activities);
    highLightedHisActivities.addAll(activities);
    ProcessImageVo processImage = new ProcessImageVo();
    processImage.setProcessXml(processTemplate.getProcessXml());
    processImage.setHighLightedFlows(StringUtils.join(highLightedFlows, ","));
    processImage.setHighLightedActivities(StringUtils.join(highLightedActivities, ","));
    processImage.setHighLightedHisActivities(StringUtils.join(highLightedHisActivities, ","));
    Set<ProcessImageNodeVo> imageNodes = getProcessImageNodes(bpmnModel, processInstance);
    processImage.setImageNodes(imageNodes);
    return processImage;
  }

  @Override
  public Map<String, Object> setProcessVariables(String processInstanceId, Map<String, Object> variables) {
    Validate.notBlank(processInstanceId, "流程实例ID不能为空！");
    ProcessInstanceEntity processInstance = processInstanceRepository.findByProcessInstanceId(processInstanceId);
    Validate.notNull(processInstance, "未找到流程实例对象");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.COMPLETED.getState()), "流程已完成，不能设置变量");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.STOPPED.getState()), "流程已终止，不能设置变量");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.CANCELED.getState()), "流程已撤销，不能设置变量");
    Execution execution = runtimeService.createExecutionQuery().processInstanceId(processInstanceId).onlyProcessInstanceExecutions().singleResult();
    Validate.notNull(execution, "未找到流程实例执行对象");
    Map<String, Object> oldVariables = runtimeService.getVariables(execution.getId());
    if(CollectionUtils.isEmpty(variables)) return oldVariables;
    oldVariables.putAll(variables);
    runtimeService.setVariables(execution.getId(), oldVariables);
    return oldVariables;
  }

  @Override
  public Map<String, Object> findVariablesByProcessInstanceId(String processInstanceId) {
    if(StringUtils.isBlank(processInstanceId)) {
      return null;
    }
    ProcessInstanceEntity processInstance = processInstanceRepository.findByProcessInstanceId(processInstanceId);
    if(processInstance == null) {
      return null;
    }
    ExecutionQuery executionQuery = runtimeService.createExecutionQuery();
    Execution execution = executionQuery.processInstanceId(processInstanceId).onlyProcessInstanceExecutions().singleResult();
    if(execution == null) {
      return null;
    }
    return runtimeService.getVariables(execution.getId());
  }

  /**
   * 获取流程图片的节点信息，节点信息包括节点的坐标和高宽，节点自最后一次提交后的节点的操作记录，以及节点最后的审批人信息
   * @param bpmnModel
   * @param processInstance
   * @return
   */
  private Set<ProcessImageNodeVo> getProcessImageNodes(BpmnModel bpmnModel, ProcessInstanceEntity processInstance){
    Set<ProcessImageNodeVo> nodes = new HashSet<>();
    ProcessTemplateEntity processTemplate = processInstance.getProcessTemplate();
    Process process = bpmnModel.getProcesses().get(0);
    Map<String, GraphicInfo> locationMap = bpmnModel.getLocationMap();
    Set<UserTask> userTasks = getProcessUserTasks(process);
    for (UserTask userTask : userTasks) {
      ProcessImageNodeVo imageNode = new ProcessImageNodeVo();
      this.initNodeTaskInfo(imageNode, processInstance, userTask.getId());
      GraphicInfo graphicInfo = locationMap.get(userTask.getId());
      imageNode.setX(graphicInfo.getX());
      imageNode.setY(graphicInfo.getY());
      imageNode.setWidth(graphicInfo.getWidth());
      imageNode.setHeight(graphicInfo.getHeight());
      ProcessTemplateNodeEntity node = processTemplateNodeService.findDetailByProcessTemplateIdAndProcessNodeId(processTemplate.getId(), userTask.getId());
      Set<ProcessInstanceOperateRecordEntity> records = processInstanceOperateRecordService.findByProcessInstanceIdAndProcessNodeId(processInstance.getProcessInstanceId(), userTask.getId());
      imageNode.setNode(node);
      imageNode.setRecords(records);
      nodes.add(imageNode);
    }
    return nodes;
  }

  /**
   * 初始化图片节点的审批人信息和节点的创建和结束时间
   * @param imageNode
   * @param processInstance
   * @param taskDefinitionKey
   * @return
   */
  private ProcessImageNodeVo initNodeTaskInfo(ProcessImageNodeVo imageNode, ProcessInstanceEntity processInstance, String taskDefinitionKey){
    HistoricTaskInstanceQuery query = historyService.createHistoricTaskInstanceQuery()
        .processInstanceId(processInstance.getProcessInstanceId())
        .taskDefinitionKey(taskDefinitionKey)
        .orderByHistoricTaskInstanceStartTime()
        .desc();
    // 如果是发起节点，不加时间限制
    if(!taskDefinitionKey.equals(NODE_ID_DEFAULT_START)) query.taskCreatedAfter(processInstance.getLatestSubmitTime());
    List<HistoricTaskInstance> taskInstances = query.list();
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).taskDefinitionKey(taskDefinitionKey).listPage(0, 1);
    if(CollectionUtils.isEmpty(tasks) && !CollectionUtils.isEmpty(taskInstances)){
      imageNode.setState(1);
    } else {
      imageNode.setState(0);
    }
    if(CollectionUtils.isEmpty(taskInstances)) return imageNode;
    StringBuilder assignments = new StringBuilder();
    StringBuilder assignmentNames = new StringBuilder();
    Set<String> assignmentSet = new HashSet<>();
    for (int i = 0; i < taskInstances.size(); i++) {
      HistoricTaskInstance taskInstance = taskInstances.get(i);
      if(assignmentSet.contains(taskInstance.getAssignee())) continue;
      assignmentSet.add(taskInstance.getAssignee());
      if(assignments.length() > 0){
        assignments.append("|");
        assignmentNames.append("|");
      }
      assignments.append(taskInstance.getAssignee());
      ProcessAssignmentEntity assignment = processAssignmentService.findAssignment(taskInstance.getAssignee());
      if (assignment != null) {
        assignmentNames.append(assignment.getAssignmentName());
      } else {
        assignmentNames.append("无审批人");
      }
    }
    imageNode.setAssignments(assignments.toString());
    imageNode.setAssignmentNames(assignmentNames.toString());
    imageNode.setStartTime(taskInstances.get(taskInstances.size() - 1).getCreateTime());
    imageNode.setEndTime(taskInstances.get(0).getEndTime());
    return imageNode;
  }

  /**
   * 获取流程模版的所有用户任务节点
   * @param process
   * @return
   */
  private Set<UserTask> getProcessUserTasks(Process process){
    Set<UserTask> userTasks = new HashSet<>();
    Collection<FlowElement> flowElements = process.getFlowElements();
    for (FlowElement flowElement : flowElements) {
      if(flowElement instanceof UserTask){
        UserTask userTask = (UserTask) flowElement;
        userTasks.add(userTask);
      }
    }
    return userTasks;
  }

}
