package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.titan.entity.ProcessListenerEntity;
import com.bizunited.platform.titan.starter.repository.ProcessListenerRepository;
import com.bizunited.platform.titan.starter.service.ProcessListenerService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-15 20:12
 * @Description: 流程监听器服务接口的实现
 */
@Service("ProcessListenerServiceImpl")
public class ProcessListenerServiceImpl implements ProcessListenerService {

  @Autowired
  private ProcessListenerRepository processListenerRepository;

  @Override
  public Page<ProcessListenerEntity> findByConditions(ProcessListenerEntity processListener, Pageable pageable) {
    return processListenerRepository.findByConditions(processListener, pageable);
  }

  /**
   * 创建前数据验证
   * @param processListener
   */
  private void createValidation(ProcessListenerEntity processListener){
    Validate.notNull(processListener, "创建对象不能为空");
    Validate.isTrue(processListener.getId() == null, "创建对象不能有ID");
    Validate.notBlank(processListener.getListenerEvent(), "事件不能为空");
    Validate.notNull(processListener.getState(), "状态不能为空");
    Validate.inclusiveBetween(0, 1, processListener.getState(), "状态非法");
    Validate.notBlank(processListener.getName(), "监听器名称不能为空");
    Validate.notBlank(processListener.getListenerClass(), "监听器类不能为空");
    Validate.notNull(processListener.getType(), "监听器类型不能为空");
    Validate.inclusiveBetween(1, 3, processListener.getType(), "监听器类型不合法");
    Validate.notNull(processListener.getExecuteMode(), "执行方式不能为空");
    Validate.inclusiveBetween(0, 1, processListener.getExecuteMode(), "执行方式不合法");
    Long count = processListenerRepository.countByName(processListener.getName());
    Validate.isTrue(count == null || count == 0, "监听器名称重复:%s", processListener.getName());
  }

  @Override
  @Transactional
  public ProcessListenerEntity create(ProcessListenerEntity processListener) {
    this.createValidation(processListener);
    processListener.setCreateTime(new Date());
    processListener.setModifyTime(new Date());
    return processListenerRepository.save(processListener);
  }

  /**
   * 更新前验证
   * @param processListener
   */
  private void updateValidation(ProcessListenerEntity processListener){
    Validate.notNull(processListener, "更新对象不能为空");
    Validate.notBlank(processListener.getId(), "更新对象ID不能为空");
    ProcessListenerEntity listenerEntity = processListenerRepository.findById(processListener.getId()).orElse(null);
    Validate.notNull(listenerEntity, "更新对象不存在");
    Validate.notBlank(processListener.getListenerEvent(), "事件不能为空");
    Validate.notBlank(processListener.getName(), "监听器名称不能为空");
    Validate.notBlank(processListener.getListenerClass(), "监听器类不能为空");
    Validate.notNull(processListener.getType(), "监听器类型不能为空");
    Validate.inclusiveBetween(1, 3, processListener.getType(), "监听器类型不合法");
    Validate.notNull(processListener.getExecuteMode(), "执行方式不能为空");
    Validate.inclusiveBetween(0, 1, processListener.getExecuteMode(), "执行方式不合法");
    Long count = processListenerRepository.countByNameExclusiveId(processListener.getName(), processListener.getId());
    Validate.isTrue(count == null || count == 0, "监听器名称重复:%s", processListener.getName());
  }

  @Override
  @Transactional
  public ProcessListenerEntity update(ProcessListenerEntity processListener) {
    this.updateValidation(processListener);
    ProcessListenerEntity oldListener = processListenerRepository.findById(processListener.getId()).orElse(null);
    processListener.setModifyTime(new Date());
    processListener.setCreateTime(oldListener.getCreateTime());
    return processListenerRepository.save(processListener);
  }

  @Override
  @Transactional
  public void handleState(String id, Integer state) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notNull(state, "状态不能为空");
    Validate.inclusiveBetween(0, 1, state, "状态非法");
    ProcessListenerEntity listener = processListenerRepository.findById(id).orElse(null);
    Validate.notNull(listener, "更新对象不存在");
    if(!listener.getState().equals(state)){
      listener.setState(state);
      processListenerRepository.save(listener);
    }
  }

  @Override
  public ProcessListenerEntity findById(String id) {
    if(StringUtils.isBlank(id)) return null;
    return processListenerRepository.findById(id).orElse(null);
  }

}
