package com.bizunited.platform.titan.starter.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.service.invoke.InvokeProxy;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.serviceable.model.InputParamsModel;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.command.ProphesyMultiCompletionBeforeRejectCommand;
import com.bizunited.platform.titan.starter.common.Constants;
import com.bizunited.platform.titan.starter.common.enums.ProcessFormType;
import com.bizunited.platform.titan.starter.common.enums.ProcessInstanceState;
import com.bizunited.platform.titan.starter.common.enums.TaskOperateBtn;
import com.bizunited.platform.titan.starter.repository.internal.ExecutionRepositoryCustom;
import com.bizunited.platform.titan.starter.repository.internal.HistoricActivityInstanceRepositoryCustom;
import com.bizunited.platform.titan.starter.repository.internal.ProcessTaskRepositoryCustom;
import com.bizunited.platform.titan.starter.service.*;
import com.bizunited.platform.titan.starter.service.invoke.model.ProcessInputParamsModel;
import com.bizunited.platform.titan.vo.TaskVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.flowable.bpmn.model.Process;
import org.flowable.bpmn.model.*;
import org.flowable.engine.HistoryService;
import org.flowable.engine.RepositoryService;
import org.flowable.engine.RuntimeService;
import org.flowable.engine.TaskService;
import org.flowable.engine.history.HistoricActivityInstance;
import org.flowable.engine.history.HistoricActivityInstanceQuery;
import org.flowable.engine.history.HistoricProcessInstance;
import org.flowable.engine.impl.RuntimeServiceImpl;
import org.flowable.engine.impl.persistence.entity.ExecutionEntityImpl;
import org.flowable.engine.runtime.Execution;
import org.flowable.engine.runtime.ExecutionQuery;
import org.flowable.task.api.Task;
import org.flowable.task.api.TaskInfo;
import org.flowable.task.api.history.HistoricTaskInstance;
import org.flowable.task.api.history.HistoricTaskInstanceQuery;
import org.flowable.task.service.impl.persistence.entity.TaskEntityImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.*;
import java.util.stream.Collectors;

import static com.bizunited.platform.titan.starter.common.Constants.*;
import static com.bizunited.platform.titan.starter.common.enums.TaskOperateBtn.*;
import static org.flowable.bpmn.constants.BpmnXMLConstants.ELEMENT_TASK_USER;

/**
 * 流程任务服务接口的实现
 * @Author: Paul Chan
 * @Date: 2019-05-23 10:08
 */
@Service("ProcessTaskServiceImpl")
public class ProcessTaskServiceImpl extends BaseService implements ProcessTaskService {

  private final static Logger LOGGER = LoggerFactory.getLogger(ProcessTaskServiceImpl.class);

  @Autowired
  private ExecutionRepositoryCustom executionRepositoryCustom;
  @Autowired
  private ProcessTaskRepositoryCustom processTaskRepositoryCustom;
  @Autowired
  private HistoricActivityInstanceRepositoryCustom historicActivityInstanceRepositoryCustom;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private TaskService taskService;
  @Autowired
  private HistoryService historyService;
  @Autowired
  private RuntimeService runtimeService;
  @Autowired
  private PlatformContext platformContext;
  @Autowired(required = false)
  private TemplateService templateService;
  @Autowired
  private RepositoryService repositoryService;
  @Autowired
  private ProcessInstanceService processInstanceService;
  @Autowired
  private ProcessTemplateService processTemplateService;
  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessCarbonCopyService processCarbonCopyService;
  @Autowired
  private ProcessTemplateNodeService processTemplateNodeService;
  @Autowired
  private ProcessInstanceOperateRecordService processInstanceOperateRecordService;

  /**
   * 处理任务
   * @param model
   * @return
   * @throws InvokeProxyException
   */
  @Override
  @Transactional(rollbackOn = Exception.class)
  public Object handleTask(ProcessInputParamsModel model) throws InvokeProxyException {
    /**
     * 1、验证参数
     * 2、获取流程实例并验证对象是否存在
     * 3、获取流程模版，并验证对象是否存在
     * 4、定义返回结果对象
     * 5、定义参数责任链参数对象
     * 6、将流程参数添加到参数中
     * 7、如果是表单实例的流程，并且操作是正向流转
     *   1、验证表单引擎是否开启
     *   2、获取表单模版对象，并验证对象是否存在
     *   3、如果是静态表单
     *     1、将表单数据添加到参数中
     *     2、获取静态表单保存责任链
     *     3、执行责任链
     *   4、如果是动态表单
     *     1、定义数据保存方式，create或update
     *     2、获取表单参数对象
     *     3、设置责任链参数值
     *     4、获取动态表单数据保存的责任链
     *     5、执行责任链
     * 8、如果非表单实例
     *   1、获取处理流程任务的责任链
     *   2、执行责任链
     */
    Validate.notNull(model, "如参对象不能为空");
    Validate.notBlank(model.getBtn(), "btn参数不能为空");
    Validate.notBlank(model.getProcessInstanceId(), "流程实例ID不能为空");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(model.getProcessInstanceId());
    Validate.notNull(processInstance, "未找到流程实例对象");
    ProcessTemplateEntity processTemplate = processInstance.getProcessTemplate();
    Validate.notNull(processTemplate, "未找到流程模版对象");
    Object result = null;
    Map<String, Object> params = new HashMap<>(8);
    params.put(ProcessInputParamsModel.class.getName(), model);
    if(processTemplate.getFormType().equals(ProcessFormType.FORM_TEMPLATE.getType()) && !this.isRecoveryOperate(model.getBtn())){
      Validate.isTrue(platformContext.isEnableKuiper(), "系统未启用表单引擎，请联系管理员");
      TemplateEntity formTemplate = templateService.findById(processTemplate.getFormTemplateId());
      Validate.notNull(formTemplate, "未找到表单模版对象");
      if("static".equals(formTemplate.getType())){
        // 保存静态表单模版数据
        params.put(InputParamsModel.class.getName(), model.getInputParamsModel());
        // 在表单引擎启用和流程有表单的时候，走需要更新表单数据的责任链
        InvokeProxy invokeProxy = (InvokeProxy) applicationContext.getBean("ProcessOperateAndWriteFormServicableProxy");
        result = invokeProxy.doHandle(params);
      } else if("dynamic".equals(formTemplate.getType())){
        // 保存动态表单模版数据
        String opType = "update";
        if(processInstance.getProcessState().equals(ProcessInstanceState.DRAFT.getState())){
          opType = "create";
        }
        InputParamsModel inputParamsModel = model.getInputParamsModel();
        params.put("opType", opType);
        params.put("instanceActivityId", inputParamsModel.getInstanceActivityId());
        JSONObject formData = inputParamsModel.getFormData();
        String formInstanceId = formData.getString("formInstanceId");
        params.put("formInstanceId", formInstanceId);
        params.put("formData", formData);
        InvokeProxy invokeProxy = (InvokeProxy) applicationContext.getBean("ProcessOperateAndDynamicFormWriteProxy");
        result = invokeProxy.doHandle(params);
      }
    } else {
      InvokeProxy invokeProxy = (InvokeProxy) applicationContext.getBean("ProcessOperateServicableProxy");
      result = invokeProxy.doHandle(params);
    }
    return result;
  }

  /**
   * 是否是回退的操作
   * @param btnCode
   * @return
   */
  private boolean isRecoveryOperate(String btnCode){
    TaskOperateBtn btn = Constants.TASK_OPERATION_BTN_MAP.get(btnCode);
    Validate.notNull(btn, "不支持的操作：%s", btnCode);
    switch (btn){
      case BTN_002:
        return true;
      case BTN_003:
        return true;
      case BTN_004:
        return true;
      case BTN_012:
        return true;
      default:
        return false;
    }
  }

  /**
   * 处理提交表单操作
   * 1、验证参数
   * 2、调用流程引擎taskService#createTaskQuery方法查询当前任务
   * 3、验证任务是否存在
   * 4、验证当前节点是否是发起节点，当前任务是否属于当前登录人
   * 5、获取任务处理前流程实例的所有任务对象，用于处理自动跳过任务时的排除任务
   * 6、完成当前任务
   * 7、获取当前任务处理后流程实例的任务人
   * 8、获取当前任务处理后的任务节点信息
   * 9、获取当前任务的流程节点信息
   * 10、更新流程实例最后提交时间
   * 11、更新流程实例信息
   * 12、保存当前任务的操作记录
   * 13、处理自动跳过的任务节点
   * @param processInstanceId
   * @param principal
   * @param variables
   */
  @Override
  @Transactional
  public void handleSubmitForm(String processInstanceId, Principal principal, Map<String, Object> variables) {
    Validate.notBlank(processInstanceId, "processInstanceId不能为空");
    UserEntity user = super.getLoginUser(principal);
    Task task = taskService.createTaskQuery().processInstanceId(processInstanceId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(Constants.NODE_ID_DEFAULT_START.equals(task.getTaskDefinitionKey()), "当前任务节点不是发起节点");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    taskService.complete(task.getId(), variables);
    // 保存流程实例下一审批节点的信息
    Set<ProcessAssignmentEntity> nextAssignments = this.findCurrentAssignments(processInstance);
    Set<ProcessTemplateNodeEntity> nextNodes = this.findCurrentNodes(processInstance);
    ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    processInstance.setLatestSubmitTime(task.getCreateTime());
    processInstanceService.update(processInstance, user, currentNode, nextAssignments, nextNodes, ProcessInstanceState.APPROVING.getState());
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, BTN_011, "提交申请");
    this.autoSkipTasks(tasks, processInstance, user);
  }

  /**
   * 处理审批通过任务
   * 1、验证入参
   * 2、获取当前登录用户
   * 3、根据任务ID获取当前任务对象
   * 4、验证任务是否存在，是否是发起节点，是否属于当前登录人
   * 5、获取当前任务对应的流程实例对象，并验证对象是否存在
   * 6、获取当前当前任务的用户任务定义对象，并验证对象是否存在
   * 7、获取当前任务处理前当前流程实例的所有任务，用于处理自动跳过任务时的排除任务
   * 8、初始化审批信息
   * 9、调用taskService#setAssignee设置当前任务的审批人，主要是更新任务历史记录表的审批人字段
   * 10、完成当前任务
   * 11、处理当前任务的串行会签任务，如果当前任务是串行会签的话
   * 12、处理当前任务的平行会签任务，如果当前任务是平行会签的话
   * 13、获取当前任务处理后流程实例的任务人
   * 14、获取当前任务处理后的任务节点信息
   * 15、获取当前任务的流程节点信息
   * 16、判断当前流程实例状态：
   *   1、如果当前流程实例已结束，则将状态更新到完成状态
   * 17、更新流程实例信息
   * 18、保存当前任务的操作记录
   * 19、处理自动跳过的任务节点
   * @param taskId
   * @param principal
   * @param content
   * @param variables
   */
  @Override
  @Transactional
  public void handleComplete(String taskId, Principal principal, String content, Map<String, Object> variables) {
    Validate.notBlank(taskId, "taskId不能为空");
    UserEntity user = super.getLoginUser(principal);
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(!Constants.NODE_ID_DEFAULT_START.equals(task.getTaskDefinitionKey()), "发起节点不能做审批操作");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    UserTask userTask = (UserTask) processTemplateService.findFlowElement(processInstance.getProcessTemplate().getProcessDefinitionId(), task.getTaskDefinitionKey());
    Validate.notNull(userTask, "未找到用户任务节点定义【%s】", task.getTaskDefinitionKey());
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    if(StringUtils.isBlank(content)) content = BTN_001.getDesc();
    // 这样做是将审批人更新到历史数据表中
    taskService.setAssignee(task.getId(), task.getAssignee());
    taskService.complete(task.getId(), variables);
    // 处理串行会签
    this.handleSequentialMultiAfterComplete(processInstance, task, userTask);
    // 处理平行会签
    this.handleMultiAssignmentsAfterComplete(processInstance, task, userTask);
    // 保存流程实例下一审批节点的信息
    Set<ProcessAssignmentEntity> nextAssignments = this.findCurrentAssignments(processInstance);
    Set<ProcessTemplateNodeEntity> nextNodes = this.findCurrentNodes(processInstance);
    ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    int state = processInstance.getProcessState();
    if(isProcessEnd(processInstance.getProcessInstanceId())){
      // 如果流程在审批后就结束了，则更新流程实例的状态
      state = ProcessInstanceState.COMPLETED.getState();
    }
    processInstanceService.update(processInstance, user, currentNode, nextAssignments, nextNodes, state);
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, BTN_001, content);
    this.autoSkipTasks(tasks, processInstance, user);
  }

  /**
   * 处理审批通过后串行会签, 设置任务节点的审批人
   * 1、获取当前任务的下一个任务实例
   * 2、如果不存在下一个任务实例对象，则直接返回
   * 3、获取当前任务对象任务定义对象，并验证对象是否存在
   * 4、获取当前任务节点的会签信息
   * 5、如果当前任务节点不存在会签配置，则直接返回
   * 6、获取当前会签任务的会签类型
   * 7、如果当前会签任务不是串行会签，则直接返回
   * 8、调用runtimeService#createExecutionQuery方法获取当前流程实例的执行实例对象
   * 9、初始化会签人员变量的key
   * 10、获取当前任务节点可编辑的会签人员列表
   * 11、获取当前任务执行实例的会签计数器
   * 12、如果当前任务节点没有预置会签人员，会签计数器需要减一
   * 13、通过会签计数器获取到当前任务的任务人
   * 14、将当前任务分配给当前会签任务人
   * @param processInstance
   * @param task
   * @param userTask
   */
  private void handleSequentialMultiAfterComplete(ProcessInstanceEntity processInstance, Task task, UserTask userTask){
    // 获取节点会签信息
    MultiInstanceLoopCharacteristics loopCharacteristics = userTask.getLoopCharacteristics();
    if(loopCharacteristics == null) return;
    // 是否是串行会签
    Boolean isSequential = loopCharacteristics.isSequential();
    if(!isSequential) return;
    ProcessTemplateNodeEntity node = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    Validate.notNull(node, "未找到节点【%s】的配置信息", task.getTaskDefinitionKey());
    Task nextTask = taskService.createTaskQuery().processInstanceId(task.getProcessInstanceId())
        .taskDefinitionKey(task.getTaskDefinitionKey()).singleResult();
    if(nextTask == null) return;
    // 获取该节点的预置会签人员
    Execution processExecution = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).onlyProcessInstanceExecutions().singleResult();
    String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COLLECTION_SUFFIX);
    // 获取预置的会签人员和新增的会签人员
    List<String> miAssignments = this.getEditableMultiAssignments(processExecution.getId(), collection, false);
    int loopCounter = (int) runtimeService.getVariable(nextTask.getExecutionId(), LOOP_COUNTER);
    // 如果没有预置会签人员，节点的实例数量会比加签会签人员多一个，所以下标需要减一
    if(!node.getProcessTemplateNodeMulti().getPresetMiAssignments()) loopCounter--;
    String assignment = miAssignments.get(loopCounter);
    taskService.setAssignee(nextTask.getId(), assignment);
  }

  /**
   * 处理审批通过后的会签人员，将已处理任务的审批人加入到上下文中
   * 1、获取当前任务节点的会签配置信息
   * 2、如果当前任务节点没有会签配置信息，则直接返回
   * 3、初始化当前任务已完成任务审批人的key
   * 4、获取当前任务可编辑的已完成会签任务的会签人员列表
   * 5、将当前审批人加入到已完成会签任务的会签人员列表中，并保存
   * @param processInstance
   * @param task
   * @param userTask
   */
  private void handleMultiAssignmentsAfterComplete(ProcessInstanceEntity processInstance, Task task, UserTask userTask){
    MultiInstanceLoopCharacteristics loopCharacteristics = userTask.getLoopCharacteristics();
    if(loopCharacteristics == null) return;
    Execution execution = runtimeService.createExecutionQuery().executionId(task.getExecutionId()).singleResult();
    if(execution == null) return;
    String parentExecutionId = execution.getParentId();
    // 将当前处理人加到已处理人员列表中
    String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COMPLETED_ASSIGNMENTS_SUFFIX);
    List<String> completeAssignments = this.getEditableMultiAssignments(parentExecutionId, collection, true);
    completeAssignments.add(task.getAssignee());
    runtimeService.setVariableLocal(parentExecutionId, collection, completeAssignments);
  }


  /**
   * 自动跳过当前任务，只跳过当前审批后产生的任务，根据配置有为空跳过和重复跳过
   * 1、将排除的任务节点放置到set集合中
   * 2、获取当前流程实例的所有待办任务
   * 3、如果当前流程实例的待办用户不存在，则直接返回
   * 4、初始化haveSkipTask变量为false
   * 5、遍历当前流程实例的所有待办任务：
   *   1、如果当前任务在排除任务中，则跳过当前任务处理
   *   2、调用autoSkipTask方法处理当前任务：
   *     1、如果当前任务被自动跳过，haveSkipTask设置为true，将当前任务设置最后一个跳过任务
   * 6、如果haveSkipTask为true，说明有自动跳过的任务：
   *   1、获取流程实例当前的所有审批人
   *   2、获取当前实例当前所有的待办任务节点
   *   3、获取当前自动跳过的最后一个任务的节点信息
   *   4、获取当前流程实例的状态
   *   5、更新流程实例信息
   *   6、递归处理需要自动跳过的任务
   * @param processInstance
   */
  private void autoSkipTasks(List<Task> excludeTasks, ProcessInstanceEntity processInstance, UserEntity user){
    Set<String> excludeTaskSet = excludeTasks.stream().map(Task::getTaskDefinitionKey).collect(Collectors.toSet());
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    if(CollectionUtils.isEmpty(tasks)) return;
    boolean haveSkipTask = false;
    Task latestSkipTask = null;
    for (Task task : tasks) {
      if(excludeTaskSet.contains(task.getTaskDefinitionKey())) continue;
      if(this.autoSkipTask(processInstance, task, user)) {
        haveSkipTask = true;
        latestSkipTask = task;
      }
    }
    if(haveSkipTask) {
      // 保存流程实例下一审批节点的信息
      Set<ProcessAssignmentEntity> nextAssignments = this.findCurrentAssignments(processInstance);
      Set<ProcessTemplateNodeEntity> nextNodes = this.findCurrentNodes(processInstance);
      ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), latestSkipTask.getTaskDefinitionKey());
      int state = processInstance.getProcessState();
      if(isProcessEnd(processInstance.getProcessInstanceId())){
        // 如果流程在审批后就结束了，则更新流程实例的状态
        state = ProcessInstanceState.COMPLETED.getState();
      }
      processInstanceService.update(processInstance, user, currentNode, nextAssignments, nextNodes, state);
      this.autoSkipTasks(excludeTasks, processInstance, user);
    }
  }

  /**
   * 自动跳过当前任务，根据配置有为空跳过和重复跳过,如果要跳过，则返回true
   * 1、获取当前任务的节点配置信息
   * 2、定义变量skip为false
   * 3、初始化审批意见
   * 4、初始化操作人为当前用户，当当前任务没有审批人时，则作为当前任务的处理人
   * 5、如果当前任务人为空并且当前任务节点的配置为自动跳过：
   *   1、skip设置为true
   *   2、完成当前任务
   *   3、初始化为空跳过审批意见
   * 6、如果当前任务人不为空，并且当前任务节点的配置为重复跳过：
   *   1、获取当前任务人的所有历史任务记录
   *   2、排除掉非正常审批的任务
   *   3、如果存在历史审批记录
   *     1、取第一条历史数据的审批通过记录
   *     2、如果存在历史审批记录：
   *       1、skip设置为true
   *       2、将历史审批记录的操作人赋值给当前任务操作人对象
   *       3、更新流程任务实例历史记录表的任务人字段
   *       4、完成当前任务
   *       5、初始化重复跳过审批意见
   * 7、如果skip为true：
   *   保存当前任务的操作记录信息
   * 8、返回skip
   * @param processInstance
   * @param task
   */
  private boolean autoSkipTask(ProcessInstanceEntity processInstance, Task task, UserEntity user){
    ProcessTemplateNodeEntity node = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    boolean skip = false;
    String content = "自动跳过";
    UserEntity operator = user;
    if(StringUtils.isBlank(task.getAssignee()) && node.getNullSkip()){
      // 为空跳过
      skip = true;
      taskService.complete(task.getId());
      content = "审批人为空自动跳过";
    }
    if(StringUtils.isNotBlank(task.getAssignee()) && node.getRepeatSkip()){
      // 重复跳过
      List<HistoricTaskInstance> taskInstances = historyService.createHistoricTaskInstanceQuery()
          .processInstanceId(processInstance.getProcessInstanceId())
          .taskCompletedAfter(processInstance.getLatestSubmitTime())
          .taskAssignee(task.getAssignee())
          .orderByTaskCreateTime().desc().list();
      // 筛选掉非正常审批的任务
      taskInstances = taskInstances.stream().filter(t -> StringUtils.isBlank(t.getDeleteReason())).collect(Collectors.toList());
      if(!CollectionUtils.isEmpty(taskInstances)) {
        // 取第一条历史数据的操作记录
        Set<ProcessInstanceOperateRecordEntity> records = processInstanceOperateRecordService.findByTaskIdAndBtns(taskInstances.get(0).getId(), BTN_001.getBtn());
        if(!CollectionUtils.isEmpty(records)){
          skip = true;
          operator = records.iterator().next().getOperator();
          // 这样做是将审批人更新到历史数据表中
          taskService.setAssignee(task.getId(), task.getAssignee());
          taskService.complete(task.getId());
          content = "审批人重复审批自动跳过";
        }
      }
    }
    if(skip){
      // 保存操作记录
      processInstanceOperateRecordService.create(task, processInstance, operator, BTN_001, content);
    }
    return skip;
  }

  /**
   * 处理驳回操作
   * 1、验证入参
   * 2、获取登录用户
   * 3、根据任务ID获取当前任务，并验证任务是否存在
   * 4、验证当前任务是否是发起节点，是否属于当前登录人
   * 5、获取当前任务的流程实例对象，并验证对象是否存在
   * 6、获取当前任务的流程定义对象，并验证对象是否存在
   * 7、获取当前任务的会签配置
   * 8、定义变量isMultiTask和isSequential，记录当前任务是否是会签和是否是串行会签
   * 9、定义prophesyCompletion变量接收会签预言的结果
   * 10、根据会签配置设置isMultiTask和isSequential的值
   * 11、调用ProphesyMultiCompletionBeforeRejectCommand预言当前驳回后，会签是否还可以继续
   * 12、如果当前任务是平行会签，并且预言成功：
   *   1、获取当前任务的执行实例对象
   *   2、获取当前任务的执行实例对象的父级ID
   *   3、获取当前任务的已驳回的会签人数，并验证初始化
   *   4、将当前任务的已驳回的会签人数增加一，并保存
   *   5、获取当前任务可编辑的已处理的会签人员列表
   *   6、将当前审批人添加到已处理的会签人员列表，并保存
   *   7、将当前任务的执行实例ID设置成null，这样做是因为不这样做，直接删除任务会报错
   *   8、删除当前任务，删除当前任务的历史记录
   *   9、将当前任务的执行实例的活动状态设置成false
   *   10、删除当前任务的历史活动记录
   * 13、如果是普通驳回：
   *   1、获取当前流程实例的所有待办任务，并将所有任务的活动ID放置到临时列表中
   *   2、查找在目标节点的历史活动实例后创建的活动实例，用于删除
   *   3、初始化当前流程实例的所有任务节点的会签人员
   *   4、删除平行网关实例信息，保证流程再次流转到已走过网关时不会自动跳过
   *   5、调用runtimeService#changeState方法将当前任务跳回到发起节点
   *   6、获取当前任务驳回后当前流程实例的节点任务信息，并更新到当前流程实例中
   *   7、设置当前流程实例的最后提交时间是当前时间，在这里设置是保证发起用户在没有提交表单时，也能有提交时间
   *   8、更新当前流程实例
   * 14、保存当前任务的操作记录信息
   * @param taskId
   * @param principal
   * @param content
   * @param variables
   */
  @Override
  @Transactional
  public void handleReject(String taskId, Principal principal, String content, Map<String, Object> variables) {
    // 发起节点不能执行该操作
    Validate.notBlank(taskId, "taskId不能为空");
    Validate.notBlank(content, "审批内容不能为空");
    UserEntity user = super.getLoginUser(principal);
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(!Constants.NODE_ID_DEFAULT_START.equals(task.getTaskDefinitionKey()), "发起节点不能做驳回操作");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    // 获取节点的会签配置信息
    UserTask userTask = (UserTask) processTemplateService.findFlowElement(processInstance.getProcessTemplate().getProcessDefinitionId(), task.getTaskDefinitionKey());
    Validate.notNull(userTask, "未找到用户任务节点定义【%s】", task.getTaskDefinitionKey());
    MultiInstanceLoopCharacteristics loopCharacteristics = userTask.getLoopCharacteristics();
    // 是否是会签任务
    Boolean isMultiTask = false;
    // 是否是串行会签
    Boolean isSequential = false;
    // 预言通过结果,串行会签不需要预言，如果为false，则会签结束，如果为true，则会签继续
    Boolean prophesyCompletion = false;
    if(loopCharacteristics != null) {
      isMultiTask = true;
      // 是否是串行会签
      isSequential = loopCharacteristics.isSequential();
      // 如果是平行会签，则预言在本次驳回后，剩余的会签人员是否还能完成该任务
      if(!isSequential) prophesyCompletion = ((RuntimeServiceImpl) runtimeService).getCommandExecutor().execute(new ProphesyMultiCompletionBeforeRejectCommand(runtimeService, task, userTask, 1));
    }
    if(isMultiTask && !isSequential && prophesyCompletion){
      // 继续平行会签处理
      // 更新驳回总数
      Execution execution = runtimeService.createExecutionQuery().executionId(task.getExecutionId()).singleResult();
      String executionId = task.getExecutionId();
      String parentExecutionId = execution.getParentId();
      Integer nrOfRejectedInstances = (Integer) runtimeService.getVariableLocal(parentExecutionId, NR_OF_REJECTED_INSTANCES);
      if(nrOfRejectedInstances == null) nrOfRejectedInstances = 0;
      runtimeService.setVariableLocal(parentExecutionId, NR_OF_REJECTED_INSTANCES, ++nrOfRejectedInstances);
      // 将当前处理人加到已处理人员列表中
      String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COMPLETED_ASSIGNMENTS_SUFFIX);
      List<String> completeAssignments = this.getEditableMultiAssignments(parentExecutionId, collection, true);
      completeAssignments.add(task.getAssignee());
      runtimeService.setVariableLocal(parentExecutionId, collection, completeAssignments);
      // 必须要把task的executionId设置为null才能删除task
      TaskEntityImpl taskEntity = (TaskEntityImpl) task;
      taskEntity.setExecutionId(null);
      taskService.saveTask(taskEntity);
      taskService.deleteTask(taskId, "会签驳回");
      historyService.deleteHistoricTaskInstance(taskId);
      executionRepositoryCustom.updateIsActiveById(executionId, 0);
      historicActivityInstanceRepositoryCustom.deleteByTaskId(taskId);
      // 保存操作记录
      processInstanceOperateRecordService.create(task, processInstance, user, TaskOperateBtn.BTN_002, content);
    } else {
      // 处理驳回到发起节点
      // 驳回的同时可能存在平行任务，需要将平行任务同时驳回
      List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
      List<String> activityIds = tasks.stream().map(TaskInfo::getTaskDefinitionKey).collect(Collectors.toCollection(() -> new ArrayList<>(tasks.size())));
      // 查找在目标节点的历史活动实例后创建的活动实例
      List<HistoricActivityInstance> deleteActivities = historyService.createHistoricActivityInstanceQuery()
          .processInstanceId(processInstance.getProcessInstanceId())
          .startedAfter(task.getCreateTime())
          .list();
      processInstanceService.initInstanceMulti(processInstance);
      runtimeService.createChangeActivityStateBuilder()
          .processInstanceId(processInstance.getProcessInstanceId())
          .processVariables(variables)
          .moveActivityIdsToSingleActivityId(activityIds, Constants.NODE_ID_DEFAULT_START)
          .changeState();
      // 删除在当前任务后面创建的activity实例
      historicActivityInstanceRepositoryCustom.delete(deleteActivities);
      // 删除运行时网关事件
      this.deleteParallelGetawayExecutions(processInstance);
      // 保存操作记录
      processInstanceOperateRecordService.create(task, processInstance, user, TaskOperateBtn.BTN_002, content);
      // 保存流程实例下一审批节点的信息
      Set<ProcessAssignmentEntity> nextAssignments = this.findCurrentAssignments(processInstance);
      Set<ProcessTemplateNodeEntity> nextNodes = this.findCurrentNodes(processInstance);
      ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
      taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).taskDefinitionKey(NODE_ID_DEFAULT_START).singleResult();
      processInstance.setLatestSubmitTime(new Date());
      processInstanceService.update(processInstance, user, currentNode, nextAssignments, nextNodes, ProcessInstanceState.REJECTED.getState());
    }
  }

  /**
   * 删除运行时网关事件, 这样做的目的是重置平行网关的已到达数量
   * 1、获取当前流程实例的所有执行实例
   * 2、获取当前流程实例的流程定义模版
   * 3、遍历所有执行实例：
   *   1、如果当前执行实例没有活动ID，则跳过本次遍历
   *   2、获取当前执行实例的流程节点
   *   3、如果当前节点是平行网关：
   *     1、删除当前执行实例
   * @param processInstance
   */
  private void deleteParallelGetawayExecutions(ProcessInstanceEntity processInstance){
    List<Execution> executions = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    if(CollectionUtils.isEmpty(executions)) return;
    BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessTemplate().getProcessDefinitionId());
    Validate.notNull(bpmnModel, "未找到流程定义对象");
    Process process = bpmnModel.getProcesses().get(0);
    for (Execution execution : executions) {
      if(StringUtils.isBlank(execution.getActivityId())) continue;
      FlowElement flowElement = process.getFlowElement(execution.getActivityId());
      if(flowElement instanceof ParallelGateway){
        executionRepositoryCustom.deleteRuExecutionById(execution.getId());
      }
    }
  }

  /**
   * 处理任务转办
   * 1、验证入参
   * 2、验证目标转办人
   * 3、获取登录用户，并验证目标转办人是否是当前登录人
   * 4、获取目标转办人的代理对象
   * 5、获取当前任务对象，并验证对象是否存在，验证当前任务是否属于当前登录人
   * 6、获取当前任务的流程实例对象，并验证对象是否存在
   * 7、初始化操作记录内容
   * 8、保存操作记录
   * 9、设置当前任务执行人为目标用户
   * @param taskId
   * @param targetAssignment
   * @param principal
   * @param content
   */
  @Override
  @Transactional
  public void handleTransferAssignee(String taskId, String targetAssignment, Principal principal, String content) {
    // 转办用户不能是当前任务人
    Validate.notBlank(taskId, "taskId不能为空");
    Validate.notBlank(targetAssignment, "转办目标用户不能为空");
    processAssignmentService.valid(targetAssignment);
    UserEntity user = super.getLoginUser(principal);
    Validate.isTrue(!processAssignmentService.equals(user, targetAssignment), "任务人不能转办给自己");
    ProcessAssignmentEntity assignment = processAssignmentService.findAssignment(targetAssignment);
    Validate.notNull(assignment, "未找到转办人：%s", targetAssignment);
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    if(content == null) content = "";
    content += String.format("  用户任务转办给代理人【%s】", assignment.getAssignmentName());
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, TaskOperateBtn.BTN_007, content);
    taskService.setAssignee(taskId, targetAssignment);
  }

  /**
   * 处理回退操作
   * 1、验证入参
   * 2、验证回退目标节点是否是发起节点
   * 3、获取登录用户
   * 4、获取当前任务对象，并验证对象是否存在
   * 5、验证当前任务是否属于当前登录人，验证回退目标节点是否是当前任务节点
   * 6、获取当前任务的流程实例对象，并验证对象是否存在
   * 7、获取当前流程实例的流程定义对象，并验证对象是否存在
   * 8、验证是否可以回退到目标节点
   * 9、获取当前流程实例的所有待办任务
   * 10、将所有待办任务的活动放置到临时列表中
   * 11、初始当前流程实例的所有任务节点的会签人员
   * 12、调用runtimeService#changeState方法从当前任务跳到目标节点
   * 13、处理节点回退后的平行网关实例，主要处理回退目标节点在分支上的问题
   * 14、删除在咪表节点后的活动实例
   * 15、获取当前任务驳回后当前流程实例的节点任务信息，并更新到当前流程实例中
   * 16、保存当前任务操作记录
   * @param taskId
   * @param targetActivitiId
   * @param principal
   * @param content
   * @param variables
   */
  @Override
  @Transactional
  public void handleBackTask(String taskId, String targetActivitiId, Principal principal, String content, Map<String, Object> variables) {
    // 回退节点不能是当前节点和发起节点
    Validate.notBlank(taskId, "taskId不能为空");
    Validate.notBlank(content, "审批内容不能为空");
    Validate.notBlank(targetActivitiId, "目标节点不能为空");
    Validate.isTrue(!Constants.NODE_ID_DEFAULT_START.equals(targetActivitiId), "不能回退到发起节点");
    UserEntity user = super.getLoginUser(principal);
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    Validate.isTrue(!targetActivitiId.equals(task.getTaskDefinitionKey()), "回退节点不能是当前节点");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessTemplate().getProcessDefinitionId());
    Validate.notNull(bpmnModel, "未找到发布的流程定义对象");
    this.validateBackTaskTargetActivity(processInstance, targetActivitiId);
    // 回退的同时可能存在平行任务，需要将平行任务同时回退
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    List<String> activityIds = tasks.stream().map(TaskInfo::getTaskDefinitionKey).collect(Collectors.toCollection(() -> new ArrayList<>(tasks.size())));
    processInstanceService.initInstanceMulti(processInstance);
    runtimeService.createChangeActivityStateBuilder()
        .processInstanceId(processInstance.getProcessInstanceId())
        .processVariables(variables)
        .moveActivityIdsToSingleActivityId(activityIds, targetActivitiId)
        .changeState();
    // 将在回退目标节点的历史活动实例后创建的活动实例删除
    // 在删除前处理需要保留的平行网关
    this.handleGetaway(bpmnModel, processInstance, task.getTaskDefinitionKey(), targetActivitiId);
    this.deleteHistoricActivityInstanceAfterActivity(bpmnModel, processInstance, targetActivitiId);
    // 保存流程实例下一审批节点的信息
    Set<ProcessAssignmentEntity> nextAssignments = this.findCurrentAssignments(processInstance);
    Set<ProcessTemplateNodeEntity> nextNodes = this.findCurrentNodes(processInstance);
    ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    processInstanceService.update(processInstance, user, currentNode, nextAssignments, nextNodes, processInstance.getProcessState());
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, TaskOperateBtn.BTN_003, content);
  }

  /**
   * 处理任务回退、取回时网关节点，主要是计算历史节点中平行网关需要保留的数量，
   * 用于解决回退到平行网关分支上节点，继续往下审批会卡在网关或直接跳过的的问题
   * @param processInstance 流程实例
   * @param activityId 当前的活动ID
   * @param targetActivityId 退回目标节点的ID
   * @return
   */
  private void handleGetaway(BpmnModel bpmnModel, ProcessInstanceEntity processInstance, String activityId, String targetActivityId){
    Process process = bpmnModel.getProcesses().get(0);
    FlowElement flowElement = process.getFlowElement(targetActivityId);
    {
      // 处理需要删除的execution网关,此问题出现在平行的节点中，部分节点审核通过了，但是部分节点发生了回退
      List<Execution> executions = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
      Map<String, List<Execution>> executionMap = new HashMap<>(8);
      for (Execution execution : executions) {
        if(StringUtils.isBlank(execution.getActivityId())) continue;
        if(process.getFlowElement(execution.getActivityId()) instanceof ParallelGateway){
          List<Execution> list = executionMap.get(execution.getActivityId());
          if(list == null) list = new ArrayList<>();
          list.add(execution);
          executionMap.put(execution.getActivityId(), list);
        }
      }
      executionMap.forEach((k, v) -> {
        FlowElement current = process.getFlowElement(k);
        // 获取可以从目标节点的几条进线进入
        int lines = getCanGotoTheTargetLines(flowElement, current);
        lines--; //减一是减去当前节点到达的线
        // 删除会重新生成的平行网关
        for (Execution execution : v) {
          if(lines > 0){
            executionRepositoryCustom.deleteRuExecutionById(execution.getId());
          }
          lines--;
        }
      });
    }
    {
      // 处理需要增加的网关,主要处理回退到平行网关后，在平行网关卡住的问题
      Map<String, Integer> parallelGetaways = new HashMap<>(8);
      this.getParallelGetaways(parallelGetaways, flowElement, flowElement, activityId);
      // 遍历将需要保留的平行网关添加到execution中
      Task task = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).list().get(0);
      ExecutionEntityImpl execution = (ExecutionEntityImpl) runtimeService.createExecutionQuery().executionId(task.getExecutionId()).singleResult();
      parallelGetaways.forEach( (k, v) ->{
        if(v > 0){
          for (Integer i = 0; i < v; i++) {
            execution.setActivityId(k);
            execution.setActive(false);
            executionRepositoryCustom.insert(execution);
          }
        }
      });
    }
  }

  /**
   * 判断开始节点，可以从目标节点的几条进线进入
   * @param start
   * @param target
   * @return
   */
  private int getCanGotoTheTargetLines(FlowElement start, FlowElement target){
    int line = 0;
    List<FlowElement> elements = this.getSourceFlowElements(target);
    for (FlowElement element : elements) {
      boolean result = this.canGotoTheTarget(start, element);
      if(result) line++;
    }
    return line;
  }

  /**
   * 判断开始节点能否到达目标节点
   * @param start
   * @param target
   * @return
   */
  private boolean canGotoTheTarget(FlowElement start, FlowElement target){
    List<FlowElement> elements = this.getNextFlowElements(start);
    if(!CollectionUtils.isEmpty(elements)){
      for (FlowElement element : elements) {
        if(element.getId().equals(target.getId())){
          return true;
        }
        boolean result = canGotoTheTarget(element, target);
        if(result) return true;
      }
    }
    return false;
  }

  /**
   * 获取从开始节点到目标节点中的平行网关，同时计算平行网关需要保留的实例数量
   * @param parallelGetaways 平行网关存储器
   * @param startFlowElement 开始节点
   * @param currentFlowElement 当前节点
   * @param targetFlowId 目标节点ID
   * @return
   */
  private boolean getParallelGetaways(Map<String, Integer> parallelGetaways, FlowElement startFlowElement, FlowElement currentFlowElement, String targetFlowId){
    // 获取当前节点的下行节点
    List<FlowElement> nextFlowElements = this.getNextFlowElements(currentFlowElement);
    boolean result = false;
    if(!CollectionUtils.isEmpty(nextFlowElements)){
      // 如果有下行节点则继续，没有则返回结果
      for (FlowElement flowElement : nextFlowElements) {
        if(flowElement.getId().equals(targetFlowId)){
          // 如果节点是目标节点，则返回true
          return true;
        }
        // 获取下行节点是否能达到目标节点
        boolean canGoToTheTargetFlowId = this.getParallelGetaways(parallelGetaways, startFlowElement, flowElement, targetFlowId);
        // 如果能达到目标节点，则将结果设置成true
        if(canGoToTheTargetFlowId) result = true;
        if(flowElement instanceof ParallelGateway && canGoToTheTargetFlowId){
          // 如果当前节点是平行网关，并且可以达到目标节点，则将该节点添加到存储器中
          if(!parallelGetaways.containsKey(flowElement.getId())){
            // 如果存储器中没有当前平行网关，避免重复计算
            // 当前网关的总的进线数量
            int totalLines = this.getSourceFlowElements(flowElement).size();
            // 获取当前节点的进线有多少线可以达到开始节点
            int canBackLines = this.getCanBackLines(flowElement, startFlowElement.getId());
            // 总线数-可达开始线数=须保留网关数
            int autoCompleteLines = totalLines - canBackLines;
            parallelGetaways.put(flowElement.getId(), autoCompleteLines);
          }
        }

      }
    }
    return result;
  }

  /**
   * 获取当前节点的进线能回溯到目标节点的数量
   * @param currentFlowElement
   * @param targetFlowId
   * @return
   */
  private int getCanBackLines(FlowElement currentFlowElement, String targetFlowId){
    int lines = 0;
    // 获取节点的进线
    List<FlowElement> sourceElements = this.getSourceFlowElements(currentFlowElement);
    if(!CollectionUtils.isEmpty(sourceElements)){
      for (FlowElement sourceElement : sourceElements) {
        // 计算当前进线能否达到开始节点
        boolean result = this.canBackToTheTargetFlowId(sourceElement, targetFlowId);
        if(result) lines++;
      }
    }
    return lines;
  }

  /**
   * 计算当前节点能否回溯到目标节点
   * @param currentFlowElement
   * @param targetFlowId
   * @return
   */
  private boolean canBackToTheTargetFlowId(FlowElement currentFlowElement, String targetFlowId){
    List<FlowElement> sourceElements = this.getSourceFlowElements(currentFlowElement);
    if(!CollectionUtils.isEmpty(sourceElements)){
      for (FlowElement sourceElement : sourceElements) {
        if(sourceElement.getId().equals(targetFlowId)){
          // 如果当前节点是目标节点，返回true
          return true;
        }
        // 判断当前节点是否能回溯到目标节点
        boolean canBackToTheTargetFlowId = this.canBackToTheTargetFlowId(sourceElement, targetFlowId);
        if(canBackToTheTargetFlowId) {
          // 如果回溯节点能回溯到目标节点，返回true
          return true;
        }
      }
    }
    return false;
  }

  /**
   * 获取当前节点的下流节点,节点类型总分为普通节点和连线两种
   * @param flowElement
   * @return
   */
  private List<FlowElement> getNextFlowElements(FlowElement flowElement){
    List<FlowElement> nextFlowElements = new ArrayList<>();
    if(flowElement instanceof FlowNode){
      nextFlowElements.addAll(((FlowNode) flowElement).getOutgoingFlows());
    } else if(flowElement instanceof SequenceFlow){
      nextFlowElements.add(((SequenceFlow)flowElement).getTargetFlowElement());
    }
    return nextFlowElements;
  }

  /**
   * 获取当前节点的来源节点,节点类型总分为普通节点和连线两种
   * @param flowElement
   * @return
   */
  private List<FlowElement> getSourceFlowElements(FlowElement flowElement){
    List<FlowElement> sourceElements = new ArrayList<>();
    if(flowElement instanceof FlowNode){
      sourceElements.addAll(((FlowNode) flowElement).getIncomingFlows());
    } else if(flowElement instanceof SequenceFlow){
      sourceElements.add(((SequenceFlow)flowElement).getSourceFlowElement());
    }
    return sourceElements;
  }

  @Override
  @Transactional
  public List<ProcessCarbonCopyEntity> handleCarbonCopy(String taskId, List<String> assignments, Principal principal) {
    Validate.notBlank(taskId, "taskId不能为空");
    Validate.notEmpty(assignments, "任务任务抄送接收人不能为空");
    processAssignmentService.valid(assignments);
    UserEntity user = super.getLoginUser(principal);
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    return processCarbonCopyService.create(task, processInstance, user, assignments);
  }

  @Override
  public Set<ProcessTemplateNodeEntity> findCurrentNodes(ProcessInstanceEntity instance) {
    Set<ProcessTemplateNodeEntity> nodes = new HashSet<>();
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(instance.getProcessInstanceId()).list();
    if (!CollectionUtils.isEmpty(tasks)) {
      for (Task task : tasks) {
        ProcessTemplateNodeEntity node = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(instance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
        nodes.add(node);
      }
    }
    return nodes;
  }

  @Override
  public Set<ProcessAssignmentEntity> findCurrentAssignments(ProcessInstanceEntity instance) {
    Set<ProcessAssignmentEntity> assignments = new HashSet<>();
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(instance.getProcessInstanceId()).list();
    if (!CollectionUtils.isEmpty(tasks)) {
      for (Task task : tasks) {
        if(StringUtils.isNotBlank(task.getAssignee())){
          ProcessAssignmentEntity assignment = processAssignmentService.findAssignment(task.getAssignee());
          assignments.add(assignment);
        }
      }
    }
    return processAssignmentService.save(instance.getId(), assignments);
  }

  @Override
  public Page<TaskVo> findMyTasksByConditions(TaskVo taskVo, Principal principal, Pageable pageable) {
    UserEntity user = this.getLoginUser(principal);
    UserVo userVo = userService.findDetailsById(user.getId());
    return processTaskRepositoryCustom.findMyTasksByConditions(userVo, taskVo, pageable);
  }

  /**
   * 处理流程终止的操作
   * 1、验证入参
   * 2、获取登录用户
   * 3、获取当前任务对象，并验证对象是否存在
   * 4、验证当前任务是否是发起节点，验证当前任务是否属于当前登录人
   * 5、获取当前流程实例对象，并验证对象是否存在
   * 6、获取流程结束节点的节点ID
   * 7、获取当前流程实例的所有待办任务，并将待办任务的活动ID放置到临时列表中
   * 8、调用runtimeService#changeState方法从当前待办任务节点跳到结束节点
   * 9、获取当前任务驳回后当前流程实例的节点任务信息，并更新到当前流程实例中
   * 10、保存当前任务操作记录
   * @param taskId
   * @param principal
   * @param content
   */
  @Override
  @Transactional
  public void handleStop(String taskId, Principal principal, String content) {
    Validate.notBlank(taskId, "taskId不能为空");
    Validate.notBlank(content, "审批内容不能为空");
    UserEntity user = super.getLoginUser(principal);
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "当前任务不存在");
    Validate.isTrue(!Constants.NODE_ID_DEFAULT_START.equals(task.getTaskDefinitionKey()), "发起节点不能做终止操作，请做撤销操作");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    String endActivitiId = processTemplateService.getEndActivityId(processInstance.getProcessTemplate());
    List<Task> tasks = taskService.createTaskQuery().
        processInstanceId(processInstance.getProcessInstanceId())
        .list();
    List<String> activityIds = tasks.stream().map(TaskInfo::getTaskDefinitionKey).collect(Collectors.toCollection(() -> new ArrayList<>(tasks.size())));
    runtimeService.createChangeActivityStateBuilder()
        .processInstanceId(processInstance.getProcessInstanceId())
        .moveActivityIdsToSingleActivityId(activityIds, endActivitiId)
        .changeState();
    // 保存流程实例下一审批节点的信息
    ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), task.getTaskDefinitionKey());
    processInstanceService.update(processInstance, user, currentNode, null, null, ProcessInstanceState.STOPPED.getState());
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, BTN_013, content);
  }

  @Override
  public List<Task> findCurrentTasks(String processInstanceId) {
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstanceId).list();
    return tasks;
  }

  /**
   * 处理撤销操作
   * 1、验证入参
   * 2、获取登录用户
   * 3、获取当前流程实例对象，并验证对象是否存在
   * 4、验证当前任务是否是发起节点，验证当前流程实例是否属于当前登录人
   * 5、验证当前流程实例是否可以撤销
   * 6、获取流程结束节点的节点ID
   * 7、获取当前流程实例的所有待办任务，并将待办任务的活动ID放置到临时列表中
   * 8、调用runtimeService#changeState方法从当前待办任务节点跳到结束节点
   * 9、取流程实例的最后一个发起节点的历史任务作为当前任务
   * 10、获取当前任务驳回后当前流程实例的节点任务信息，并更新到当前流程实例中
   * 11、保存当前任务操作记录
   * @param processInstanceId
   * @param principal
   * @param content
   * @param variables 流程变量
   */
  @Override
  @Transactional
  public void handleCancel(String processInstanceId, Principal principal, String content, Map<String, Object> variables) {
    Validate.notBlank(processInstanceId, "processInstanceId不能为空");
    Validate.notBlank(content, "审批内容不能为空");
    UserEntity user = super.getLoginUser(principal);
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(processInstanceId);
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    Validate.isTrue(user.getAccount().equals(processInstance.getApplicantUser().getAccount()), "当前登录人不是当前流程的发起者");
    // 验证流程是否能撤销
    this.validateTaskCancel(processInstance);
    String endActivitiId = processTemplateService.getEndActivityId(processInstance.getProcessTemplate());
    List<Task> tasks = taskService.createTaskQuery().
        processInstanceId(processInstance.getProcessInstanceId())
        .list();
    List<String> activityIds = tasks.stream().map(TaskInfo::getTaskDefinitionKey).collect(Collectors.toCollection(() -> new ArrayList<>(tasks.size())));
    runtimeService.createChangeActivityStateBuilder()
        .processInstanceId(processInstance.getProcessInstanceId())
        .processVariables(variables)
        .moveActivityIdsToSingleActivityId(activityIds, endActivitiId)
        .changeState();
    // 保存流程实例下一审批节点的信息
    ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), Constants.NODE_ID_DEFAULT_START);
    processInstanceService.update(processInstance, user, currentNode, null, null, ProcessInstanceState.CANCELED.getState());
    // 保存操作记录
    HistoricTaskInstance task = historyService.createHistoricTaskInstanceQuery().processInstanceId(processInstanceId)
        .taskDefinitionKey(Constants.NODE_ID_DEFAULT_START).taskCreatedAfter(processInstance.getLatestSubmitTime()).singleResult();
    if(task == null) {
      task = historyService.createHistoricTaskInstanceQuery().processInstanceId(processInstanceId)
          .taskDefinitionKey(Constants.NODE_ID_DEFAULT_START).taskCompletedAfter(processInstance.getLatestSubmitTime()).singleResult();
    }
    TaskEntityImpl newTask = (TaskEntityImpl) taskService.newTask(task.getId());
    newTask.setTaskDefinitionKey(task.getTaskDefinitionKey());
    processInstanceOperateRecordService.create(newTask, processInstance, user, BTN_012, content);
  }

  /**
   * 获取属于用户的活动历史记录，activityId是可选筛选条件
   * @param user
   * @param processInstance
   * @param activityId 活动ID，可选
   * @return
   */
  private List<HistoricActivityInstance> findUserHistoricActivityInstances(UserEntity user, ProcessInstanceEntity processInstance, String activityId){
    HistoricActivityInstanceQuery query = historyService.createHistoricActivityInstanceQuery()
        .finishedAfter(processInstance.getLatestSubmitTime())
        .processInstanceId(processInstance.getProcessInstanceId())
        .orderByHistoricActivityInstanceEndTime()
        .desc();
    if(StringUtils.isNotBlank(activityId)){
      query.activityId(activityId);
    }
    List<HistoricActivityInstance> activityInstances = query.list();
    if(!CollectionUtils.isEmpty(activityInstances)){
      activityInstances = activityInstances.stream().filter(a -> this.isTaskAssignment(a.getTaskId(), user)).collect(Collectors.toList());
    }
    return activityInstances;
  }

  /**
   * 判断任务是否属于当前用户
   * @param taskId
   * @param user
   * @return
   */
  private boolean isTaskAssignment(String taskId, UserEntity user){
    Long count = processInstanceOperateRecordService.countByTaskIdAndUserIdAndBtns(taskId, user.getId(), BTN_001.getBtn());
    return count != null && count.longValue() > 0;
  }

  /**
   * 处理取回任务操作
   * 1、验证入参
   * 2、获取登录用户
   * 3、获取当前流程实例对象，并验证对象是否存在
   * 4、验证当前流程实例是否还能取回
   * 5、获取当前流程实例的流程定义对象，并验证对象是否存在
   * 6、获取取回目标节点的历史任务
   * 7、验证节点是否可以取回,并且获取当前待办任务的活动ID
   * 8、获取历史记录的第一条记录
   * 9、初始化当前流程实例的任务节点的会签人员
   * 10、如果当前流程实例存在后续的待办任务：
   *   1、调用runtimeService#changeState方法从当前待办任务跳到目标任务节点
   * 11、如果当前流程实例不存在后续的待办任务，说明任务卡在了后续的网关处：
   *   1、获取目标节点后的平行网关执行实例
   *   2、取第一个平行网关执行实例
   *   3、调用runtimeService#changeState方法从当前平行网关实例跳到目标任务节点
   * 12、获取取回任务后当前流程实例的当前待办信息
   * 13、处理网关实例信息
   * 14、删除在回退目标节点的历史活动实例后创建的活动实例
   * 15、更新流程实例信息
   * 16、保存当前任务的操作记录
   * @param processInstanceId
   * @param activityId
   * @param principal
   */
  @Override
  @Transactional
  public void handleRetrieve(String processInstanceId, String activityId, Principal principal) {
    Validate.notBlank(processInstanceId, "processInstanceId不能为空");
    Validate.isTrue(!Constants.NODE_ID_DEFAULT_START.equals(activityId), "不能取回到发起节点");
    Validate.notBlank(activityId, "activityId不能为空");
    UserEntity user = getLoginUser(principal);
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(processInstanceId);
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.COMPLETED.getState()), "流程已完成，不能取回");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.STOPPED.getState()), "流程已终止，不能取回");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.CANCELED.getState()), "流程已撤销，不能取回");
    BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessTemplate().getProcessDefinitionId());
    Validate.notNull(bpmnModel, "未找到流程定义对象");
    // 获取取回目标节点的历史任务
    List<HistoricActivityInstance> activityInstances = this.findUserHistoricActivityInstances(user, processInstance, activityId);
    // 获取当前活动的任务
    List<String> currentActivities = this.validateTaskIsCanRetrieveAndFindCurrentTask(bpmnModel, processInstance, activityId, activityInstances);
    HistoricActivityInstance activityInstance = activityInstances.get(0);
    String nextActivityId;
    processInstanceService.initInstanceMulti(processInstance);
    if(!CollectionUtils.isEmpty(currentActivities)){
      nextActivityId = currentActivities.get(0);
      // 从当前任务跳到目标节点
      runtimeService.createChangeActivityStateBuilder()
          .processInstanceId(processInstanceId)
          .moveActivityIdsToSingleActivityId(currentActivities, activityId)
          .changeState();
    } else {
      // 从平行网关跳回目标节点
      List<Execution> executions = this.getNextParallelExecutions(bpmnModel, processInstance, activityId);
      Execution execution = executions.get(0);
      nextActivityId = execution.getActivityId();
      runtimeService.createChangeActivityStateBuilder()
          .processInstanceId(processInstanceId)
          .moveExecutionToActivityId(execution.getId(), activityId)
          .changeState();
    }
    // 保存流程实例下一审批节点的信息
    Set<ProcessAssignmentEntity> nextAssignments = this.findCurrentAssignments(processInstance);
    Set<ProcessTemplateNodeEntity> nextNodes = this.findCurrentNodes(processInstance);
    // 获取在取回目标任务的前一个完成任务
    List<HistoricActivityInstance> historicActivityInstances = historyService.createHistoricActivityInstanceQuery()
        .processInstanceId(processInstanceId)
        .finishedBefore(activityInstance.getStartTime())
        .activityType(ELEMENT_TASK_USER)
        .orderByHistoricActivityInstanceEndTime()
        .desc().list();
    String currentTaskKey = "";
    // 保证前一个完成的任务不是取回目标节点的任务
    for (HistoricActivityInstance historicActivityInstance : historicActivityInstances) {
      if(!historicActivityInstance.getActivityId().equals(activityId)){
        currentTaskKey = historicActivityInstance.getActivityId();
        break;
      }
    }
    this.handleGetaway(bpmnModel, processInstance, nextActivityId, activityId);
    // 删除在回退目标节点的历史活动实例后创建的活动实例
    this.deleteHistoricActivityInstanceAfterActivity(bpmnModel, processInstance, activityId);
    ProcessTemplateNodeEntity currentNode = processTemplateNodeService.findByProcessTemplateIdAndProcessNodeId(processInstance.getProcessTemplate().getId(), currentTaskKey);
    processInstanceService.update(processInstance, user, currentNode, nextAssignments, nextNodes, processInstance.getProcessState());
    // 保存操作记录
    TaskEntityImpl newTask = (TaskEntityImpl) taskService.newTask(activityInstance.getId());
    newTask.setTaskDefinitionKey(activityInstance.getActivityId());
    processInstanceOperateRecordService.create(newTask, processInstance, user, BTN_004, "取回任务");
  }


  /**
   * 删除在活动之后的活动实例
   * @param bpmnModel
   * @param processInstance
   * @param activityId
   */
  private void deleteHistoricActivityInstanceAfterActivity(BpmnModel bpmnModel, ProcessInstanceEntity processInstance, String activityId){
    Process process = bpmnModel.getProcesses().get(0);
    FlowElement flowElement = process.getFlowElement(activityId);
    this.deleteHistoricActivityInstanceAfterActivity(flowElement, processInstance);
  }

  /**
   * 删除在活动之后的活动实例
   * @param flowElement
   * @param processInstance
   */
  private void deleteHistoricActivityInstanceAfterActivity(FlowElement flowElement, ProcessInstanceEntity processInstance){
    List<FlowElement> elements = this.getNextFlowElements(flowElement);
    if(!CollectionUtils.isEmpty(elements)){
      for (FlowElement element : elements) {
        List<HistoricActivityInstance> activityInstances = historyService.createHistoricActivityInstanceQuery().processInstanceId(processInstance.getProcessInstanceId())
            .activityId(element.getId()).startedAfter(processInstance.getLatestSubmitTime()).list();
        if(!CollectionUtils.isEmpty(activityInstances)){
          if(element instanceof ParallelGateway){
            HistoricActivityInstance activityInstance = activityInstances.get(0);
            activityInstances = Collections.singletonList(activityInstance);
          }
          historicActivityInstanceRepositoryCustom.delete(activityInstances);
        }
        deleteHistoricActivityInstanceAfterActivity(element, processInstance);
      }
    }
  }

  /**
   * 验证节点是否可以取回,并且获取当前待办任务的活动ID
   */
  private List<String> validateTaskIsCanRetrieveAndFindCurrentTask(BpmnModel bpmnModel, ProcessInstanceEntity processInstance, String activityId, List<HistoricActivityInstance> activityInstances){
    Validate.notEmpty(activityInstances, "未找到历史任务节点");
    HistoricActivityInstance historicActivityInstance = activityInstances.get(0);
    Validate.isTrue(StringUtils.isBlank(historicActivityInstance.getDeleteReason()), "该节点任务非正常审批，不能取回");
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).taskDefinitionKey(activityId).list();
    Validate.isTrue(CollectionUtils.isEmpty(tasks), "当前任务节点有正在审批的任务，不能取回该任务");
    Process process = bpmnModel.getProcesses().get(0);
    FlowElement flowElement = process.getFlowElement(activityId);
    // 获取下行的用户任务
    List<UserTask> nextUserTasks = this.findNextUserTask(flowElement);
    List<String> currentTasks = new ArrayList<>();
    for (UserTask userTask : nextUserTasks) {
      List<HistoricTaskInstance> taskInstances = historyService.createHistoricTaskInstanceQuery().processInstanceId(processInstance.getProcessInstanceId())
          .taskCompletedAfter(historicActivityInstance.getEndTime()).taskDefinitionKey(userTask.getId())
          .finished().list();
      if(!CollectionUtils.isEmpty(taskInstances)){
        // 排除掉被删除的节点
        Iterator<HistoricTaskInstance> iterator = taskInstances.iterator();
        while (iterator.hasNext()){
          HistoricTaskInstance next = iterator.next();
          if(StringUtils.isNotBlank(next.getDeleteReason())) iterator.remove();
        }
      }
      Validate.isTrue(CollectionUtils.isEmpty(taskInstances), "后续任务已审批，不能取回该任务");
      List<Task> list = taskService.createTaskQuery().processInstanceId(processInstance.getProcessInstanceId()).taskDefinitionKey(userTask.getId()).list();
      if(!CollectionUtils.isEmpty(list)) {
        currentTasks.add(userTask.getId());
      }
    }
    return currentTasks;
  }

  /**
   * 获取当前活动节点的下一个平行网关
   * @param processInstance
   * @param activityId
   * @return
   */
  private List<Execution> getNextParallelExecutions(BpmnModel bpmnModel, ProcessInstanceEntity processInstance, String activityId){
    Process process = bpmnModel.getProcesses().get(0);
    FlowElement flowElement = process.getFlowElement(activityId);
    ParallelGateway parallelGateway = this.findNextParallelGateway(flowElement);
    Validate.notNull(parallelGateway, "未找到活动节点的下一平行网关");
    List<Execution> executions = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).list();
    List<Execution> parallelExecutions = executions.stream().filter(execution -> parallelGateway.getId().equals(execution.getActivityId())).collect(Collectors.toList());
    return parallelExecutions;
  }

  /**
   * 获取当前节点的下一个平行网关
   * @param flowElement
   * @return
   */
  private ParallelGateway findNextParallelGateway(FlowElement flowElement){
    List<FlowElement> elements = this.getNextFlowElements(flowElement);
    if(!CollectionUtils.isEmpty(elements)){
      for (FlowElement element : elements) {
        if(element instanceof ParallelGateway){
          return (ParallelGateway) element;
        }
        return findNextParallelGateway(element);
      }
    }
    return null;
  }

  /**
   * 获取当前节点的后续节点，只取分支线上的第一个用户节点
   * @param flowElement
   * @return
   */
  private List<UserTask> findNextUserTask(FlowElement flowElement){
    List<UserTask> userTasks = new ArrayList<>();
    List<FlowElement> elements = this.getNextFlowElements(flowElement);
    if(!CollectionUtils.isEmpty(elements)){
      for (FlowElement element : elements) {
        if(element instanceof UserTask){
          userTasks.add((UserTask) element);
        } else {
          userTasks.addAll(findNextUserTask(element));
        }
      }
    }
    return userTasks;
  }

  /**
   * 处理加签
   * 1、验证入参
   * 2、获取登录用户
   * 3、验证加签人员是否包含当前登录人
   * 4、获取当前任务对象，并验证对象是否存在，验证当前任务是否属于当前登录人
   * 5、获取当前任务的流程实例对象，并验证对象是否存在
   * 6、获取当前流程实例的任务节点定义对象，并验证对象是否存在
   * 7、获取当前任务节点的会签信息，并验证当前任务节点是否是会签节点
   * 8、获取当前任务节点的会签类型
   * 9、获取当前任务节点的会签人员、已处理任务的会签人员
   * 10、验证加签人员
   * 11、将加签人员添加到当前任务会签人员列表中，并更新到当前流程实例的上下文中
   * 12、遍历处理加签人员：
   *   1、获取当前会签人员的代理对象，并验证
   *   2、调用runtimeService#addMultiInstanceExecution方法将当前会签人员添加到当前任务节点中
   *   3、如果当前任务是平行会签：
   *     1、根据新增的执行实例ID获取到新增的任务实例
   *     2、将当前会签人更新到新增的任务实例中
   * 13、保存当前任务的操作记录
   * @param taskId
   * @param assignments
   * @param principal
   * @param content
   */
  @Override
  @Transactional
  public void handleAddMulti(String taskId, List<String> assignments, Principal principal, String content) {
    Validate.notBlank(taskId, "任务ID不能为空");
    Validate.notEmpty(assignments, "会签人员不能为空");
    content = content == null ? "" : content;
    UserEntity user = getLoginUser(principal);
    Validate.isTrue(!processAssignmentService.contains(assignments, user), "不能添加自己");
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "未找到该任务对象");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    UserTask userTask = (UserTask) processTemplateService.findFlowElement(processInstance.getProcessTemplate().getProcessDefinitionId(), task.getTaskDefinitionKey());
    Validate.notNull(userTask, "未找到用户任务节点定义【%s】", task.getTaskDefinitionKey());
    // 获取节点会签信息
    MultiInstanceLoopCharacteristics loopCharacteristics = userTask.getLoopCharacteristics();
    Validate.notNull(loopCharacteristics, "该任务节点不是会签节点，不能加签");
    StringBuilder contentSb = new StringBuilder(content).append(" ===>增加会签人员[");
    // 是否是串行会签
    Boolean isSequential = loopCharacteristics.isSequential();
    // 获取该节点的预置会签人员
    Execution processExecution = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).onlyProcessInstanceExecutions().singleResult();
    Execution taskExecution = runtimeService.createExecutionQuery().executionId(task.getExecutionId()).singleResult();
    // 新的会签人员信息用新的列表保存，从引擎获取的会签人员列表不能修改
    String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COLLECTION_SUFFIX);
    String taskAssignmentKey = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_ELEMENT_VARIABLE);
    String completedCollection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COMPLETED_ASSIGNMENTS_SUFFIX);
    // 会签人员
    List<String> miAssignments = this.getEditableMultiAssignments(processExecution.getId(), collection, false);
    List<String> completedMiAssignments = this.getEditableMultiAssignments(taskExecution.getParentId(), completedCollection, true);
    this.validateAddMultiAssignments(miAssignments, completedMiAssignments, assignments);
    // 更新最新的会签人员
    miAssignments.addAll(assignments);
    runtimeService.setVariable(task.getExecutionId(), collection, miAssignments);
    // 开始增加会签任务
    for (int i = 0; i < assignments.size(); i++) {
      String assignment = assignments.get(i);
      ProcessAssignmentEntity assignmentEntity = processAssignmentService.findAssignment(assignment);
      Validate.notNull(assignmentEntity, "未找到加签用户:%s", assignment);
      if(i > 0) contentSb.append(",");
      contentSb.append(assignmentEntity.getAssignmentName());
      Map<String, Object> variables = new HashMap<>(2);
      if(!isSequential) {
        variables.put(taskAssignmentKey, assignment);
      }
      Execution execution = runtimeService.addMultiInstanceExecution(task.getTaskDefinitionKey(), task.getProcessInstanceId(), variables);
      // 如果是平行会签，则找到加签的任务，设置任务人
      if(!isSequential){
        Task addTask = taskService.createTaskQuery().executionId(execution.getId()).singleResult();
        taskService.setAssignee(addTask.getId(), assignment);
      }
    }
    contentSb.append("]");
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, TaskOperateBtn.BTN_009, contentSb.toString());
  }

  /**
   * 处理减签操作
   * 1、验证入参
   * 2、获取当前登录用户
   * 3、验证减签人员中是否有登录人自己
   * 4、获取当前任务对象，并验证对象是否存在，验证当前任务是否属于当前登录人
   * 5、获取当前任务的流程实例对象，并验证对象是否存在
   * 6、获取当前任务的任务节点定义信息
   * 7、获取当前任务节点的会签配置，并验证当前任务节点是否是会签节点
   * 8、获取当前任务节点的会签人员列表和已处理任务的会签人员列表
   * 9、验证减签人员
   * 10、遍历处理减签人员：
   *   1、获取当前减签人员的代理对象
   *   2、将当前减签人员从当前任务会签人员中移除
   *   3、如果当前任务节点是平行会签：
   *     1、获取当前会签人员的任务实例对象
   *     2、调用runtimeService#deleteMultiInstanceExecution方法删除当前会签人员任务实例
   * 11、将处理后的会签人员更新到当前流程实例上下文中
   * 12、保存当前任务的操作记录
   * @param taskId
   * @param assignments
   * @param principal
   * @param content
   */
  @Override
  @Transactional
  public void handleSubMulti(String taskId, List<String> assignments, Principal principal, String content) {
    Validate.notBlank(taskId, "任务ID不能为空");
    Validate.notEmpty(assignments, "会签人员不能为空");
    content = content == null ? "" : content;
    UserEntity user = super.getLoginUser(principal);
    Validate.isTrue(!processAssignmentService.contains(assignments, user), "不能删除自己");
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "未找到该任务对象");
    Validate.isTrue(processAssignmentService.equals(user, task.getAssignee()), "当前任务不属于当前登录人");
    ProcessInstanceEntity processInstance = processInstanceService.findByProcessInstanceId(task.getProcessInstanceId());
    Validate.notNull(processInstance, "数据异常，未找到流程实例对象");
    UserTask userTask = (UserTask) processTemplateService.findFlowElement(processInstance.getProcessTemplate().getProcessDefinitionId(), task.getTaskDefinitionKey());
    Validate.notNull(userTask, "未找到用户任务节点定义【%s】", task.getTaskDefinitionKey());
    // 获取节点会签信息
    MultiInstanceLoopCharacteristics loopCharacteristics = userTask.getLoopCharacteristics();
    Validate.notNull(loopCharacteristics, "该任务节点不是会签节点，不能减签");
    Boolean isSequential = loopCharacteristics.isSequential();
    StringBuilder contentSb = new StringBuilder(content).append(" ===>删除会签人员[");
    // 是否是串行会签
    // 获取该节点的预置会签人员
    Execution processExecution = runtimeService.createExecutionQuery().processInstanceId(processInstance.getProcessInstanceId()).onlyProcessInstanceExecutions().singleResult();
    Execution taskExecution = runtimeService.createExecutionQuery().executionId(task.getExecutionId()).singleResult();
    // 新的会签人员信息用新的列表保存，从引擎获取的会签人员列表不能修改
    String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COLLECTION_SUFFIX);
    String completedCollection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COMPLETED_ASSIGNMENTS_SUFFIX);
    List<String> miAssignments = this.getEditableMultiAssignments(processExecution.getId(), collection, false);
    List<String> completedMiAssignments = this.getEditableMultiAssignments(taskExecution.getParentId(), completedCollection, true);
    this.validateSubMultiAssignments(miAssignments, completedMiAssignments, assignments, user, task, userTask);
    for (int i = 0; i < assignments.size(); i++) {
      String assignment = assignments.get(i);
      ProcessAssignmentEntity assignmentEntity = processAssignmentService.findAssignment(assignment);
      Validate.notNull(assignmentEntity, "未找到减签用户:%s", assignment);
      if(i > 0) contentSb.append(",");
      contentSb.append(assignmentEntity.getAssignmentName());
      miAssignments.remove(assignment);
      if(!isSequential){
        // 如果是平行会签，则需要删除会签任务
        Task multiTask = taskService.createTaskQuery()
            .processInstanceId(task.getProcessInstanceId())
            .taskDefinitionKey(task.getTaskDefinitionKey())
            .taskAssignee(assignment).singleResult();
        Validate.notNull(multiTask, "会签任务不存在：%s", assignment);
        runtimeService.deleteMultiInstanceExecution(multiTask.getExecutionId(), false);
      }
    }
    if(isSequential) {
      // 如果是串行会签，则需要将总会签数量减去减签的数量
      Integer nrOnInstance = (Integer) runtimeService.getVariable(taskExecution.getParentId(), NR_OF_INSTANCES);
      runtimeService.setVariable(taskExecution.getParentId(), NR_OF_INSTANCES, nrOnInstance - assignments.size());
    }
    runtimeService.setVariable(processExecution.getId(), collection, miAssignments);
    contentSb.append("]");
    // 保存操作记录
    processInstanceOperateRecordService.create(task, processInstance, user, TaskOperateBtn.BTN_010, contentSb.toString());
  }

  /**
   * 获取可编辑的会签人员列表
   * @param executionId
   * @param collection
   * @param local 是否是本地变量
   * @return
   */
  @SuppressWarnings("unchecked")
  private List<String> getEditableMultiAssignments(String executionId, String collection, boolean local){
    List<String> assignments = new ArrayList<>();
    List<String> list;
    if(local) {
      list = runtimeService.getVariableLocal(executionId, collection, List.class);
    } else {
      list= runtimeService.getVariable(executionId, collection, List.class);
    }
    if(!CollectionUtils.isEmpty(list)){
      list.forEach(a -> assignments.add(a));
    }
    return assignments;
  }

  /**
   * 验证减签可行性
   * @param miAssignments
   * @param assignments
   * @param completedMiAssignments
   * @param user
   * @param task
   * @param userTask
   */
  private void validateSubMultiAssignments(List<String> miAssignments, List<String> completedMiAssignments, List<String> assignments, UserEntity user, Task task, UserTask userTask){
    Set<String> assignmentSet = new HashSet<>();
    for (String assignment : assignments) {
      Validate.isTrue(!processAssignmentService.equals(user, assignment), "不能删除自己");
      Validate.isTrue(!assignmentSet.contains(assignment), "重复的删除会签人员：%s", assignment);
      Validate.isTrue(miAssignments.contains(assignment), "人员【%s】不在会签人员中", assignment);
      Validate.isTrue(!completedMiAssignments.contains(assignment), "人员【%s】已处理当前会签，不能删除", assignment);
    }
    // 预言删除会签人员后会签是否还可以继续
    boolean prophesyCompletion = ((RuntimeServiceImpl) runtimeService).getCommandExecutor().execute(new ProphesyMultiCompletionBeforeRejectCommand(runtimeService, task, userTask, assignments.size()));
    Validate.isTrue(prophesyCompletion, "此操作将导致会签无法继续，请减少删除的会签人员");
  }


  /**
   * 验证会签人员
   * @param miAssignments
   * @param assignments
   */
  private void validateAddMultiAssignments(List<String> miAssignments, List<String> completedMiAssignments, List<String> assignments){
    Set<String> assignmentSet = new HashSet<>();
    for (String assignment : assignments) {
      Validate.isTrue(!miAssignments.contains(assignment), "当前会签中已有会签人员:%s", assignment);
      Validate.isTrue(!completedMiAssignments.contains(assignment), "当前会签中已有会签人员:%s", assignment);
      Validate.isTrue(!assignmentSet.contains(assignment), "重复的会签人员:%s", assignment);
      assignmentSet.add(assignment);
    }
  }

  /**
   * 验证任务节点的撤销
   * @param processInstance
   */
  private void validateTaskCancel(ProcessInstanceEntity processInstance){
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.COMPLETED.getState()), "流程已完成，不能撤销");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.STOPPED.getState()), "流程已终止，不能撤销");
    Validate.isTrue(!processInstance.getProcessState().equals(ProcessInstanceState.CANCELED.getState()), "流程已撤销，不能再次撤销");
    List<ProcessTemplateNodeEntity> nodes = processTemplateNodeService.findByProcessTemplateIdAndCancelFlag(processInstance.getProcessTemplate().getId(), true);
    List<HistoricActivityInstance> activities = historyService.createHistoricActivityInstanceQuery()
        .startedAfter(processInstance.getLatestSubmitTime()).list();
    if(CollectionUtils.isEmpty(nodes) || CollectionUtils.isEmpty(activities)) return;
    Set<String> nodesSet = nodes.stream().map(ProcessTemplateNodeEntity::getProcessNodeId).collect(Collectors.toSet());
    Set<String> activitiesSet = activities.stream().map(HistoricActivityInstance::getActivityId).collect(Collectors.toSet());
    Set<String> intersection = Sets.intersection(nodesSet, activitiesSet);
    Validate.isTrue(CollectionUtils.isEmpty(intersection), "任务已到撤销标识的节点，不能撤销");
  }

  /**
   * 验证回退节点是否在历史任务节点中
   * @param processInstance
   * @param targetActivitiId
   */
  private void validateBackTaskTargetActivity(ProcessInstanceEntity processInstance, String targetActivitiId){
    List<ProcessInstanceOperateRecordEntity> records = processInstanceOperateRecordService.findByProcessInstanceIdAndLatestSubmitTimeAndBtns(processInstance.getProcessInstanceId(), processInstance.getLatestSubmitTime(), BTN_001.getBtn());
    if(!CollectionUtils.isEmpty(records)){
      for (ProcessInstanceOperateRecordEntity record : records) {
        if(targetActivitiId.equals(record.getProcessTemplateNode().getProcessNodeId())){
          Validate.isTrue(record.getProcessTemplateNode().getCanBack(), "目标节点不允许回退");
          return;
        }
      }
    }
    throw new IllegalArgumentException("目标节点不在历史任务中");
  }

  /**
   * 判断流程是否已经结束
   * @param processInstanceId
   * @return
   */
  private boolean isProcessEnd(String processInstanceId){
    HistoricProcessInstance historicProcessInstance = historyService.createHistoricProcessInstanceQuery()
        .processInstanceId(processInstanceId)
        .finished()
        .singleResult();
    if(historicProcessInstance != null){
      return true;
    }
    return false;
  }

  @Override
  public List<ProcessTemplateNodeEntity> findBackableNodes(String processInstanceId) {
    if(StringUtils.isBlank(processInstanceId)) return null;
    ProcessInstanceEntity processInstance = processInstanceService.findDetailsByProcessInstanceId(processInstanceId);
    if(processInstance == null) return null;
    if(processInstance.getProcessTemplate() == null) return null;
    List<ProcessTemplateNodeEntity> nodes = processTemplateNodeService.findByProcessTemplateId(processInstance.getProcessTemplate().getId());
    if(CollectionUtils.isEmpty(nodes)) return null;
    Map<String, ProcessTemplateNodeEntity> nodesMap = nodes.stream().collect(Collectors.toMap(ProcessTemplateNodeEntity::getProcessNodeId, node -> node, (a, b) -> b, () -> new HashMap<>(16)));
    List<ProcessTemplateNodeEntity> backables = new ArrayList<>();
    // 查询出在最后一次提交后的历史节点
    HistoricTaskInstanceQuery historicTaskInstanceQuery = historyService.createHistoricTaskInstanceQuery();
    List<HistoricTaskInstance> tasks = historicTaskInstanceQuery.processInstanceId(processInstanceId)
        .taskCompletedAfter(processInstance.getLatestSubmitTime())
        .orderByTaskCreateTime().asc().list();
    if(CollectionUtils.isEmpty(tasks)) return null;
    Set<String> nodeIds = new HashSet<>();
    for (HistoricTaskInstance task : tasks) {
      ProcessTemplateNodeEntity node = nodesMap.get(task.getTaskDefinitionKey());
      if(!nodeIds.add(node.getProcessNodeId())) {
        continue;
      }
      // 排除掉不能回退的节点和开始节点
      if(node != null && node.getCanBack() && !NODE_ID_DEFAULT_START.equals(task.getTaskDefinitionKey())
        && StringUtils.isBlank(task.getDeleteReason())){
        backables.add(node);
      }
    }
    return backables;
  }

  @Override
  public Page<ProcessInstanceOperateRecordEntity> findDoneByConditions(Pageable pageable, ProcessInstanceOperateRecordEntity record, Principal principal) {
    UserEntity user = getLoginUser(principal);
    return processTaskRepositoryCustom.findDoneByConditions(pageable, record, user);
  }

  @Override
  public TaskVo findTaskInfoByInstanceIdAndTaskKey(String processInstanceId, String taskKey) {
    if(StringUtils.isBlank(processInstanceId)) return null;
    if(StringUtils.isBlank(taskKey)) return null;
    ProcessTemplateEntity processTemplate = processTemplateService.findDetailsByProcessInstanceId(processInstanceId);
    if(processTemplate == null) return null;
    ProcessInstanceEntity processInstance = processInstanceService.findDetailsByProcessInstanceId(processInstanceId);
    if(processInstance == null) return null;
    TaskInfo task = null;
    // 先从当前任务中找
    List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstanceId).taskDefinitionKey(taskKey).list();
    if(!CollectionUtils.isEmpty(tasks)){
      task = tasks.get(0);
    }
    if(task == null){
      // 再到历史任务中找
      List<HistoricTaskInstance> taskInstances = historyService.createHistoricTaskInstanceQuery().processInstanceId(processInstanceId).taskDefinitionKey(taskKey)
          .orderByTaskCreateTime().desc().list();
      if(!CollectionUtils.isEmpty(taskInstances)) {
        task = taskInstances.get(0);
      }
    }
    if(task == null) return null;
    ProcessTemplateNodeEntity node = processTemplateNodeService.findDetailByProcessTemplateIdAndProcessNodeId(processTemplate.getId(), taskKey);
    return this.initTaskInfo(processInstance, processTemplate, task, node);
  }

  private TaskVo initTaskInfo(ProcessInstanceEntity processInstance, ProcessTemplateEntity processTemplate, TaskInfo task, ProcessTemplateNodeEntity node){
    TaskVo taskVO = new TaskVo();
    taskVO.setCreateTime(task.getCreateTime());
    taskVO.setFormNo(processInstance.getFormNo());
    taskVO.setProcessInstanceId(processInstance.getProcessInstanceId());
    taskVO.setProcessKey(processTemplate.getProcessKey());
    taskVO.setProcessName(processTemplate.getProcessName());
    taskVO.setProcessState(processInstance.getProcessState());
    taskVO.setTaskDefinitionKey(task.getTaskDefinitionKey());
    taskVO.setTaskId(task.getId());
    taskVO.setTaskName(task.getName());
    taskVO.setNode(node);
    return taskVO;
  }

  @Override
  public TaskVo findTaskInfoByTaskId(String taskId) {
    if(StringUtils.isBlank(taskId)) return null;
    // 先从当前任务中找
    TaskInfo task = taskService.createTaskQuery().taskId(taskId).singleResult();
    if(task == null){
      // 再到历史任务中找
      task = historyService.createHistoricTaskInstanceQuery().taskId(taskId).singleResult();
    }
    if(task == null) return null;
    ProcessTemplateEntity processTemplate = processTemplateService.findDetailsByProcessInstanceId(task.getProcessInstanceId());
    if(processTemplate == null) return null;
    ProcessInstanceEntity processInstance = processInstanceService.findDetailsByProcessInstanceId(task.getProcessInstanceId());
    if(processInstance == null) return null;
    ProcessTemplateNodeEntity node = processTemplateNodeService.findDetailByProcessTemplateIdAndProcessNodeId(processTemplate.getId(), task.getTaskDefinitionKey());
    return this.initTaskInfo(processInstance, processTemplate, task, node);
  }

  @Override
  public List<ProcessAssignmentEntity> findCanSubMultiAssignments(String taskId, Principal principal) {
    Validate.notBlank(taskId, "任务ID不能为空");
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "未找到流程任务");
    UserEntity user = super.getLoginUser(principal);
    ExecutionQuery processExecutionQuery = runtimeService.createExecutionQuery();
    Validate.notNull(processExecutionQuery, "创建查询对象失败，查询对象为空");
    Execution processExecution = processExecutionQuery.processInstanceId(task.getProcessInstanceId())
        .onlyProcessInstanceExecutions()
        .singleResult();
    Validate.notNull(processExecution, "未找到流程执行实例");
    ExecutionQuery taskExecutionQuery = runtimeService.createExecutionQuery();
    Validate.notNull(processExecutionQuery, "创建查询对象失败，查询对象为空");
    Execution taskExecution = taskExecutionQuery.executionId(task.getExecutionId())
        .singleResult();
    Validate.notNull(taskExecution, "流程任务执行对象不存在");
    String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COLLECTION_SUFFIX);
    String completedCollection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COMPLETED_ASSIGNMENTS_SUFFIX);
    List<String> miAssignments = this.getEditableMultiAssignments(processExecution.getId(), collection, false);
    List<String> completedMiAssignments = this.getEditableMultiAssignments(taskExecution.getParentId(), completedCollection, true);
    if(CollectionUtils.isEmpty(miAssignments)) {
      return null;
    }
    List<ProcessAssignmentEntity> assignments = new ArrayList<>();
    for (String miAssignment : miAssignments) {
      if(processAssignmentService.equals(user, miAssignment) || completedMiAssignments.contains(miAssignment)) {
        continue;
      }
      try {
        ProcessAssignmentEntity assignment = processAssignmentService.findAssignment(miAssignment);
        assignments.add(assignment);
      } catch (Exception e) {
        // 当解析的审批人不存在时会抛出异常
        LOGGER.warn(e.getMessage(), e);
      }
    }
    return assignments;
  }

  @Override
  public List<ProcessAssignmentEntity> findTaskMultiAssignments(String taskId) {
    Validate.notBlank(taskId, "任务ID不能为空");
    Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
    Validate.notNull(task, "未找到流程任务");
    ExecutionQuery processExecutionQuery = runtimeService.createExecutionQuery();
    Validate.notNull(processExecutionQuery, "创建查询对象失败，查询对象为空");
    Execution processExecution = processExecutionQuery.processInstanceId(task.getProcessInstanceId())
        .onlyProcessInstanceExecutions()
        .singleResult();
    Validate.notNull(processExecution, "未找到流程执行实例");
    String collection = StringUtils.join(task.getTaskDefinitionKey(), TASK_MULTI_COLLECTION_SUFFIX);
    List<String> miAssignments = this.getEditableMultiAssignments(processExecution.getId(), collection, false);
    if(CollectionUtils.isEmpty(miAssignments)) {
      return null;
    }
    List<ProcessAssignmentEntity> assignments = new ArrayList<>();
    for (String miAssignment : miAssignments) {
      try {
        ProcessAssignmentEntity assignment = processAssignmentService.findAssignment(miAssignment);
        assignments.add(assignment);
      } catch (Exception e) {
        // 当解析的审批人不存在时会抛出异常
        LOGGER.warn(e.getMessage(), e);
      }
    }
    return assignments;
  }

}
