package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.titan.entity.ProcessListenerEntity;
import com.bizunited.platform.titan.entity.ProcessTemplateEntity;
import com.bizunited.platform.titan.entity.ProcessTemplateListenerEntity;
import com.bizunited.platform.titan.entity.ProcessVariableEntity;
import com.bizunited.platform.titan.starter.repository.ProcessTemplateListenerRepository;
import com.bizunited.platform.titan.starter.service.ProcessListenerService;
import com.bizunited.platform.titan.starter.service.ProcessTemplateListenerService;
import com.bizunited.platform.titan.starter.service.ProcessVariableService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-14 14:16
 * @Description: 流程模版监听器服务的实现
 */
@Service("ProcessTemplateListenerServiceImpl")
public class ProcessTemplateListenerServiceImpl implements ProcessTemplateListenerService {

  @Autowired
  private ProcessTemplateListenerRepository processTemplateListenerRepository;

  @Autowired
  private ScriptService scriptService;
  @Autowired
  private ProcessVariableService processVariableService;
  @Autowired
  private ProcessListenerService processListenerService;

  /**
   * 验证创建流程模版监听器参数
   * @param processTemplateListeners
   */
  private void createValidation(Set<ProcessTemplateListenerEntity> processTemplateListeners){
    for (ProcessTemplateListenerEntity listener : processTemplateListeners) {
      Validate.isTrue(listener.getId() == null, "创建流程模版监听器不能有ID");
      Validate.notBlank(listener.getTargetId(), "流程元素ID不能为空");
      Validate.notNull(listener.getProcessListener(), "监听器不能为空");
      this.validListenerAndScript(listener);
    }
  }

  /**
   * 验证监听器和脚本对象
   * @param listener
   */
  private void validListenerAndScript(ProcessTemplateListenerEntity listener){
    if(listener.getProcessListener() != null){
      ProcessListenerEntity processListener = listener.getProcessListener();
      Validate.notBlank(processListener.getId(), "关联的监听器ID不能为空");
      processListener = processListenerService.findById(processListener.getId());
      Validate.notNull(processListener, "未找到关联的监听器:%s", processListener.getId());
      listener.setProcessListener(processListener);
    }
    if(listener.getScript() != null){
      ScriptEntity script = listener.getScript();
      Validate.notBlank(script.getId(), "关联的脚本ID不能为空");
      script = scriptService.findById(script.getId());
      Validate.notNull(script, "未找到关联的脚本：%s", script.getId());
      listener.setScript(script);
    }
  }

  @Override
  @Transactional
  public void create(ProcessTemplateEntity processTemplateEntity, Set<ProcessTemplateListenerEntity> processTemplateListeners) {
    Validate.notNull(processTemplateEntity, "流程模版不能为空");
    if(CollectionUtils.isEmpty(processTemplateListeners)) {
      processTemplateEntity.setProcessTemplateListeners(new HashSet<>());
      return;
    }
    createValidation(processTemplateListeners);
    for (ProcessTemplateListenerEntity listener : processTemplateListeners) {
      listener.setCreateTime(new Date());
      listener.setModifyTime(new Date());
      listener.setProcessTemplate(processTemplateEntity);
      processTemplateListenerRepository.save(listener);
      Set<ProcessVariableEntity> variables = processVariableService.save(listener.getId(), listener.getVariables());
      listener.setVariables(variables);
    }
    processTemplateEntity.setProcessTemplateListeners(processTemplateListeners);
  }

  /**
   * 验证更新流程模版监听器参数
   * @param processTemplateListeners
   */
  private void updateValidation(Set<ProcessTemplateListenerEntity> processTemplateListeners){
    for (ProcessTemplateListenerEntity listener : processTemplateListeners) {
      Validate.isTrue(listener.getId() == null || (StringUtils.isNotBlank(listener.getId())), "更新流程模版监听器的ID不能为空字符串");
      Validate.notBlank(listener.getTargetId(), "流程元素ID不能为空");
      Validate.notNull(listener.getProcessListener(), "监听器不能为空");
      this.validListenerAndScript(listener);
    }
  }

  /**
   * 更新流程模版监听器,id为null则为新增，有ID则更新，不存在的就删除掉
   * @param processTemplateEntity
   * @param processTemplateListeners
   */
  @Override
  @Transactional
  public void update(ProcessTemplateEntity processTemplateEntity, Set<ProcessTemplateListenerEntity> processTemplateListeners) {
    Validate.notNull(processTemplateEntity, "流程模版不能为空");
    if(processTemplateListeners == null) processTemplateListeners = new HashSet<>();
    updateValidation(processTemplateListeners);
    List<ProcessTemplateListenerEntity> oldListeners = processTemplateListenerRepository.findByProcessTemplateId(processTemplateEntity.getId());
    Map<String, ProcessTemplateListenerEntity> oldListenersMap = new HashMap<>(16);
    if(!CollectionUtils.isEmpty(oldListeners)){
      for (ProcessTemplateListenerEntity listener : oldListeners) {
        oldListenersMap.put(listener.getId(), listener);
      }
    }
    Set<ProcessTemplateListenerEntity> createListeners = new HashSet<>();
    Set<ProcessTemplateListenerEntity> updateListeners = new HashSet<>();
    for (ProcessTemplateListenerEntity listener : processTemplateListeners) {
      if(listener.getId() == null){
        // id为空，暂时放到新增列表
        createListeners.add(listener);
        continue;
      }
      // 更新需要更新的对象
      ProcessTemplateListenerEntity listenerEntity = oldListenersMap.get(listener.getId());
      Validate.notNull(listenerEntity, "更新对象不存在，ID:%s", listener.getId());
      listenerEntity.setModifyTime(new Date());
      listenerEntity.setProcessListener(listener.getProcessListener());
      listenerEntity.setScript(listener.getScript());
      listenerEntity.setTargetId(listener.getTargetId());
      processTemplateListenerRepository.save(listenerEntity);
      Set<ProcessVariableEntity> variables = processVariableService.save(listenerEntity.getId(), listener.getVariables());
      listenerEntity.setVariables(variables);
      updateListeners.add(listenerEntity);
      oldListenersMap.remove(listener.getId());
    }
    // 调创建方法保存新增的数据
    this.create(processTemplateEntity, createListeners);
    processTemplateEntity.getProcessTemplateListeners().addAll(updateListeners);
    // 删除不存在的数据
    if(!oldListenersMap.isEmpty()){
      oldListenersMap.forEach((key, listener) -> {
        processVariableService.deleteBySourceId(listener.getId());
        processTemplateListenerRepository.delete(listener);
      });
    }
  }

  @Override
  public Set<ProcessTemplateListenerEntity> findDetailsByProcessTemplateId(String processTemplateId) {
    if(StringUtils.isBlank(processTemplateId)) return null;
    Set<ProcessTemplateListenerEntity> listeners = processTemplateListenerRepository.findDetailsByProcessTemplateId(processTemplateId);
    if(!CollectionUtils.isEmpty(listeners)){
      for (ProcessTemplateListenerEntity listener : listeners) {
        Set<ProcessVariableEntity> variables = processVariableService.findBySourceId(listener.getId());
        listener.setVariables(variables);
      }
    }
    return listeners;
  }

  @Override
  public List<ProcessTemplateListenerEntity> findDetailsByProcessDefinitionIdAndTargetIdAndEvent(String processDefinitionId, String targetId, String event) {
    if(StringUtils.isBlank(processDefinitionId)) return null;
    if(StringUtils.isBlank(targetId)) return null;
    if(StringUtils.isBlank(event)) return null;
    List<ProcessTemplateListenerEntity> listeners = processTemplateListenerRepository.findDetailsByProcessDefinitionIdAndTargetIdAndEvent(processDefinitionId, targetId, event);
    if(!CollectionUtils.isEmpty(listeners)){
      for (ProcessTemplateListenerEntity listener : listeners) {
        Set<ProcessVariableEntity> variables = processVariableService.findBySourceId(listener.getId());
        listener.setVariables(variables);
      }
    }
    return listeners;
  }

}
