package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.titan.entity.ProcessTemplateNodePermissionEntity;
import com.bizunited.platform.titan.starter.repository.ProcessTemplateNodePermissionRepository;
import com.bizunited.platform.titan.starter.service.ProcessTemplateNodePermissionService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.Date;
import java.util.Optional;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-21 17:38
 * @Description: 流程模版节点操作权限的服务接口的实现
 */
@Service("ProcessTemplateNodePermissionServiceImpl")
public class ProcessTemplateNodePermissionServiceImpl extends BaseService implements ProcessTemplateNodePermissionService {

  @Autowired
  private ProcessTemplateNodePermissionRepository processTemplateNodePermissionRepository;

  private void createValidation(ProcessTemplateNodePermissionEntity nodePermission){
    Validate.notNull(nodePermission, "保存对象不能为空");
    Validate.isTrue(nodePermission.getId() == null, "创建对象不能有ID");
    Validate.notBlank(nodePermission.getOperateCode(), "操作编码不能为空");
    Validate.notBlank(nodePermission.getOperateName(), "操作名称不能为空");
    Validate.notNull(nodePermission.getOperations(), "操作内容不能为空");
    Long operateCodeCount = processTemplateNodePermissionRepository.countByOperateCode(nodePermission.getOperateCode());
    Validate.isTrue(operateCodeCount == null || operateCodeCount < 1, "已存在操作编码:%s", nodePermission.getOperateCode());
    Long operateNameCount = processTemplateNodePermissionRepository.countByOperateName(nodePermission.getOperateName());
    Validate.isTrue(operateNameCount == null || operateNameCount < 1, "已存在操作名称:%s", nodePermission.getOperateName());
  }

  @Override
  @Transactional
  public ProcessTemplateNodePermissionEntity create(ProcessTemplateNodePermissionEntity nodePermission, Principal principal) {
    createValidation(nodePermission);
    UserEntity user = super.getLoginUser(principal);
    nodePermission.setCreateTime(new Date());
    nodePermission.setCreateUser(user);
    nodePermission.setModifyTime(new Date());
    nodePermission.setModifyUser(user);
    processTemplateNodePermissionRepository.save(nodePermission);
    return nodePermission;
  }

  private void updateValidation(ProcessTemplateNodePermissionEntity nodePermission){
    Validate.notNull(nodePermission, "更新对象不能为空");
    Validate.isTrue(StringUtils.isNotBlank(nodePermission.getId()), "更新对象必须有ID");
    Validate.notBlank(nodePermission.getOperateCode(), "操作编码不能为空");
    Validate.notBlank(nodePermission.getOperateName(), "操作名称不能为空");
    Validate.notNull(nodePermission.getOperations(), "操作内容不能为空");
    Long operateCodeCount = processTemplateNodePermissionRepository.countByOperateCodeWithoutId(nodePermission.getOperateCode(), nodePermission.getId());
    Validate.isTrue(operateCodeCount == null || operateCodeCount < 1, "已存在操作编码:%s", nodePermission.getOperateCode());
    Long operateNameCount = processTemplateNodePermissionRepository.countByOperateNameWithoutId(nodePermission.getOperateName(), nodePermission.getId());
    Validate.isTrue(operateNameCount == null || operateNameCount < 1, "已存在操作名称:%s", nodePermission.getOperateName());
  }

  @Override
  @Transactional
  public ProcessTemplateNodePermissionEntity update(ProcessTemplateNodePermissionEntity nodePermission, Principal principal) {
    updateValidation(nodePermission);
    UserEntity user = super.getLoginUser(principal);
    Optional<ProcessTemplateNodePermissionEntity> op = processTemplateNodePermissionRepository.findById(nodePermission.getId());
    ProcessTemplateNodePermissionEntity processTemplateNodePermission = op.orElse(null);
    Validate.notNull(processTemplateNodePermission, "更新对象不存在");

    processTemplateNodePermission.setModifyUser(user);
    processTemplateNodePermission.setModifyTime(new Date());
    processTemplateNodePermission.setOperateCode(nodePermission.getOperateCode());
    processTemplateNodePermission.setOperateName(nodePermission.getOperateName());
    processTemplateNodePermission.setOperations(nodePermission.getOperations());
    processTemplateNodePermissionRepository.save(processTemplateNodePermission);
    return processTemplateNodePermission;
  }

  @Override
  public Page<ProcessTemplateNodePermissionEntity> findByConditions(ProcessTemplateNodePermissionEntity nodePermission, Pageable pageable) {
    return processTemplateNodePermissionRepository.findByConditions(nodePermission, pageable);
  }

  @Override
  public ProcessTemplateNodePermissionEntity findDetailById(String id) {
    if(StringUtils.isBlank(id)) return null;
    return processTemplateNodePermissionRepository.findDetailById(id);
  }
}
