package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.repository.ProcessTemplateNodeRepository;
import com.bizunited.platform.titan.starter.service.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-13 18:00
 * @Description: 流程节点的服务接口实现
 */
@Service("ProcessTemplateNodeServiceImpl")
public class ProcessTemplateNodeServiceImpl implements ProcessTemplateNodeService {

  @Autowired
  private ProcessTemplateNodeRepository processTemplateNodeRepository;

  @Autowired
  private ProcessVariableService processVariableService;
  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessTemplateNodeMultiService processTemplateNodeMultiService;
  @Autowired
  private ProcessTemplateNodeAssignmentService processTemplateNodeAssignmentService;

  private void createValidation(Set<ProcessTemplateNodeEntity> processTemplateNodes){
    Set<String> nodeIds = new HashSet<>();
    for (ProcessTemplateNodeEntity node : processTemplateNodes) {
      Validate.isTrue(node.getId() == null, "创建流程节点不能有ID");
      Validate.notBlank(node.getProcessNodeId(), "节点ID不能为空");
      Validate.notBlank(node.getProcessNodeName(), "节点名称不能为空");
      Validate.isTrue(!nodeIds.contains(node.getProcessNodeId()), "节点ID不能重复");
      nodeIds.add(node.getProcessNodeId());
    }
  }

  @Override
  @Transactional
  public void create(ProcessTemplateEntity processTemplateEntity, Set<ProcessTemplateNodeEntity> processTemplateNodes) {
    if(CollectionUtils.isEmpty(processTemplateNodes)) return;
    createValidation(processTemplateNodes);
    for (ProcessTemplateNodeEntity node : processTemplateNodes) {
      if(node.getCanBack() == null) node.setCanBack(false);
      if(node.getCancelFlag() == null) node.setCancelFlag(false);
      if(node.getManualNext() == null) node.setManualNext(false);
      if(node.getNullSkip() == null) node.setNullSkip(false);
      if(node.getRepeatSkip() == null) node.setRepeatSkip(false);
      node.setCreateTime(new Date());
      node.setModifyTime(new Date());
      node.setProcessTemplate(processTemplateEntity);
      ProcessTemplateNodeMultiEntity processTemplateNodeMulti = node.getProcessTemplateNodeMulti();
      ProcessTemplateNodeAssignmentEntity assignment = node.getAssignment();
      node.setProcessTemplateNodeMulti(null);
      node.setAssignment(null);
      processTemplateNodeRepository.save(node);
      processTemplateNodeAssignmentService.create(node, assignment);
      ProcessTemplateNodeMultiEntity nodeMultiEntity = processTemplateNodeMultiService.save(node, processTemplateNodeMulti);
      node.setProcessTemplateNodeMulti(nodeMultiEntity);
    }
    processTemplateEntity.setProcessTemplateNodes(processTemplateNodes);
  }

  /**
   * 验证更新流程模版节点信息
   * @param processTemplateNodes
   */
  private void updateValidation(Set<ProcessTemplateNodeEntity> processTemplateNodes){
    if(CollectionUtils.isEmpty(processTemplateNodes)){
      return;
    }
    Set<String> nodeIds = new HashSet<>();
    for (ProcessTemplateNodeEntity node : processTemplateNodes) {
      Validate.isTrue(node.getId() == null || (StringUtils.isNotBlank(node.getId())), "更新流程模版节点的ID不能为空字符串");
      Validate.notBlank(node.getProcessNodeId(), "节点ID不能为空");
      Validate.notBlank(node.getProcessNodeName(), "节点名称不能为空");
      Validate.isTrue(!nodeIds.contains(node.getProcessNodeId()), "节点ID不能重复");
      nodeIds.add(node.getProcessNodeId());
    }
  }

  /**
   * 更新流程模版节点，与旧数据对比，删除不存在的，更新已存在的，插入新增的
   * @param processTemplateEntity
   * @param processTemplateNodes
   */
  @Override
  @Transactional
  public void update(ProcessTemplateEntity processTemplateEntity, Set<ProcessTemplateNodeEntity> processTemplateNodes) {
    if(processTemplateNodes == null) processTemplateNodes = new HashSet<>();
    updateValidation(processTemplateNodes);
    Set<ProcessTemplateNodeEntity> oldNodes = this.findDetailsByProcessTemplateId(processTemplateEntity.getId());
    // 用map暂存，减少遍历
    Map<String, ProcessTemplateNodeEntity> oldNodesMap = new HashMap<>(16);
    if(!CollectionUtils.isEmpty(oldNodes)){
      for (ProcessTemplateNodeEntity oldNode : oldNodes) {
        oldNodesMap.put(oldNode.getId(), oldNode);
      }
    }
    Set<ProcessTemplateNodeEntity> createNodes = new HashSet<>();
    Set<ProcessTemplateNodeEntity> updateNodes = new HashSet<>();
    for (ProcessTemplateNodeEntity node : processTemplateNodes) {
      if(node.getId() == null){
        // id为null则加到新增列表
        createNodes.add(node);
        continue;
      }
      // id不为null则更新对象并加到更新列表
      ProcessTemplateNodeEntity oldNode = oldNodesMap.get(node.getId());
      Validate.notNull(oldNode, "更新对象不存在，ID:%s", node.getId());
      if(node.getCanBack() != null) oldNode.setCanBack(node.getCanBack());
      if(node.getCancelFlag() != null) oldNode.setCancelFlag(node.getCancelFlag());
      if(node.getManualNext() != null) oldNode.setManualNext(node.getManualNext());
      if(node.getNullSkip() != null) oldNode.setNullSkip(node.getNullSkip());
      if(node.getRepeatSkip() != null) oldNode.setRepeatSkip(node.getRepeatSkip());
      oldNode.setProcessNodeId(node.getProcessNodeId());
      oldNode.setProcessNodeName(node.getProcessNodeName());
      oldNode.setProcessNodeDoc(node.getProcessNodeDoc());
      oldNode.setBtnAuth(node.getBtnAuth());
      oldNode.setModifyTime(new Date());
      oldNode.setFormVisibilityName(node.getFormVisibilityName());
      oldNode.setProcessTemplateNodePermission(node.getProcessTemplateNodePermission());
      processTemplateNodeRepository.save(oldNode);
      ProcessTemplateNodeMultiEntity nodeMultiEntity = processTemplateNodeMultiService.save(oldNode, node.getProcessTemplateNodeMulti());
      oldNode.setProcessTemplateNodeMulti(nodeMultiEntity);
      processTemplateNodeAssignmentService.update(oldNode, node.getAssignment());
      updateNodes.add(oldNode);
      oldNodesMap.remove(oldNode.getId());
    }
    // 调创建方法保存新增的节点
    this.create(processTemplateEntity, createNodes);
    if(processTemplateEntity.getProcessTemplateNodes() == null){
      processTemplateEntity.setProcessTemplateNodes(new HashSet<>());
    }
    processTemplateEntity.getProcessTemplateNodes().addAll(updateNodes);
    // 删除不存在的节点
    if(!oldNodesMap.isEmpty()){
      oldNodesMap.forEach((key, node) -> {
        processTemplateNodeRepository.delete(node);
        if(node.getProcessTemplateNodeMulti() != null){
          processAssignmentService.deleteByResourceId(node.getProcessTemplateNodeMulti().getId());
        }
      });
    }
  }

  @Override
  public ProcessTemplateNodeEntity findByProcessTemplateIdAndProcessNodeId(String templateId, String processNodeId) {
    return processTemplateNodeRepository.findByProcessTemplateIdAndProcessNodeId(templateId, processNodeId);
  }

  @Override
  public ProcessTemplateNodeEntity findDetailByProcessTemplateIdAndProcessNodeId(String templateId, String processNodeId) {
    ProcessTemplateNodeEntity node = processTemplateNodeRepository.findDetailsByProcessTemplateIdAndProcessNodeId(templateId, processNodeId);
    this.loadNodeDetails(node);
    return node;
  }

  /**
   * 加载节点详情信息
   * @param node
   * @return
   */
  private ProcessTemplateNodeEntity loadNodeDetails(ProcessTemplateNodeEntity node){
    if(node == null) return null;
    if(node.getAssignment() != null){
      Set<ProcessVariableEntity> variables = processVariableService.findBySourceId(node.getAssignment().getId());
      node.getAssignment().setVariables(variables);
    }
    if(node.getProcessTemplateNodeMulti() != null){
      Set<ProcessAssignmentEntity> assignments = processAssignmentService.findByResourceId(node.getProcessTemplateNodeMulti().getId());
      node.getProcessTemplateNodeMulti().setAssignments(assignments);
    }
    return node;
  }

  @Override
  public Set<ProcessTemplateNodeEntity> findDetailsByProcessTemplateId(String processTemplateId) {
    if(StringUtils.isBlank(processTemplateId)) return null;
    Set<ProcessTemplateNodeEntity> nodes = processTemplateNodeRepository.findDetailsByProcessTemplateId(processTemplateId);
    if(!CollectionUtils.isEmpty(nodes)){
      for (ProcessTemplateNodeEntity node : nodes) {
        this.loadNodeDetails(node);
      }
    }
    return nodes;
  }

  @Override
  public List<ProcessTemplateNodeEntity> findByProcessTemplateIdAndCancelFlag(String processTemplateId, boolean cancelFlag) {
    return processTemplateNodeRepository.findByProcessTemplateIdAndCancelFlag(processTemplateId, cancelFlag);
  }

  @Override
  public List<ProcessTemplateNodeEntity> findByProcessTemplateId(String templateId) {
    return processTemplateNodeRepository.findByProcessTemplateId(templateId);
  }

}
