package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.titan.entity.ProcessTemplateEntity;
import com.bizunited.platform.titan.entity.ProcessTemplatePermissionEntity;
import com.bizunited.platform.titan.starter.repository.ProcessTemplatePermissionRepository;
import com.bizunited.platform.titan.starter.service.ProcessTemplatePermissionService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-13 17:00
 * @Description: 流程模版权限服务的实现
 */
@Service("ProcessTemplatePermissionServiceImpl")
public class ProcessTemplatePermissionServiceImpl implements ProcessTemplatePermissionService {

  @Autowired
  private ProcessTemplatePermissionRepository processTemplatePermissionRepository;

  private void createValidation(Set<ProcessTemplatePermissionEntity> permissions){
    for (ProcessTemplatePermissionEntity permission : permissions) {
      Validate.isTrue(permission.getId() == null, "创建流程模版权限不能有ID");
      Validate.notNull(permission.getType(), "权限类型不能空");
      if(!permission.getType().equals(0)){
        Validate.notBlank(permission.getPermissionValue(), "权限值不能为空");
      }
    }
  }

  @Override
  @Transactional
  public void create(ProcessTemplateEntity processTemplateEntity, Set<ProcessTemplatePermissionEntity> permissions) {
    if(CollectionUtils.isEmpty(permissions)){
      processTemplateEntity.setPermissions(new HashSet<>());
      return;
    }
    createValidation(permissions);
    for (ProcessTemplatePermissionEntity permission : permissions) {
      permission.setCreateTime(new Date());
      permission.setModifyTime(new Date());
      permission.setProcessTemplate(processTemplateEntity);
    }
    processTemplatePermissionRepository.saveAll(permissions);
    processTemplateEntity.setPermissions(permissions);
  }

  private void updateValidation(Set<ProcessTemplatePermissionEntity> permissions){
    for (ProcessTemplatePermissionEntity permission : permissions) {
      Validate.notNull(permission.getType(), "权限类型不能空");
      if(!permission.getType().equals(0)){
        Validate.notBlank(permission.getPermissionValue(), "权限值不能为空");
      }
    }
  }

  @Override
  @Transactional
  public void update(ProcessTemplateEntity processTemplateEntity, Set<ProcessTemplatePermissionEntity> permissions) {
    if(permissions == null) permissions = new HashSet<>();
    updateValidation(permissions);
    Set<ProcessTemplatePermissionEntity> oldPermissions = processTemplatePermissionRepository.findByProcessTemplateId(processTemplateEntity.getId());
    Map<String, ProcessTemplatePermissionEntity> oldPermissionsMap = new HashMap<>(16);
    if(!CollectionUtils.isEmpty(oldPermissions)){
      for (ProcessTemplatePermissionEntity oldPermission : oldPermissions) {
        oldPermissionsMap.put(oldPermission.getId(), oldPermission);
      }
    }
    Set<ProcessTemplatePermissionEntity> createPermissions = new HashSet<>();
    Set<ProcessTemplatePermissionEntity> updatePermissions = new HashSet<>();
    for (ProcessTemplatePermissionEntity permission : permissions) {
      if(permission.getId() == null){
        createPermissions.add(permission);
        continue;
      }
      ProcessTemplatePermissionEntity oldPermission = oldPermissionsMap.get(permission.getId());
      Validate.notNull(oldPermission, "更新对象不存在，ID:%s", permission.getId());
      oldPermission.setType(permission.getType());
      oldPermission.setPermissionValue(permission.getPermissionValue());
      oldPermission.setModifyTime(new Date());
      updatePermissions.add(oldPermission);
      oldPermissionsMap.remove(oldPermission.getId());
    }
    this.create(processTemplateEntity, createPermissions);
    processTemplatePermissionRepository.saveAll(updatePermissions);
    processTemplateEntity.getPermissions().addAll(updatePermissions);
    if(!CollectionUtils.isEmpty(oldPermissionsMap)){
      oldPermissionsMap.forEach( (k, v) -> processTemplatePermissionRepository.delete(v));
    }
  }

  @Override
  public Set<ProcessTemplatePermissionEntity> findByProcessTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) return null;
    return processTemplatePermissionRepository.findByProcessTemplateId(templateId);
  }

}
