package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.bizunited.platform.titan.entity.*;
import com.bizunited.platform.titan.starter.common.Constants;
import com.bizunited.platform.titan.starter.common.enums.*;
import com.bizunited.platform.titan.starter.repository.ProcessTemplateRepository;
import com.bizunited.platform.titan.starter.service.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.flowable.bpmn.converter.BpmnXMLConverter;
import org.flowable.bpmn.model.Process;
import org.flowable.bpmn.model.*;
import org.flowable.common.engine.api.io.InputStreamProvider;
import org.flowable.common.engine.impl.util.io.BytesStreamSource;
import org.flowable.common.engine.impl.util.io.StringStreamSource;
import org.flowable.engine.RepositoryService;
import org.flowable.engine.repository.Deployment;
import org.flowable.engine.repository.ProcessDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-13 10:55
 * @Description: 流程模版服务的接口实现
 */
@Service("ProcessTemplateServiceImpl")
public class ProcessTemplateServiceImpl extends BaseService implements ProcessTemplateService {

  private static final Logger LOGGER = LoggerFactory.getLogger(ProcessTemplateServiceImpl.class);

  private final static String XML_FILE_SUFFIX = "xml";

  @Autowired
  private ProcessTemplateRepository processTemplateRepository;

  @Autowired
  private RepositoryService repositoryService;
  @Autowired
  private NebulaFileService kuiperFileService;
  @Autowired
  private ProcessVariableService processVariableService;
  @Autowired
  private ProcessAssignmentService processAssignmentService;
  @Autowired
  private ProcessTemplateNodeService processTemplateNodeService;
  @Autowired
  private ProcessTemplateListenerService processTemplateListenerService;
  @Autowired
  private ProcessTemplatePermissionService processTemplatePermissionService;

  @Override
  public Page<ProcessTemplateEntity> findByConditions(Pageable pageable, ProcessTemplateEntity processTemplate) {
    return processTemplateRepository.findByConditions(pageable, processTemplate);
  }

  /**
   * 创建参数验证
   * @param processTemplateEntity
   */
  private void createValidation(ProcessTemplateEntity processTemplateEntity){
    Validate.notNull(processTemplateEntity, "保存对象不存在");
    Validate.isTrue(processTemplateEntity.getId() == null, "创建数据不能有ID");
    Validate.notBlank(processTemplateEntity.getProcessKey(), "流程Key不能为空");
    Validate.isTrue(processTemplateEntity.getCversion() > 0, "流程版本号不能小于1");
    if(processTemplateEntity.getCversion().equals(1)){
      Long count = processTemplateRepository.countByProcessKey(processTemplateEntity.getProcessKey());
      Validate.isTrue(count == null || count == 0, "流程Key已存在，请更换");
    }
    Validate.notBlank(processTemplateEntity.getProcessName(), "流程名称不能为空");
    Validate.notBlank(processTemplateEntity.getProcessXml(), "流程文件不能为空");
    Validate.notNull(processTemplateEntity.getFormType(), "表单类型不能为空");
    if(ProcessFormType.FORM_TEMPLATE.getType().equals(processTemplateEntity.getFormType())){
      Validate.notBlank(processTemplateEntity.getFormTemplateId(), "表单模版Id不能为空");
    } else if(ProcessFormType.FORM_URL.getType().equals(processTemplateEntity.getFormType())){
      Validate.notBlank(processTemplateEntity.getFormUrl(), "表单地址不能为空");
    }
    // 解析xml文件较慢，放到最后
    this.validateProcessXml(processTemplateEntity, false);
  }

  /**
   * 验证流程文件
   * @param processTemplateEntity
   */
  private void validateProcessXml(ProcessTemplateEntity processTemplateEntity, boolean isDeploy){
    InputStreamProvider inputStreamProvider = new StringStreamSource(processTemplateEntity.getProcessXml(), "UTF-8");
    BpmnXMLConverter bpmnXMLConverter = new BpmnXMLConverter();
    BpmnModel bpmnModel;
    try {
      bpmnModel = bpmnXMLConverter.convertToBpmnModel(inputStreamProvider, true, false, "UTF-8");
    } catch (Exception e){
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("解析流程文件错误，请检查流程文件", e);
    }
    Process process = bpmnModel.getProcessById(processTemplateEntity.getProcessKey());
    Validate.notNull(process, "流程key与流程文件ID不一致，请检查");
    Collection<FlowElement> flowElements = process.getFlowElements();
    List<FlowElement> userTasks = flowElements.stream().filter(f -> f instanceof UserTask).collect(Collectors.toList());
    if(isDeploy || !CollectionUtils.isEmpty(userTasks)){
      // 如果是部署前或者流程有节点，则需要检验第一个节点的配置
      Validate.notEmpty(userTasks, "流程必须有一个节点");
      UserTask userTask = (UserTask) userTasks.get(0);
      Validate.isTrue(userTask.getId().equals(Constants.NODE_ID_DEFAULT_START), "发起节点的ID必须为：%s", Constants.NODE_ID_DEFAULT_START);
      Validate.isTrue(Constants.TASK_DEFAULT_START_ASSIGNMENT.equals(userTask.getAssignee()), "发起节点的审批值必须为：%s", Constants.TASK_DEFAULT_START_ASSIGNMENT);
    }
  }

  /**
   * 创建流程模版,新创建的cvertion必须为1，后面升级的版本要高于1
   * @param processTemplateEntity
   * @param principal
   * @param cvertion
   * @return
   */
  @Override
  @Transactional
  public ProcessTemplateEntity create(ProcessTemplateEntity processTemplateEntity, Principal principal, int cvertion) {
    processTemplateEntity.setCversion(cvertion);
    UserEntity creator = super.getLoginUser(principal);
    createValidation(processTemplateEntity);
    Date nowDate = new Date();
    processTemplateEntity.setCreateUser(creator);
    processTemplateEntity.setProcessState(ProcessTemplateState.DRAFT.getState());
    processTemplateEntity.setCreateTime(nowDate);
    processTemplateEntity.setModifyUser(creator);
    processTemplateEntity.setModifyTime(nowDate);
    processTemplateEntity.setLastVersion(true);
    processTemplateEntity.setLastDeployedVersion(false);
    {
      //保存流程内容到文件
      String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
      String renameImage = UUID.randomUUID().toString();
      String fileRename = renameImage + "." + XML_FILE_SUFFIX;
      String relativePath = StringUtils.join("/process/xml/" , folderName , "/" , (new Random().nextInt(100) % 10));
      byte[] processContent = processTemplateEntity.getProcessXml().getBytes(StandardCharsets.UTF_8);
      kuiperFileService.saveFile(relativePath, fileRename, fileRename, processContent);
      processTemplateEntity.setFilePath(relativePath);
      processTemplateEntity.setFileName(fileRename);
    }
    processTemplateRepository.save(processTemplateEntity);
    if(processTemplateEntity.getPermissions() != null){
      processTemplatePermissionService.create(processTemplateEntity, processTemplateEntity.getPermissions());
    }
    if(processTemplateEntity.getProcessTemplateNodes() != null){
      processTemplateNodeService.create(processTemplateEntity, processTemplateEntity.getProcessTemplateNodes());
    }
    if(processTemplateEntity.getProcessTemplateListeners() != null){
      processTemplateListenerService.create(processTemplateEntity, processTemplateEntity.getProcessTemplateListeners());
    }
    Set<ProcessVariableEntity> variables = processVariableService.save(processTemplateEntity.getId(), processTemplateEntity.getVariables());
    processTemplateEntity.setVariables(variables);
    return processTemplateEntity;
  }

  /**
   * 更新流程参数验证
   * @param processTemplateEntity
   */
  private void updateValidation(ProcessTemplateEntity processTemplateEntity){
    Validate.notNull(processTemplateEntity, "保存对象不存在");
    Validate.notBlank(processTemplateEntity.getProcessKey(), "流程Key不能为空");
    if(processTemplateEntity.getCversion().equals(1)) {
      Long count = processTemplateRepository.countByProcessKeyWithoutId(processTemplateEntity.getProcessKey(), processTemplateEntity.getId());
      Validate.isTrue(count == null || count == 0, "流程Key重复，请更换");
    }
    Validate.notBlank(processTemplateEntity.getProcessName(), "流程名称不能为空");
    Validate.notBlank(processTemplateEntity.getProcessXml(), "流程文件不能为空");
    Validate.notNull(processTemplateEntity.getFormType(), "表单类型不能为空");
    if(ProcessFormType.FORM_TEMPLATE.getType().equals(processTemplateEntity.getFormType())){
      Validate.notBlank(processTemplateEntity.getFormTemplateId(), "表单模版Id不能为空");
    } else if(ProcessFormType.FORM_URL.getType().equals(processTemplateEntity.getFormType())){
      Validate.notBlank(processTemplateEntity.getFormUrl(), "表单地址不能为空");
    }
    // 解析xml文件较慢，放到最后
    this.validateProcessXml(processTemplateEntity, false);
  }

  @Override
  @Transactional
  public ProcessTemplateEntity update(ProcessTemplateEntity processTemplateEntity, Principal principal) {
    Validate.isTrue( StringUtils.isNotBlank(processTemplateEntity.getId()), "更新数据必须要有ID");
    UserEntity user = super.getLoginUser(principal);
    ProcessTemplateEntity updateTemplate = processTemplateRepository.findDetailsById(processTemplateEntity.getId());
    Validate.notNull(updateTemplate, "更新对象不存在");

    Validate.isTrue(updateTemplate.getProcessState().equals(ProcessTemplateState.DRAFT.getState()), "已发布的流程模版不能进行编辑");
    processTemplateEntity.setCversion(updateTemplate.getCversion());
    this.updateValidation(processTemplateEntity);
    ProcessTemplateEntity lastDeployedVersion = processTemplateRepository.findLastDeployVersionByProcessKey(updateTemplate.getProcessKey());
    if(lastDeployedVersion != null){
      Validate.isTrue(lastDeployedVersion.getProcessKey().equals(processTemplateEntity.getProcessKey()), "流程key必须与已发布的模版一致");
    }
    updateTemplate.setModifyTime(new Date());
    updateTemplate.setModifyUser(user);
    updateTemplate.setFormName(processTemplateEntity.getFormName());
    updateTemplate.setFormVersion(processTemplateEntity.getFormVersion());
    updateTemplate.setProcessName(processTemplateEntity.getProcessName());
    updateTemplate.setProcessKey(processTemplateEntity.getProcessKey());
    updateTemplate.setFormUrl(processTemplateEntity.getFormUrl());
    updateTemplate.setFormCallBackUrl(processTemplateEntity.getFormCallBackUrl());
    updateTemplate.setFormTemplateId(processTemplateEntity.getFormTemplateId());
    updateTemplate.setFormType(processTemplateEntity.getFormType());
    updateTemplate.setProcessDoc(processTemplateEntity.getProcessDoc());
    updateTemplate.setProcessXml(processTemplateEntity.getProcessXml());
    updateTemplate.setOnSubmitScript(processTemplateEntity.getOnSubmitScript());
    {
      //更新流程文件
      byte[] processContent = processTemplateEntity.getProcessXml().getBytes(StandardCharsets.UTF_8);
      String fileName = updateTemplate.getFileName();
      kuiperFileService.saveFile(updateTemplate.getFilePath(), fileName, fileName, processContent);
    }
    processTemplateRepository.save(updateTemplate);
    processTemplatePermissionService.update(processTemplateEntity, processTemplateEntity.getPermissions());
    processTemplateNodeService.update(processTemplateEntity, processTemplateEntity.getProcessTemplateNodes());
    processTemplateListenerService.update(processTemplateEntity, processTemplateEntity.getProcessTemplateListeners());
    Set<ProcessVariableEntity> variables = processVariableService.save(updateTemplate.getId(), processTemplateEntity.getVariables());
    updateTemplate.setVariables(variables);
    return updateTemplate;
  }

  @Override
  public ProcessTemplateEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) return null;
    ProcessTemplateEntity processTemplate = processTemplateRepository.findDetailsById(id);
    if(processTemplate == null) return null;
    return this.loadDetails(processTemplate);
  }

  /**
   * 加载明细
   * @param processTemplate
   * @return
   */
  private ProcessTemplateEntity loadDetails(ProcessTemplateEntity processTemplate){
    Set<ProcessTemplateNodeEntity> nodes = processTemplateNodeService.findDetailsByProcessTemplateId(processTemplate.getId());
    Set<ProcessTemplatePermissionEntity> permissions = processTemplatePermissionService.findByProcessTemplateId(processTemplate.getId());
    Set<ProcessTemplateListenerEntity> listeners = processTemplateListenerService.findDetailsByProcessTemplateId(processTemplate.getId());
    Set<ProcessVariableEntity> variables = processVariableService.findBySourceId(processTemplate.getId());
    processTemplate.setProcessTemplateNodes(nodes);
    processTemplate.setPermissions(permissions);
    processTemplate.setProcessTemplateListeners(listeners);
    processTemplate.setVariables(variables);
    String fileName = processTemplate.getFileName();
    byte[] bytes = kuiperFileService.readFileContent(processTemplate.getFilePath(), fileName);
    if(bytes != null && bytes.length > 0){
      try {
        processTemplate.setProcessXml(new String(bytes, StandardCharsets.UTF_8));
      } catch (Exception e) {
        LOGGER.warn(e.getMessage(), e);
      }
    }
    return processTemplate;
  }

  @Override
  public ProcessTemplateEntity findDetailsByProcessInstanceId(String processInstanceId) {
    if(StringUtils.isBlank(processInstanceId)) return null;
    ProcessTemplateEntity processTemplate = processTemplateRepository.findDetailsByProcessInstanceId(processInstanceId);
    if(processTemplate == null) return null;
    return this.loadDetails(processTemplate);
  }

  @Override
  @Transactional
  public ProcessTemplateEntity upgradeVersion(ProcessTemplateEntity processTemplate, Principal principal) {
    Validate.notBlank(processTemplate.getProcessKey(), "流程key不能为空");
    UserEntity user = super.getLoginUser(principal);
    ProcessTemplateEntity lastVersion = processTemplateRepository.findLastVersionByProcessKey(processTemplate.getProcessKey());
    Validate.notNull(lastVersion, "无可用于升级的流程模版");
    Validate.isTrue(lastVersion.getProcessState().equals(ProcessTemplateState.PUBLISHED.getState()), "当前流程模版存在草稿模版，请在草稿模版上进行编辑");
    int newVersion = lastVersion.getCversion() + 1;
    {
      // 更新当前版本
      lastVersion.setModifyUser(user);
      lastVersion.setModifyTime(new Date());
      lastVersion.setLastVersion(false);
      processTemplateRepository.save(lastVersion);
    }
    processTemplate.setProcessDefinitionId(null);
    processTemplate.setDeployTime(null);
    {
      // 将最后版本的所有实体的ID设置成null
      processTemplate.setId(null);
      processTemplate.getPermissions().forEach(permission -> permission.setId(null));
      if(!CollectionUtils.isEmpty(processTemplate.getProcessTemplateListeners())){
        processTemplate.getProcessTemplateListeners().forEach(listener -> {
          listener.setId(null);
          if(listener.getVariables() != null){
            listener.getVariables().forEach(v -> v.setId(null));
          }
        });
      }
      if(!CollectionUtils.isEmpty(processTemplate.getProcessTemplateNodes())){
        processTemplate.getProcessTemplateNodes().forEach( node -> {
          node.setId(null);
          if(node.getAssignment() !=  null){
            node.getAssignment().setId(null);
            if(node.getAssignment().getVariables() != null){
              node.getAssignment().getVariables().forEach(v -> v.setId(null));
            }
          }
          if(node.getProcessTemplateNodeMulti() != null){
            node.getProcessTemplateNodeMulti().setId(null);
            if(!CollectionUtils.isEmpty(node.getProcessTemplateNodeMulti().getAssignments())){
              for (ProcessAssignmentEntity assignment : node.getProcessTemplateNodeMulti().getAssignments()) {
                assignment.setId(null);
              }
            }
          }
        });
      }
    }
    return this.create(processTemplate, principal, newVersion);
  }

  @Override
  public List<ProcessTemplateEntity> findByProcessKey(String processKey) {
    if(StringUtils.isBlank(processKey)) return null;
    Sort sort = new Sort(Sort.Direction.DESC, "cversion");
    return processTemplateRepository.findByProcessKey(processKey, sort);
  }

  @Override
  @Transactional
  public void deploy(String id, Principal principal) {
    Validate.notBlank(id, "ID不能为空");
    UserEntity user = super.getLoginUser(principal);
    ProcessTemplateEntity processTemplate = this.findDetailsById(id);
    deployValidation(processTemplate);
    {
      // 将当前已部署的最新版本修改为不是最新发布版本
      ProcessTemplateEntity lastDeployedVersion = processTemplateRepository.findLastDeployVersionByProcessKey(processTemplate.getProcessKey());
      if(lastDeployedVersion != null){
        lastDeployedVersion.setModifyTime(new Date());
        lastDeployedVersion.setModifyUser(user);
        lastDeployedVersion.setLastDeployedVersion(false);
        processTemplateRepository.save(lastDeployedVersion);
      }
    }
    // 部署流程到流程引擎
    String deployName = processTemplate.getProcessKey() + ".bpmn";
    Deployment deployment = repositoryService.createDeployment()
        .addString(deployName, processTemplate.getProcessXml())
        .name(processTemplate.getProcessKey())
        .key(processTemplate.getProcessKey())
        .deploy();
    ProcessDefinition processDefinition = repositoryService.createProcessDefinitionQuery().deploymentId(deployment.getId()).latestVersion().singleResult();
    // 更新流程模版信息
    processTemplate.setLastDeployedVersion(true);
    processTemplate.setModifyTime(new Date());
    processTemplate.setModifyUser(user);
    processTemplate.setDeployTime(new Date());
    processTemplate.setProcessDefinitionId(processDefinition.getId());
    processTemplate.setProcessState(ProcessTemplateState.PUBLISHED.getState());
    processTemplateRepository.save(processTemplate);
  }

  @Override
  @Transactional
  public void deleteById(String id, Principal principal) {
    Validate.notBlank(id, "ID不能为空");
    UserEntity user = super.getLoginUser(principal);
    Optional<ProcessTemplateEntity> op = processTemplateRepository.findById(id);
    ProcessTemplateEntity processTemplate = op.orElse(null);
    if(processTemplate == null) return;

    Validate.isTrue(processTemplate.getProcessState().equals(ProcessTemplateState.DRAFT.getState()), "流程已发布，不能删除");
    ProcessTemplateEntity lastDeployedVersion = processTemplateRepository.findLastDeployVersionByProcessKey(processTemplate.getProcessKey());
    if(lastDeployedVersion != null){
      lastDeployedVersion.setLastVersion(true);
      lastDeployedVersion.setModifyUser(user);
      lastDeployedVersion.setModifyTime(new Date());
      processTemplateRepository.save(lastDeployedVersion);
    }
    processTemplateRepository.delete(processTemplate);
    String fileName = processTemplate.getFileName();
    kuiperFileService.deleteFile(processTemplate.getFilePath(), fileName, fileName);
  }

  @Override
  public ProcessTemplateEntity findById(String id) {
    Optional<ProcessTemplateEntity> op = processTemplateRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public Page<ProcessTemplateEntity> findStartableByConditions(Pageable pageable, ProcessTemplateEntity processTemplate, Principal principal) {
    if(principal == null) return null;
    UserEntity user = getLoginUser(principal);
    UserVo userVo = userService.findDetailsById(user.getId());
    return processTemplateRepository.findStartableByConditions(pageable, processTemplate, userVo);
  }

  @Override
  public String getEndActivityId(ProcessTemplateEntity processTemplate) {
    Validate.notNull(processTemplate, "流程模版不能为空");
    BpmnModel bpmnModel = repositoryService.getBpmnModel(processTemplate.getProcessDefinitionId());
    Process process = bpmnModel.getProcessById(processTemplate.getProcessKey());
    Collection<FlowElement> flowElements = process.getFlowElements();
    for (FlowElement flowElement : flowElements) {
      if(flowElement instanceof EndEvent){
        return flowElement.getId();
      }
    }
    throw new IllegalArgumentException("未找到流程结束节点");
  }

  @Override
  public List<String> getStartEventAndOutLine(Process process) {
    List<String> activities = new ArrayList<>();
    Collection<FlowElement> flowElements = process.getFlowElements();
    String startEventId = null;
    for (FlowElement flowElement : flowElements) {
      if(flowElement instanceof StartEvent){
        startEventId = flowElement.getId();
        activities.add(startEventId);
        break;
      }
    }
    if(startEventId != null){
      for (FlowElement flowElement : flowElements) {
        if(flowElement instanceof SequenceFlow){
          SequenceFlow sequenceFlow = (SequenceFlow) flowElement;
          if(sequenceFlow.getSourceFlowElement().getId().equals(startEventId)){
            activities.add(sequenceFlow.getId());
            break;
          }
        }
      }
    }
    return activities;
  }

  @Override
  public FlowElement findFlowElement(String processDefinitionId, String taskDefinitionKey) {
    BpmnModel bpmnModel = repositoryService.getBpmnModel(processDefinitionId);
    Validate.notNull(bpmnModel, "未找到流程定义[%s]", processDefinitionId);
    Process process = bpmnModel.getProcesses().get(0);
    return process.getFlowElement(taskDefinitionKey);
  }

  @Override
  public String findProcessXmlById(String id) {
    if(StringUtils.isBlank(id)) return null;
    Optional<ProcessTemplateEntity> op = processTemplateRepository.findById(id);
    ProcessTemplateEntity template = op.orElse(null);
    Validate.notNull(template, "未找到流程模版对象");

    byte[] bytes = kuiperFileService.readFileContent(template.getFilePath(), template.getFileName());
    if(bytes == null || bytes.length == 0) return null;
    return new String(bytes, StandardCharsets.UTF_8);
  }

  @Override
  public BpmnModel findBpmnById(String id) {
    Optional<ProcessTemplateEntity> op = processTemplateRepository.findById(id);
    ProcessTemplateEntity template = op.orElse(null);
    Validate.notNull(template, "未找到流程模版对象");

    byte[] bytes = kuiperFileService.readFileContent(template.getFilePath(), template.getFileName());
    if(bytes == null || bytes.length == 0) return null;
    InputStreamProvider inputStreamProvider = new BytesStreamSource(bytes);
    BpmnXMLConverter bpmnXMLConverter = new BpmnXMLConverter();
    BpmnModel bpmnModel;
    try {
      bpmnModel = bpmnXMLConverter.convertToBpmnModel(inputStreamProvider, true, false, "UTF-8");
    } catch (Exception e){
      LOGGER.warn(e.getMessage(), e);
      return null;
    }
    return bpmnModel;
  }

  /**
   * 流程部署前流程验证
   * @param processTemplate
   */
  private void deployValidation(ProcessTemplateEntity processTemplate){
    { // 验证模版基础信息
      Validate.notNull(processTemplate, "流程模版不存在");
      Validate.isTrue(processTemplate.getProcessState().equals(ProcessTemplateState.DRAFT.getState()), "流程模版已发布，不能重复发布");
      Validate.notBlank(processTemplate.getProcessKey(), "流程Key不能为空");
      Validate.notBlank(processTemplate.getProcessName(), "流程名称不能为空");
      Validate.notBlank(processTemplate.getProcessXml(), "流程文件不能为空");
      Validate.notNull(processTemplate.getFormType(), "表单类型不能为空");
      if(ProcessFormType.FORM_TEMPLATE.getType().equals(processTemplate.getFormType())){
        Validate.notBlank(processTemplate.getFormTemplateId(), "表单模版Id不能为空");
      } else if(ProcessFormType.FORM_URL.getType().equals(processTemplate.getFormType())){
        Validate.notBlank(processTemplate.getFormUrl(), "表单地址不能为空");
      }
    }
    { // 验证流程节点信息
      Set<ProcessTemplateNodeEntity> nodes = processTemplate.getProcessTemplateNodes();
      Validate.isTrue(!CollectionUtils.isEmpty(nodes), "流程必须包含一个节点");
      Set<String> nodeIds = new HashSet<>();
      for (ProcessTemplateNodeEntity node : nodes) {
        Validate.notBlank(node.getProcessNodeId(), "节点ID不能为空");
        Validate.notBlank(node.getProcessNodeName(), "节点名称不能为空");
        if(ProcessFormType.FORM_TEMPLATE.getType().equals(processTemplate.getFormType())){
          Validate.notBlank(node.getFormVisibilityName(), "流程节点必须选择表单可见性");
        }
        Validate.isTrue(!nodeIds.contains(node.getProcessNodeId()), "节点ID不能重复");
        nodeIds.add(node.getProcessNodeId());
        // 验证节点的会签配置
        ProcessTemplateNodeMultiEntity nodeMulti = node.getProcessTemplateNodeMulti();
        if(nodeMulti != null){
          Validate.notNull(nodeMulti.getMiType(), "会签类型不能为空");
          if(!nodeMulti.getMiType().equals(MultiType.NONE_MULTI.getType())){
            // 会签情况下验证配置
            Validate.notNull(nodeMulti.getPresetMiAssignments(), "是否预置会签人员配置不能为空");
            Validate.notNull(nodeMulti.getCompletionCondition(), "会签通过条件不能为空");
            if(nodeMulti.getPresetMiAssignments()){
              Validate.notEmpty(nodeMulti.getAssignments(), "未配置会签人员");
              for (ProcessAssignmentEntity assignment : nodeMulti.getAssignments()) {
                processAssignmentService.valid(assignment.getAssignment());
              }
            }
          }
        }
        boolean isMultiAssignment = nodeMulti == null || nodeMulti.getMiType().equals(MultiType.NONE_MULTI.getType())
            || (!nodeMulti.getMiType().equals(MultiType.NONE_MULTI.getType()) && CollectionUtils.isEmpty(nodeMulti.getAssignments()));
        if(isMultiAssignment){
          // 当该节点不是会签节点或者没有设置预置会签人员时需要验证审批人配置信息
          ProcessTemplateNodeAssignmentEntity assignment = node.getAssignment();
          Validate.notNull(assignment, "流程节点必须配置审批人");
          Validate.notNull(assignment.getType(), "审批人类型不能为空");
          if(assignment.getType().equals(ProcessTemplateNodeAssignmentType.USERNAME.getType())
              || assignment.getType().equals(ProcessTemplateNodeAssignmentType.EXPRESSION.getType())){
            Validate.notNull(assignment.getVariables(), "流程节点[%s]未配置审批变量", node.getProcessNodeId());
          } else if(assignment.getType().equals(ProcessTemplateNodeAssignmentType.LISTENER.getType())){
            Validate.notNull(assignment.getProcessListener(), "流程节点[%s]未配置审批人监听器", node.getProcessNodeId());
            if(assignment.getProcessListener().getExecuteMode().equals(ProcessListenerExecuteMode.SCRIPT.getMode())){
              Validate.notNull(assignment.getScript(), "流程节点[%s]未配置审批人监听器脚本", node.getProcessNodeId());
            }
          }
        }
      }
    }
    { // 验证流程模版的权限信息
      Validate.notNull(processTemplate.getPermissions(), "流程模版未配置权限信息");
      Set<ProcessTemplatePermissionEntity> permissions = processTemplate.getPermissions();
      for (ProcessTemplatePermissionEntity permission : permissions) {
        if(permission.getType() > 0){
          Validate.notBlank(permission.getPermissionValue(), "流程模版未配置权限值");
        }
      }
    }
    { // 验证流程模版的监听器
      if(!CollectionUtils.isEmpty(processTemplate.getProcessTemplateListeners())){
        for (ProcessTemplateListenerEntity listener : processTemplate.getProcessTemplateListeners()) {
          Validate.notBlank(listener.getTargetId(), "监听器配置流程元素ID不能为空");
          Validate.notNull(listener.getProcessListener(), "监听器配置监听器不能为空");
          if(listener.getProcessListener().getExecuteMode().equals(ProcessListenerExecuteMode.SCRIPT.getMode())){
            Validate.notNull(listener.getScript(), "脚本监听器未配置脚本信息");
          }
        }
      }
    }
    // 解析xml文件较慢，放到最后
    this.validateProcessXml(processTemplate, true);
  }


}
