package com.bizunited.platform.titan.starter.service.internal;

import com.bizunited.platform.titan.entity.ProcessVariableEntity;
import com.bizunited.platform.titan.starter.repository.ProcessVariableRepository;
import com.bizunited.platform.titan.starter.service.ProcessVariableService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: Paul Chan
 * @Date: 2019-05-21 10:10
 * @Description: 流程变量服务接口实现
 */
@Service("ProcessVariableServiceImpl")
public class ProcessVariableServiceImpl implements ProcessVariableService {

  @Autowired
  private ProcessVariableRepository processVariableRepository;

  private void saveValidation(Set<ProcessVariableEntity> variables){
    Set<String> names = new HashSet<>();
    for (ProcessVariableEntity variable : variables) {
      Validate.notNull(variable.getName(), "变量名不能为null");
      Validate.notNull(variable.getValue(), "变量值不能为null");
      Validate.isTrue(!names.contains(variable.getName()), "变量名[%s]重复,请检查", variable.getName());
      names.add(variable.getName());
    }
  }

  @Override
  @Transactional
  public Set<ProcessVariableEntity> save(String sourceId, Set<ProcessVariableEntity> variables) {
    Validate.notBlank(sourceId, "数据源ID不能为空");
    if(variables == null) variables = new HashSet<>();
    saveValidation(variables);
    Set<ProcessVariableEntity> resultSet = new HashSet<>();
    // 将老变量和新变量缓存到map和set中
    Set<ProcessVariableEntity> oldVariables = processVariableRepository.findBySourceId(sourceId);
    Map<String, ProcessVariableEntity> oldVariablesMap = new HashMap<>(16);
    Map<String, ProcessVariableEntity> variablesMap = new HashMap<>(16);
    Set<String> oldVariableNames = new HashSet<>();
    Set<String> variableNames = new HashSet<>();
    oldVariables.forEach( v -> {
      oldVariablesMap.put(v.getName(), v);
      oldVariableNames.add(v.getName());
    });
    variables.forEach( v -> {
      variablesMap.put(v.getName(), v);
      variableNames.add(v.getName());
    });
    // 过滤出新增的变量
    Set<String> newVariablesSet = Sets.difference(variableNames, oldVariableNames);
    // 过滤出删除的变量
    Set<String> deleteVariablesSet = Sets.difference(oldVariableNames, variableNames);
    // 过滤出更新的变量
    Set<String> updateVariablesSet = Sets.intersection(variableNames, oldVariableNames);
    // 保存新增的变量
    for (String name : newVariablesSet) {
      ProcessVariableEntity variable = variablesMap.get(name);
      variable.setId(null);
      variable.setSourceId(sourceId);
      variable.setCreateTime(new Date());
      variable.setModifyTime(new Date());
      processVariableRepository.save(variable);
      resultSet.add(variable);
    }
    // 保存更新的变量
    for (String name : updateVariablesSet) {
      ProcessVariableEntity oldVariable = oldVariablesMap.get(name);
      ProcessVariableEntity variable = variablesMap.get(name);
      if(variable.getValue().equals(oldVariable.getValue())) {
        resultSet.add(oldVariable);
        continue;
      }
      oldVariable.setModifyTime(new Date());
      oldVariable.setValue(variable.getValue());
      processVariableRepository.save(oldVariable);
      resultSet.add(oldVariable);
    }
    // 删除不存在的变量
    for (String name : deleteVariablesSet) {
      ProcessVariableEntity oldVariable = oldVariablesMap.get(name);
      processVariableRepository.delete(oldVariable);
    }
    return resultSet;
  }

  @Override
  @Transactional
  public void deleteBySourceId(String sourceId) {
    processVariableRepository.deleteBySourceId(sourceId);
  }

  @Override
  public Set<ProcessVariableEntity> findBySourceId(String sourceId) {
    return processVariableRepository.findBySourceId(sourceId);
  }

  @Override
  public Map<String, Object> getVariablesByTargetId(String targetId) {
    Map<String, Object> variables = new HashMap<>(16);
    Set<ProcessVariableEntity> variableEntities = findBySourceId(targetId);
    if(!CollectionUtils.isEmpty(variableEntities)){
      variables = variableEntities.stream().collect(Collectors.toMap(ProcessVariableEntity::getName, ProcessVariableEntity::getValue, (a, b) -> b, () -> new HashMap<>(16)));
    }
    return variables;
  }


}
