package com.bizunited.platform.user.web.starter.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

@RestController
@RequestMapping(value = "/v1/nebula/orgs")
public class OrganizationController extends BaseController {

  @Autowired
  private OrganizationService organizationService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(OrganizationController.class);

  @ApiOperation(value = "添加组织机构")
  @PostMapping("")
  public ResponseModel create(@ApiParam(name = "组织机构相关信息") @RequestBody OrganizationVo organization) {
    try {
      OrganizationVo organizationVo = this.organizationService.create(organization);
      return this.buildHttpResultW(organizationVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改组织机构")
  @PatchMapping("/update")
  public ResponseModel update(@ApiParam(name = "组织机构相关信息") @RequestBody OrganizationVo organizationEntity) {
    try {
      OrganizationVo organizationVo = this.organizationService.update(organizationEntity);
      return this.buildHttpResultW(organizationVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改组织机构状态---(启用、禁用)")
  @PatchMapping("/updateStatus/{orgId}")
  public ResponseModel updateStatus(@PathVariable("orgId") String orgId) {
    LOGGER.debug("组织机构id:{}", orgId);
    try {
      OrganizationVo organization = this.organizationService.updateStatus(orgId);
      return this.buildHttpResultW(organization);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "对一个指定的currentOrgId进行其父级组织机构parentOrgId的绑定操作（原有currentOrgId的父级信息将会被取消）")
  @PatchMapping("/bindParent")
  public ResponseModel bindParent(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId,
                                  @RequestParam(name = "parentOrgId") @ApiParam(name = "parentOrgId", value = "当前需要绑定的父级组织机构编号") String parentOrgId) {
    try {
      this.organizationService.bindParent(currentOrgId, parentOrgId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "对一个指定的currentOrgId取消其当前父级关系的绑定信息")
  @PatchMapping("/unbindParent")
  public ResponseModel unbindParent(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId) {
    try {
      this.organizationService.unbindParent(currentOrgId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "该服务方法可以在忽略当前组织机构（currentOrgId）已绑定父级机构的情况下，重新为当前组织机构绑定一个新的父级机构")
  @PatchMapping("/reBindParent")
  public ResponseModel reBindParent(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId,
                                    @RequestParam(name = "parentOrgId", required = false) @ApiParam(name = "parentOrgId", value = "当前需要新绑定的父级组织机构编号") String parentOrgId) {
    try {
      this.organizationService.reBindParent(currentOrgId, parentOrgId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "将指定组织机构（currentOrgId）和指定的一个/多个用户数据编号（userId）形成绑定关系——一个用户只能属于一个组织机构", notes = "注意，目前一个人员可以和多个组织机构进行绑定，但是呢一个人员最多只能和一个组织机构绑定一次")
  @RequestMapping(value = "/bindUsers", method = RequestMethod.PATCH)
  public ResponseModel bindUsers(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId,
                                 @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）") String[] userIds) {
    try {
      this.organizationService.bindUsers(currentOrgId, userIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解除指定用户和指定组织机构的绑定关系")
  @RequestMapping(value = "/unbindUsers", method = RequestMethod.PATCH)
  public ResponseModel unbindUsers(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "指定的组织机构数据库编号") String currentOrgId, @RequestParam(name = "userId") @ApiParam(name = "userId", value = "当前需要操作的人员id信息（多个）") String[] userId) {
    try {
      this.organizationService.unbindUsers(currentOrgId, userId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 重新绑定组织机构的用户
   * @param currentOrgId
   * @param userIds
   * @return
   */
  @ApiOperation("重新绑定组织机构的用户")
  @PatchMapping("rebindUsers")
  public ResponseModel rebindUsers(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId,
                                   @RequestParam(name = "userIds", required = false) @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）") String[] userIds) {
    try {
      this.organizationService.rebindUsers(currentOrgId, userIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询目前没有指定父级组织机构的所有组织机构信息（查询所有根节点组织机构），无论其状态是否正常（status无论什么值）", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByNullParent", method = RequestMethod.GET)
  public ResponseModel findByNullParent() {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByNullParent();
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定父级组织机构下的所有子级组织机构信息，无论其状态是否正常（status无论什么值）", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByParent", method = RequestMethod.GET)
  public ResponseModel findByParent(@RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级组织机构id") String parentId) {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByParent(parentId);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 根据条件查询树结构
   * 返回的数据，除了返回匹配的数据外，还包含匹配数据的所有上级
   * @return
   */
  @GetMapping("findTreeByConditions")
  @ApiOperation(value = "根据条件查询树结构", notes = "返回的数据，除了返回匹配的数据外，还包含匹配数据的所有上级")
  public ResponseModel findTreeByConditions(@RequestParam(required = false) @ApiParam("编码") String code,
                                            @RequestParam(required = false) @ApiParam("名称") String orgName,
                                            @RequestParam(required = false) @ApiParam("状态") Integer tstatus) {
    try {
      Map<String, Object> parameters = new HashMap<>();
      parameters.put("code", code);
      parameters.put("orgName", orgName);
      parameters.put("tstatus", tstatus);
      List<OrganizationVo> resutls = this.organizationService.findTreeByConditions(parameters);
      return this.buildHttpResultW(resutls, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询目前没有指定父级组织机构的所有组织机构信息，并且依据其状态进行查询", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByNullParentAndStatus", method = RequestMethod.GET)
  public ResponseModel findByNullParentAndStatus(@RequestParam(name = "status") @ApiParam(name = "status", value = "状态信息（1:正常；0禁用/异常）") Integer status) {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByNullParentAndStatus(status);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定父级组织机构下的所有子级组织机构信息，并且依据其状态进行查询", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByParentAndStatus", method = RequestMethod.GET)
  public ResponseModel findByParentAndStatus(@RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级组织机构id") String parentId, @RequestParam(name = "status") @ApiParam(name = "status", value = "状态信息（1:正常；0禁用/异常）") Integer status) {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByParentAndStatus(parentId, status);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据组织机构名称，查询这个组织机构以及这个组织（这些组织机构）对应的所有父级、子级结构信息")
  @RequestMapping(value = "/findByOrgName", method = RequestMethod.GET)
  public ResponseModel findByOrgName(@RequestParam(name = "orgName") @ApiParam(name = "orgName", value = "组织机构名称信息") String orgName) {
    try {
      JSONArray resutls = this.organizationService.findByOrgName(orgName);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定组织下的详细信息，无论其状态是否正常（status无论什么值）", notes = "注意：包括关联的用户信息")
  @RequestMapping(value = "/findDetailsById", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@RequestParam(name = "id") @ApiParam(name = "id", value = "组织机构id") String id) {
    try {
      OrganizationVo organization = this.organizationService.findDetailsById(id);
      return this.buildHttpResultW(organization, "parent","child","users", "positions");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据指定组织机构，查询这个组织机构以及这个组织（这些组织机构）对应的所有子级结构信息")
  @RequestMapping(value = "/findUnbindOrgById", method = RequestMethod.GET)
  public ResponseModel findUnbindOrgById(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前组织机构") String currentOrgId) {
    try {
      Object[] resutls = this.organizationService.findUnbindOrgById(currentOrgId);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询所有的组织机构信息，无论其状态是否正常（status无论什么值）", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      List<OrganizationVo> resutls = this.organizationService.findAll();
      return this.buildHttpResultW(resutls, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取用户的主组织机构
   * @param userId
   * @return
   */
  @GetMapping("findMainOrgByUserId")
  public ResponseModel findMainOrgByUserId(@RequestParam @ApiParam("用户ID") String userId) {
    try {
      OrganizationVo org = this.organizationService.findMainOrgByUserId(userId);
      return this.buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "预制控件查询，查询用户组织机构相关信息")
  @RequestMapping(value = "/findOrgByUserId",method = RequestMethod.GET)
  public ResponseModel findOrgByUserId(@ApiParam(name = "userId", value = "用户id信息") @RequestParam("userId")String userId){
    try {
      Set<OrganizationVo> org = this.organizationService.findOrgByUserId(userId);
      return this.buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询用户组织机构相关信息")
  @RequestMapping(value = "/findByIds",method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids){
    try {
      Set<OrganizationVo> org = this.organizationService.findByIds(ids);
      return this.buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 组织机构-树结构查询-根据指定组织机构ID，查询这个组织机构以及这个组织机构对应的所有子级结构信息
   * @param id 当前组织机构Id
   * @return
   */
  @ApiOperation("组织机构-树结构查询-根据指定组织机构ID，" +
          "查询这个组织机构以及这个组织机构对应的所有子级结构信息")
  @GetMapping("/findById")
  public ResponseModel findById(@RequestParam(value = "id") @ApiParam(name = "id", value = "当前组织机构Id") String id){
    try {
      OrganizationVo result = this.organizationService.findById(id);
      return this.buildHttpResult(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 组织机构-根据组织机构名称实现模糊查询
   * @param orgName 组织机构名称
   * @return
   */
  @ApiOperation("组织机构-根据组织机构名称实现模糊查询")
  @GetMapping("/findByOrgNameLike")
  public ResponseModel findByOrgNameLike(@RequestParam(name = "orgName") @ApiParam(name = "orgName", value = "组织机构名称") String orgName){
    try {
      Set<OrganizationVo> organization = this.organizationService.findByOrgNameLike(orgName);
      return this.buildHttpResultW(organization);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据组织编码查询
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  @ApiOperation("根据组织编码查询")
  public ResponseModel findByCode(@RequestParam @ApiParam("组织编码") String code) {
    try {
      OrganizationVo org = organizationService.findByCode(code);
      return buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据指定的用户信息，查询这个用户所绑定的组织机构信息(一个或者多个)，以及这个组织机构的所有父级组织机构信息
   * 由于这里的构造是无限级别的，所以需要在方法内做递归处理，且直接返回json信息。<p>
   * 注意：这些组织机构的返回信息中，包括了组织机构已经绑定的角色信息。<p>
   * 一定注意该方法和RoleService.findByUserId(String userId)方法在实现逻辑上的区别。
   * @return
   */
  @ApiOperation(value = "根据指定的用户信息，查询这个用户所绑定的组织机构信息(一个或者多个)，以及这个组织机构的所有父级组织机构信息")
  @GetMapping("/findByUserId")
  public ResponseModel findByUserId(@RequestParam(name = "userId") String userId){
    try {
      JSONArray org = this.organizationService.findByUserId(userId);
      return this.buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取组织的上级组织
   * @param childId
   * @return
   */
  @GetMapping("findByChild")
  @ApiOperation("根据组织编码查询")
  public ResponseModel findByChild(@RequestParam(name = "childId") String childId) {
    try {
      OrganizationVo org = organizationService.findByChild(childId);
      return buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 更新用户的主组织机构
   * @param userId
   * @param orgId
   * @return
   */
  @PatchMapping("updateUserMainOrg")
  @ApiOperation("更新用户的主组织机构")
  public ResponseModel updateUserMainOrg(@RequestParam @ApiParam("用户ID") String userId,
                                         @RequestParam @ApiParam("组织机构ID") String orgId) {
    try {
      organizationService.updateUserMainOrg(userId, orgId);
      return buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }


  /**
   * 根据登陆人关联的组织查询控件
   * @param type
   * @return
   */
  @GetMapping("findByPrincipal")
  @ApiOperation("根据登陆人关联的组织查询控件")
  public ResponseModel findByPrincipal(@RequestParam(required = true) @ApiParam("类型, 1:关联组织上级, 2：关联组织下级, 3：关联组织上下级, 4：关联组织") Integer type) {
    try {
      Principal principal = this.getPrincipal();
      Set<OrganizationVo> orgs = organizationService.findByPrincipal(type, principal);
      return buildHttpResult(orgs);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
