package com.bizunited.platform.user.web.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * PositionController
 *
 * @description:
 * @author: yanwe
 * @date: 18/Jan/2019 10:00
 */
@RestController
@RequestMapping(value = "/v1/nebula/position")
public class PositionController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(PositionController.class);
  @Autowired 
  private PositionService positionService;
  
  @ApiOperation(value = "按照条件搜索用户，不支持模糊查询")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(
      @ApiParam(name = "name", value = "名称") String name,
      @ApiParam(name = "code", value = "编码") String code,
      @ApiParam(name = "tstatus", value = "状态") Integer tstatus,
      @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<PositionVo> page = positionService.findByConditions(code, name, tstatus, pageable);
      return this.buildHttpResultW(page, "organization", "users", "roles", "parent", "children");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据ID查询岗位的详细信息，包括相关组织、角色、用户等")
  @RequestMapping(value = "/find/{positionId}", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@PathVariable("positionId") String positionId) {
    try {
      PositionVo positionVo = positionService.findDetailsById(positionId);
      return buildHttpResultW(positionVo, "organization", "users", "roles");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "添加岗位")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@ApiParam(name = "岗位相关信息") @RequestBody PositionVo position) {
    try {
      PositionVo positionVo = positionService.create(position);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改岗位")
  @RequestMapping(value = "update", method = RequestMethod.PATCH)
  public ResponseModel update(@ApiParam(name = "岗位相关信息") @RequestBody PositionVo position) {
    try {
      PositionVo positionVo = positionService.update(position);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改岗位状态（启用禁用）")
  @RequestMapping(value = "updateStatus/{positionId}", method = RequestMethod.PATCH)
  public ResponseModel updateStatus(@PathVariable("positionId") String positionId) {
    try {
      PositionVo positionVo = positionService.updateStatus(positionId);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "绑定指定的岗位和指定的组织机构")
  @RequestMapping(value = "/bindOrg", method = RequestMethod.PATCH)
  public ResponseModel bindOrg(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "orgId") @ApiParam(name = "orgId", value = "指定的组织机构（orgId）")
          String orgId) {
    try {
      positionService.bindOrg(orgId, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解绑指定的岗位和任何组织机构的绑定关系")
  @RequestMapping(value = "/unbindOrg", method = RequestMethod.PATCH)
  public ResponseModel unbindOrg(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID") String positionId) {
    try {
      positionService.unbindOrg(positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "在忽略指定岗位是否已经绑定组织机构的前提下，重新为这个指定的岗位绑定新的组织机构")
  @RequestMapping(value = "/reBindOrg", method = RequestMethod.PATCH)
  public ResponseModel reBindOrg(
       @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
           String positionId,
       @RequestParam(name = "orgId") @ApiParam(name = "orgId", value = "指定的组织机构（orgId）")
           String orgId) {
    try {
      positionService.reBindOrg(orgId, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "将指定一个岗位和指定的多个用户形成绑定关系")
  @RequestMapping(value = "/bindUsersAndPosition", method = RequestMethod.PATCH)
  public ResponseModel bindUsers(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "当前的岗位ID")
          String positionId,
      @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）")
          String[] userIds) {
    try {
      positionService.bindUsersAndPosition(Arrays.asList(userIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解绑多个用户与某一指定的岗位")
  @RequestMapping(value = "/unbindUsersAndPosition", method = RequestMethod.PATCH)
  public ResponseModel unbindUsersAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的用户ID")
          String[] userIds) {
    try {
      positionService.unbindUsersAndPosition(Arrays.asList(userIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "预制控件查询，查询用户岗位相关信息")
  @RequestMapping(value = "/findByUserId",method = RequestMethod.GET)
  public ResponseModel findByUserId(@ApiParam(name = "userId", value = "用户id") @RequestParam("userId")String userId){
    try {
      List<PositionVo> position = this.positionService.findByUserId(userId);
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询多个用户岗位相关信息")
  @RequestMapping(value = "/findByIds",method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids){
    try {
      Set<PositionVo> position = this.positionService.findByIds(ids);
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据岗位编码查询
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  @ApiOperation("根据岗位编码查询")
  public ResponseModel findByCode(@RequestParam @ApiParam("岗位编码") String code) {
    try {
      PositionVo position = positionService.findByCode(code);
      return buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 切换岗位
   * @param positionId
   * @param userId
   * @return
   */
  @PostMapping("changePosition")
  @ApiOperation("切换岗位")
  public ResponseModel changePosition(@RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "岗位id", required=true) String positionId, @RequestParam(name = "userId") @ApiParam(name = "userId", value = "指定的用户ID", required=true) String userId){
    try{
      UserVo userVo = this.positionService.changePosition(positionId, userId);
      return buildHttpResultW(userVo, "roles", "positions");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   * @param userId
   * @return
   */
  @GetMapping("findMainPositionByUserId")
  @ApiOperation("查询主岗位")
  public ResponseModel findMainPositionByUserId(@RequestParam @ApiParam("用户id") String userId) {
    try {
      PositionVo position = positionService.findMainPositionByUserId(userId);
      return buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "将指定一个岗位和指定的多个用户重新绑定关系")
  @RequestMapping(value = "/rebindUsers", method = RequestMethod.PATCH)
  public ResponseModel rebindUsers(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "当前的岗位ID")
          String positionId,
      @RequestParam(name = "userIds", required = false) @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）")
          String[] userIds) {
    try {
      positionService.rebindUsers(userIds, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询所有岗位信息（根据创建时间排序）
   * @return
   */
  @ApiOperation(value = "查询所有岗位信息（根据创建时间排序）")
  @GetMapping("/findAll")
  public ResponseModel findAll(){
    try{
      List<PositionVo> positions = positionService.findAll();
      return buildHttpResultW(positions, "parent");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据条件查询所有岗位
   * @param status
   * @return
   */
  @GetMapping("findByStatus")
  @ApiOperation("根据状态查询岗位")
  public ResponseModel findByStatus(@RequestParam(required = false) @ApiParam("状态") Integer status) {
    try {
      List<PositionVo> positions = positionService.findByStatus(status);
      return buildHttpResultW(positions);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
