package com.bizunited.platform.user.web.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Set;

@Api(value = "UserGroupController")
@RestController
@RequestMapping("/v1/nebula/userGroups")
public class UserGroupController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(UserGroupController.class);
  @Autowired
  private UserGroupService userGroupService;
  
  @ApiOperation(value = "根据指定用户组名称、状态查询")
  @RequestMapping(value = "/findByCondition", method = RequestMethod.GET)
  public ResponseModel findByCondition(@ApiParam(name = "groupName", value = "可能的用户组名称查询条件" , required=false) @RequestParam(required=false , name="groupName") String groupName,
                                       @ApiParam(name = "groupCode", value = "可能的用户组编号查询条件")@RequestParam(name = "groupCode", required = false) String groupCode,
                                       @ApiParam(name = "status", value = "可能的用户组状态查询条件" , required=false) @RequestParam(required=false , name="status") Integer status,
                                       @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<UserGroupVo> userGroup = userGroupService.findByCondition(groupName, groupCode, status, pageable);
      return this.buildHttpResultW(userGroup);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "新增用户组")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@ApiParam(name = "userGroup") @RequestBody UserGroupVo userGroup) {
    try {
      UserGroupVo group = this.userGroupService.create(userGroup);
      return this.buildHttpResultW(group);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改用户组" , notes="只能修改用户组描述")
  @RequestMapping(value = "/update", method = RequestMethod.PATCH)
  public ResponseModel update(@ApiParam(name = "用户组相关信息") @RequestBody UserGroupVo userGroupVo) {
    try {
      UserGroupVo userGroup = this.userGroupService.update(userGroupVo);
      return this.buildHttpResultW(userGroup);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改用户组状态")
  @RequestMapping(value = "/updateStatus/{groupId}", method = RequestMethod.PATCH)
  public ResponseModel updateStatus(@PathVariable("groupId") String groupId) {
    try {
      UserGroupVo userGroup = this.userGroupService.updateStatus(groupId);
      return this.buildHttpResultW(userGroup);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "用户与用户组绑定" , notes="可以绑定多个用户id，不过只要有一个用户id绑定错误，则整个绑定过程失败")
  @RequestMapping(value = "/bindUser", method = RequestMethod.PATCH)
  public ResponseModel bindUser(@ApiParam(name = "groupId", value = "用户组id") @RequestParam("groupId") String groupId,
                                @ApiParam(name = "userIds", value = "用户id(可以是多个)") @RequestParam("userIds") String[] userIds) {
    try {
      this.userGroupService.bindUser(groupId, userIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "用户与用户组解除绑定", notes="可以解绑多个用户id，不过只要有一个用户id绑定错误，则整个绑定过程失败")
  @RequestMapping(value = "/unBindUser", method = RequestMethod.PATCH)
  public ResponseModel unBindUser(@ApiParam(name = "groupId", value = "用户组id") @RequestParam("groupId") String groupId, 
                                  @ApiParam(name = "userIds", value = "用户id") @RequestParam("userIds") String[] userIds) {
    try {
      this.userGroupService.unBindUser(groupId, userIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 用户与用户组的重新绑定
   * @param groupId
   * @param userIds
   * @return
   */
  @ApiOperation(value = "用户与用户组的重新绑定" , notes="可以绑定多个用户id，不过只要有一个用户id绑定错误，则整个绑定过程失败")
  @PatchMapping("/reBindUser")
  public ResponseModel reBindUser(@ApiParam(name = "groupId", value = "用户组id") @RequestParam(name = "groupId") String groupId,
                                @ApiParam(name = "userIds", value = "用户id(可以是多个)", required = false) @RequestParam(name = "userIds", required = false) String[] userIds) {
    try {
      this.userGroupService.reBindUser(groupId, userIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定用户组的基本详情")
  @RequestMapping(value = "/findDetailsById", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@ApiParam(name = "id", value = "用户组id") @RequestParam("id") String id) {
    LOGGER.debug("用户组:{}", id);
    try {
      Set<UserGroupVo> userGroupDetail = userGroupService.findDetailsById(id);
      return this.buildHttpResultW(userGroupDetail, "users","roles");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询多个用户组基本信息")
  @RequestMapping(value = "/findByIds", method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids) {
    LOGGER.debug("用户组:{}", ids);
    try {
      Set<UserGroupVo> userGroups = userGroupService.findByIds(ids);
      return this.buildHttpResultW(userGroups);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据分组名称查询
   * @param groupName
   * @return
   */
  @GetMapping("findByGroupName")
  @ApiOperation("根据分组名称查询")
  public ResponseModel findByGroupName(@RequestParam @ApiParam("分组名称") String groupName) {
    try {
      UserGroupVo userGroup = userGroupService.findByGroupName(groupName);
      return buildHttpResultW(userGroup);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 查询所有用户组信息（根据创建时间排序）
   * @return
   */
  @GetMapping("findAll")
  @ApiOperation("查询所有用户组信息")
  public ResponseModel findAll(){
    try{
      List<UserGroupVo> userGroups = userGroupService.findAll();
      return buildHttpResultW(userGroups, "users", "roles");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 按照指定的人员数据库编号，查询人员已经绑定的一个或者多个用户组信息。<p>
   * 注意：用户组信息中包括了用户组已经绑定的角色信息。
   * @param userId
   * @return
   */
  @GetMapping("findByUserId")
  @ApiOperation("根据分组名称查询")
  public ResponseModel findByUserId(@RequestParam("userId") @ApiParam("用户id") String userId) {
    try {
      Set<UserGroupVo> userGroups = userGroupService.findByUserId(userId);
      return buildHttpResultW(userGroups);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据分组编码查询
   * @param groupCode
   * @return
   */
  @GetMapping("findByCode")
  @ApiOperation("根据分组编码查询")
  public ResponseModel findByCode(@RequestParam @ApiParam("分组编码") String groupCode) {
    try {
      UserGroupVo userGroup = userGroupService.findByCode(groupCode);
      return buildHttpResultW(userGroup);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据用户组编码查询多个用户组信息
   * @param groupCodes
   * @return
   */
  @GetMapping("findByGroupCodes")
  @ApiOperation("根据用户组编码查询多个用户组信息")
  public ResponseModel findByGroupCodes(@RequestParam("groupCodes") @ApiParam("户组编码集合") List<String> groupCodes) {
    try {
      List<UserGroupVo> userGroups = userGroupService.findByGroupCodes(groupCodes);
      return buildHttpResultW(userGroups);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
