package com.bizunited.platform.user.web.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.user.common.service.region.AdministrativeRegionService;
import com.bizunited.platform.user.common.vo.AdministrativeRegionVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * AdministrativeRegionEntity业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author saturn
 */
@RestController
@RequestMapping("/v1/nebula/regions")
public class AdministrativeRegionController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(AdministrativeRegionController.class);

  @Autowired
  private AdministrativeRegionService administrativeRegionService;

  /**
   * 新增行政区域
   * @param region
   * @return
   */
  @PostMapping
  @ApiOperation("新增行政区域")
  public ResponseModel create(@RequestBody AdministrativeRegionVo region) {
    try {
      AdministrativeRegionVo dbRegion = administrativeRegionService.create(region);
      return buildHttpResultW(dbRegion, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  @PatchMapping("updateOrgMappingLevel")
  @ApiOperation(value = "更新组织机构关联层级", notes = "关联层级:1、省 2、市 3、区县")
  public ResponseModel updateOrgMappingLevel(@RequestParam @ApiParam("关联层级") Integer level) {
    try {
      administrativeRegionService.updateOrgMappingLevel(level);
      return buildHttpResultW(level);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  @GetMapping("findOrgMappingLevel")
  @ApiOperation("获取组织机构关联层级")
  public ResponseModel findOrgMappingLevel() {
    try {
      Integer level = administrativeRegionService.findOrgMappingLevel();
      return buildHttpResultW(level);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照AdministrativeRegionEntity实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   * @param id 主键
   */
  @ApiOperation("按照AdministrativeRegionEntity实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @GetMapping("/findDetailsById")
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      AdministrativeRegionVo result = this.administrativeRegionService.findDetailsById(id);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照AdministrativeRegionEntity实体中的（id）主键进行查询，不带出明细记录。
   * @param id 主键
   */
  @ApiOperation("按照AdministrativeRegionEntity实体中的（id）主键进行查询，不带出明细记录。")
  @GetMapping("/findById")
  public ResponseModel findById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      AdministrativeRegionVo result = this.administrativeRegionService.findById(id);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 组合查询
   * 查询条件，若为空则查询所有顶级区域
   */
  @ApiOperation("组合查询")
  @GetMapping("/findByConditions")
  public ResponseModel findByConditions(@RequestParam(required = false) @ApiParam("区域名称") String regionName,
                                        @RequestParam(required = false) @ApiParam("区域编码") String regionCode,
                                        @RequestParam(required = false) @ApiParam("经度") String longitude,
                                        @RequestParam(required = false) @ApiParam("纬度") String latitude) {
    try {
      Map<String, Object> conditions = new HashMap<>();
      conditions.put("regionName", regionName);
      conditions.put("regionCode", regionCode);
      conditions.put("longitude", longitude);
      conditions.put("latitude", latitude);
      List<AdministrativeRegionVo> regionVos = this.administrativeRegionService.findByConditions(conditions);
      return this.buildHttpResult(regionVos);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据层级查询,结果不包含任何父子关系
   * @param regionLevel
   * @return
   */
  @ApiOperation("根据层级查询,结果不包含任何父子关系")
  @GetMapping("/findByRegionLevel")
  public ResponseModel findByRegionLevel(@RequestParam("regionLevel") @ApiParam("查询条件") Integer regionLevel) {
    try {
      Set<AdministrativeRegionVo> regionVos = this.administrativeRegionService.findByRegionLevel(regionLevel);
      return this.buildHttpResult(regionVos);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据组织ID查询关联的行政区域
   * @param orgId
   * @return
   */
  @GetMapping("findByOrgId")
  @ApiOperation("根据组织ID查询关联的行政区域")
  public ResponseModel findByOrgId(@RequestParam @ApiParam("组织机构ID") String orgId) {
    try {
      List<AdministrativeRegionVo> regions = administrativeRegionService.findByOrgId(orgId);
      return buildHttpResultW(regions);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据父ID查询子区域，如果不传父ID，则返回最顶层的区域
   * @param parentId
   * @return
   */
  @GetMapping("findByParent")
  @ApiOperation(value = "根据父ID查询子区域", notes = "如果不传父ID，则返回最顶层的区域")
  public ResponseModel findByParent(@RequestParam(required = false) @ApiParam("父级ID，可以为空") String parentId) {
    try {
      List<AdministrativeRegionVo> regions = administrativeRegionService.findByParent(parentId);
      return this.buildHttpResultW(regions);
    } catch (RuntimeException e) {
      return this.buildHttpResultForException(e);
    }
  }

  @GetMapping("findByCode")
  @ApiOperation("根据编码获取行政区域")
  public ResponseModel findByCode(@RequestParam @ApiParam("区域编码") String code) {
    try {
      AdministrativeRegionVo region = administrativeRegionService.findByCode(code);
      return this.buildHttpResultW(region);
    } catch (RuntimeException e) {
      return this.buildHttpResultForException(e);
    }
  }

}
