package com.bizunited.platform.user.web.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.PositionSimpleVo;
import com.bizunited.platform.user.common.vo.UserVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.security.Principal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * PositionController
 *
 * @description:
 * @author: yanwe
 * @date: 18/Jan/2019 10:00
 */
@RestController
@RequestMapping("/v1/nebula/position")
public class PositionController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(PositionController.class);
  @Autowired 
  private PositionService positionService;
  
  @ApiOperation(value = "按照条件搜索用户，不支持模糊查询",notes = "<strong>字段描述</strong><br>" +
          "code:职位编码，用于等于匹配<br>" +
          "nameEqual:职位名称，用于等于匹配<br>" +
          "tstatus:职位状态，用于等于匹配<br>" +
          "flag:编码、id是否查询集合范围内的标记，为1查询集合范围内，为0查询集合范围外<br>" +
          "ids:id集合<br>" +
          "codes:编码集合<br>" +
          "name:职位名称，用户模糊匹配<br>" +
          "userName:用户名，用于模糊匹配<br>" +
          "positionLevel:职级id<br>" +
          "orgName:组织名称，用于模糊匹配")
  @GetMapping("/findByConditions")
  public ResponseModel findByConditions(@ApiParam(value = "查询条件,注：如果要用‘==‘匹配职位名称，传nameEqual", example = "{\n" +
          "    \"code\": \"\",\n" +
          "    \"nameEqual\": \"\",\n" +
          "    \"tstatus\": \"\",\n" +
          "    \"flag\": \"\",\n" +
          "    \"ids\": [],\n" +
          "    \"codes\": [],\n" +
          "    \"name\": \"\",\n" +
          "    \"userName\": \"\",\n" +
          "    \"positionLevel\": ,\n" +
          "    \"orgName\": \"\"\n" +
          "  }") @RequestParam Map<String, Object> conditions, @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<PositionVo> page = positionService.findByConditions(conditions, pageable);
      return this.buildHttpResultW(page, "organization", "parent", "positionLevel");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("根据ID查询岗位的详细信息，包括相关组织、角色、用户等")
  @GetMapping("/find/{positionId}")
  public ResponseModel findDetailsById(@PathVariable("positionId") String positionId) {
    try {
      PositionVo positionVo = positionService.findDetailsById(positionId);
      return buildHttpResultW(positionVo, "organization" , "roles", "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("添加岗位")
  @PostMapping
  public ResponseModel create(@ApiParam("岗位相关信息") @RequestBody PositionVo position) {
    try {
      PositionVo positionVo = positionService.create(position);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("修改岗位")
  @PatchMapping("update")
  public ResponseModel update(@ApiParam("岗位相关信息") @RequestBody PositionVo position) {
    try {
      PositionVo positionVo = positionService.update(position);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("修改岗位状态（启用禁用）")
  @PatchMapping("updateStatus/{positionId}")
  public ResponseModel updateStatus(@PathVariable("positionId") String positionId) {
    try {
      PositionVo positionVo = positionService.updateStatus(positionId);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("绑定指定的岗位和指定的组织机构")
  @PatchMapping("/bindOrg")
  public ResponseModel bindOrg(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "orgId") @ApiParam(name = "orgId", value = "指定的组织机构（orgId）")
          String orgId) {
    try {
      positionService.bindOrg(orgId, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("解绑指定的岗位和任何组织机构的绑定关系")
  @PatchMapping("/unbindOrg")
  public ResponseModel unbindOrg(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID") String positionId) {
    try {
      positionService.unbindOrg(positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation("在忽略指定岗位是否已经绑定组织机构的前提下，重新为这个指定的岗位绑定新的组织机构")
  @PatchMapping("/reBindOrg")
  public ResponseModel reBindOrg(
       @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
           String positionId,
       @RequestParam(name = "orgId") @ApiParam(name = "orgId", value = "指定的组织机构（orgId）")
           String orgId) {
    try {
      positionService.reBindOrg(orgId, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("将指定一个岗位和指定的多个用户形成绑定关系")
  @PatchMapping("/bindUsersAndPosition")
  public ResponseModel bindUsers(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "当前的岗位ID")
          String positionId,
      @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）")
          String[] userIds) {
    try {
      positionService.bindUsersAndPosition(Arrays.asList(userIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("解绑多个用户与某一指定的岗位")
  @PatchMapping("/unbindUsersAndPosition")
  public ResponseModel unbindUsersAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的用户ID")
          String[] userIds) {
    try {
      positionService.unbindUsersAndPosition(Arrays.asList(userIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("预制控件查询，查询用户岗位相关信息")
  @GetMapping("/findByUserId")
  public ResponseModel findByUserId(@ApiParam(name = "userId", value = "用户id") @RequestParam("userId")String userId){
    try {
      List<PositionVo> position = this.positionService.findByUserId(userId);
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("预制控件查询，查询用户岗位相关信息")
  @GetMapping("/findByUserAccount")
  public ResponseModel findByUserAccount(@ApiParam(name = "userAccount", value = "用户账号") @RequestParam("userAccount") String userAccount) {
    try {
      List<PositionVo> position = this.positionService.findByUserAccount(userAccount);
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("查询多个用户岗位相关信息")
  @GetMapping("/findByIds")
  public ResponseModel findByIds(@RequestParam(value = "id数组", defaultValue = "[]") @ApiParam("id数组") String[] ids){
    try {
      Set<PositionVo> position = this.positionService.findByIds(Arrays.asList(ids));
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据岗位编码查询
   * @param code
   * @return
   */
  @ApiOperation("根据岗位编码查询")
  @GetMapping("findByCode")
  public ResponseModel findByCode(@RequestParam @ApiParam("岗位编码") String code) {
    try {
      PositionVo position = positionService.findByCode(code);
      return buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 切换岗位
   * @param positionId
   * @param userId
   * @return
   */
  @ApiOperation("切换岗位")
  @PostMapping("changePosition")
  public ResponseModel changePosition(@RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "岗位id", required=true) String positionId, @RequestParam(name = "userId") @ApiParam(name = "userId", value = "指定的用户ID", required=true) String userId){
    try{
      UserVo userVo = this.positionService.changePosition(positionId, userId);
      return buildHttpResultW(userVo, "roles", "positions");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   * @param userId
   * @return
   */
  @ApiOperation("查询主岗位")
  @GetMapping("findMainPositionByUserId")
  public ResponseModel findMainPositionByUserId(@RequestParam @ApiParam("用户id") String userId) {
    try {
      PositionVo position = positionService.findMainPositionByUserId(userId);
      return buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("将指定一个岗位和指定的多个用户重新绑定关系")
  @PatchMapping("/rebindUsers")
  public ResponseModel rebindUsers(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "当前的岗位ID")
          String positionId,
      @RequestParam(name = "userIds", required = false) @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）")
          String[] userIds) {
    try {
      positionService.rebindUsers(userIds, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询所有岗位信息（根据创建时间排序）
   * @return
   */
  @ApiOperation("查询所有岗位信息（根据创建时间排序）")
  @GetMapping("/findAll")
  public ResponseModel findAll(){
    try{
      List<PositionVo> positions = positionService.findAll();
      return buildHttpResultW(positions, "parent");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据条件查询所有岗位
   * @param status
   * @return
   */
  @ApiOperation("根据状态查询岗位")
  @GetMapping("findByStatus")
  public ResponseModel findByStatus(@RequestParam(required = false) @ApiParam("状态") Integer status) {
    try {
      List<PositionVo> positions = positionService.findByStatus(status);
      return buildHttpResultW(positions);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据登陆人关联的岗位查询控件
   * @param type
   * @return
   */
  @ApiOperation("根据登陆人关联的岗位查询控件")
  @GetMapping("findByPrincipal")
  public ResponseModel findByPrincipal(@RequestParam(required = true) @ApiParam("类型, 1:关联岗位上级, 2：关联岗位下级, 3：关联岗位上下级, 4：关联岗位") Integer type) {
    try {
      Principal principal = this.getPrincipal();
      Set<PositionVo> positions = positionService.findByPrincipal(type, principal);
      return buildHttpResult(positions);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据登陆人关联的岗位和输入层级查询控件
   * @param type
   * @param level
   * @return
   */
  @ApiOperation("根据登陆人关联的岗位和输入层级查询控件")
  @GetMapping("findByTypeAndLevel")
  public ResponseModel findByTypeAndLevel(@RequestParam(required = true) @ApiParam("类型, 1:岗位上级, 2：岗位下级") Integer type,
                                       @RequestParam(required = true) @ApiParam("层级") Integer level) {
    try {
      Principal principal = this.getPrincipal();
      Set<PositionVo> positions = positionService.findByTypeAndLevel(type, principal, level);
      return buildHttpResult(positions);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据职级创建相关职位
   * @param levelCode
   * @return
   */
  @ApiOperation("根据职级创建相关职位")
  @PostMapping("createByPositionLevelCode")
  public ResponseModel createByPositionLevelCode(@RequestParam @ApiParam("层级编码") String levelCode) {
    try {
      PositionVo position = positionService.createByPositionLevelCode(levelCode);
      return buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据组织编码相关信息（包括组织，职级，用户）
   * @param orgCode
   * @return
   */
  @ApiOperation("根据组织编码相关信息（包括组织，职级，用户）")
  @GetMapping("findDetailsByOrgCode")
  public ResponseModel findDetailsByOrgCode(@RequestParam(required = true) @ApiParam("组织编码") String orgCode) {
    try {
      List<PositionVo> positions = positionService.findDetailsByOrgCode(orgCode);
      return buildHttpResultW(positions, "positionLevel", "users", "organization", "children");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据编码查询职位的所有下级信息
   * @param code
   * @return
   */
  @ApiOperation("根据编码查询职位的所有下级信息")
  @GetMapping("findDetailsByCode")
  public ResponseModel findDetailsByCode(@RequestParam @ApiParam("编码") String code) {
    try {
      List<PositionVo> positions = positionService.findDetailsByCode(code);
      return buildHttpResultW(positions, "positionLevel", "users", "organization");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据职级分页查询职位，待出用户拓展以及组织
   * @param conditions
   * @param pageable
   * @return
   */
  @ApiOperation(value = "根据编码查询职位的所有下级信息", notes = "<strong>参数说明：</strong>" +
          "name: 职位名称\n" +
          "orgName: 组织名称\n" +
          "userName: 用户名称\n" +
          "positionLevel: 职级id\n" +
          "page: 页码\n" +
          "size: 每页查询数量")
  @GetMapping("findByPositionLevelAndConditions")
  public ResponseModel findByPositionLevelAndConditions(@ApiParam("参数") Map<String,Object> conditions,
                                              @PageableDefault(value = 50)@ApiParam(name = "pageable", value = "分页信息") Pageable pageable) {
    try {
      Page<PositionSimpleVo> result = positionService.findByPositionLevelAndConditions(pageable, conditions);
      return buildHttpResultW(result,new String[]{});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据职位Id批量禁用职位
   * @param ids 职位Id列表
   */
  @ApiOperation("根据职位Id批量禁用职位")
  @PatchMapping("disableByIds")
  public ResponseModel disableByIds(@RequestParam @ApiParam("id数组") String[] ids) {
    try {
      this.positionService.disableByIds(ids);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据职位Id批量启用职位
   * @param ids 职位Id列表
   */
  @ApiOperation("根据职位Id批量启用职位")
  @PatchMapping("enableByIds")
  public ResponseModel enableByIds(@RequestParam @ApiParam("id数组") String[] ids) {
    try {
      this.positionService.enableByIds(ids);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据职位code批量绑定组织
   * @param codes 职位Id列表
   */
  @ApiOperation("根据职位Id批量启用职位")
  @PatchMapping("/reBindOrgCode")
  public ResponseModel reBindOrgCode(@RequestParam("codes") @ApiParam("职位编码") String[] codes, @RequestParam("orgCode") @ApiParam("组织编码") String orgCode) {
    try {
      this.positionService.reBindOrgCode(codes, orgCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("根据用户id查询所有下级职位")
  @GetMapping("/findByUserIdStrDescendant")
  public ResponseModel findByUserIdStrDescendant(@RequestParam @ApiParam("用户id") String userId){
    try{
      List<PositionVo> result = this.positionService.findByUserIdStruDescendant(userId);
      return this.buildHttpResultW(result, new String[]{"users"});
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("绑定父级职位")
  @PatchMapping("bindParent")
  public ResponseModel bindParent(@ApiParam("当前职位id") @RequestParam String id, @ApiParam("父级职位id") @RequestParam String parentId){
    try{
      PositionVo positionVo = this.positionService.bindParent(id, parentId);
      return this.buildHttpResult(positionVo);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
