package com.bizunited.platform.user.web.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.service.position.PositionLevelService;
import com.bizunited.platform.user.common.vo.PositionLevelVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;


/**
 * 职位级别控制层，restful风格接口
 * @author: weikangning
 * @Date: 2020/8/18 17:05
 */
@RestController
@RequestMapping(value = "/v1/nebula/positionLevels")
public class PositionLevelController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(PositionLevelController.class);

  @Autowired
  private PositionLevelService positionLevelService;

  @PostMapping("")
  @ApiOperation(value = "添加职位级别")
  public ResponseModel create(@ApiParam(name = "职位级别相关信息") @RequestBody PositionLevelVo positionLevel) {
    try {
      PositionLevelVo positionLevelVo = positionLevelService.create(positionLevel);
      return this.buildHttpResultW(positionLevelVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 修改职位级别
   * @param positionLevel
   * @return
   */
  @PatchMapping("")
  @ApiOperation(value = "修改职位级别")
  public ResponseModel update(@ApiParam(name = "职位级别相关信息") @RequestBody PositionLevelVo positionLevel) {
    try {
      PositionLevelVo positionLevelVo = positionLevelService.update(positionLevel);
      return buildHttpResultW(positionLevelVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 禁用一个或多个职位级别状态
   * @param ids
   * @return
   */
  @PatchMapping("disableByIds")
  @ApiOperation("禁用一个或多个职位级别状态")
  public ResponseModel disableByIds(@RequestParam @ApiParam("指定的一个或多个职位级别ID") String[] ids) {
    try {
      positionLevelService.disableByIds(ids);
      return buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 启用一个或多个职位级别状态
   * @param positionLevelIds
   * @return
   */
  @PatchMapping("enableByIds")
  @ApiOperation(value = "启用一个或多个职位级别状态")
  public ResponseModel enableByIds(@RequestParam @ApiParam("指定的一个或多个职位级别ID") String[] positionLevelIds) {
    try {
      positionLevelService.enableByIds(positionLevelIds);
      return buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @GetMapping("findByConditions")
  @ApiOperation(value = "按照条件搜索职位级别，名称可模糊查询，编码为精确查询")
  public ResponseModel findByConditions(
          @ApiParam("名称") @RequestParam(required = false) String name,
          @ApiParam("编码") @RequestParam(required = false) String code,
          @PageableDefault(value = 50) Pageable pageable) {
    try {
      Map<String, Object> conditions = new HashMap<>();
      conditions.put("code", code);
      conditions.put("name", name);
      InvokeParams invokeParams = new InvokeParams();
      invokeParams.putInvokeParams(conditions);
      Page<PositionLevelVo> page = positionLevelService.findByConditions(invokeParams, pageable);
      return this.buildHttpResultW(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ID查询职位级别
   * @param id
   * @return
   */
  @GetMapping("/{id}")
  @ApiOperation(value = "按照ID查询职位级别")
  public ResponseModel findById(@PathVariable("id") String id) {
    try {
      PositionLevelVo positionLevelVo = positionLevelService.findById(id);
      return this.buildHttpResultW(positionLevelVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @GetMapping("/findByCode")
  @ApiOperation(value = "按照职位级别编码查询职位级别")
  public ResponseModel findByCode(@ApiParam("编码") @RequestParam(value = "code") String code) {
    try {
      PositionLevelVo positionLevelVo = positionLevelService.findByCode(code);
      return this.buildHttpResultW(positionLevelVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @DeleteMapping("/deleteByIds")
  @ApiOperation(value = "逻辑删除一个或者多个职位级别")
  public ResponseModel deleteByIds(@ApiParam("职位级别ID数组") @RequestParam String[] ids) {
    try {
      positionLevelService.deleteByIds(ids);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("查询所有职级（未删除的）")
  @GetMapping("/findAll")
  public ResponseModel findAll(){
    try {
      List<PositionLevelVo> result = positionLevelService.findAll();
      return this.buildHttpResultW(result, new String[]{});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


}
