package com.bizunited.platform.user.web.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.security.Principal;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@Api("UserController")
@RestController
@RequestMapping("/v1/nebula/users")
public class UserController extends BaseController {

  @Autowired
  private UserService userService;

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(UserController.class);
  
  @ApiOperation("添加一个新的用户信息（后台管理人员，业务功能操作者）")
  @PostMapping
  public ResponseModel create(
      @ApiParam(name = "user", value = "用户对象信息", required = true) @RequestBody UserVo user) {
    try {
      UserVo currentPoliceMan = this.userService.create(user);
      return this.buildHttpResultW(currentPoliceMan);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改指定用户的密码信息
   * @param userId
   * @param newPassword
   * @return
   */
  @ApiOperation("修改指定用户的密码信息")
  @PostMapping("/updatePassword")
  public ResponseModel updatePassword(
          @ApiParam(name = "userId", value = "指定的用户", required = true) @RequestParam(name = "userId") String userId,
          @ApiParam(name = "newPassword", value = "新的密码（没有加密的）", required = true) @RequestParam(name = "newPassword") String newPassword) {
    try {
      UserVo currentUser = this.userService.updatePassword(userId , newPassword);
      return this.buildHttpResultW(currentUser);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改一个已经存在的用户信息", notes = "可以修改的包括属性只有：<br>" + "用户真实姓名、用户头像信息、入职时间、联系电话、性别（0保密，1男  2女）")
  @PatchMapping
  public ResponseModel update(
      @ApiParam(name = "user", value = "以对象方式传入的需要修改的属性，不能修改的属性即使传入该属性也不会发生变化", required = true) @RequestBody UserVo user) {
    try {
      UserVo currentUser = this.userService.update(user);
      return this.buildHttpResultW(currentUser);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "重置指定用户的密码信息")
  @PatchMapping("/updatePassword")
  public ResponseModel updatePassword(
      @ApiParam(name = "userId", value = "指定的用户", required = true) String userId,
      @ApiParam(name = "oldPassword", value = "原始密码（没有加密的）", required = true) String oldPassword,
      @ApiParam(name = "newPassword", value = "新的密码（没有加密的）", required = true) String newPassword) {
    try {
      UserVo currentUser = this.userService.updatePassword(userId , newPassword , oldPassword);
      return this.buildHttpResultW(currentUser);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("重置指定用户的密码信息（忽略原始密码，该方法一般由管理员使用）")
  @PatchMapping("/updatePasswordIgnoreOld")
  public ResponseModel updatePasswordIgnoreOld(
      @ApiParam(name = "userId", value = "指定的用户", required = true) String userId,
      @ApiParam(name = "newPassword", value = "新的密码（没有加密的）", required = true) String newPassword) {
    try {
      UserVo currentUser = this.userService.updatePassword(userId, newPassword);
      return this.buildHttpResultW(currentUser);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("修改指定的用户信息的状态，包括可用状态和不可用状态")
  @PatchMapping("/updateStatus")
  public ResponseModel updateStatus(
      @ApiParam(name = "account", value = "指定的新的用户账号", required = true) String account,
      @ApiParam(name = "flag", value = "指定的新的状态", required = true) boolean flag) {
    try {
      if(flag) {
        this.userService.enabled(account);
      } else {
        this.userService.disable(account);
      }      
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("用户启用")
  @PatchMapping("/enabled")
  public ResponseModel enabled(
          @ApiParam(name = "account", value = "指定的新的用户账号", required = true) String account) {
    try {
      this.userService.enabled(account);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("用户禁用")
  @PatchMapping("/disable")
  public ResponseModel disable(
          @ApiParam(name = "account", value = "指定的新的用户账号", required = true) String account) {
    try {
      this.userService.disable(account);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 成功登陆后，更新用户最后一次登录时间
   * @param account 按照用户账号更新
   */
  @ApiOperation("成功登陆后，更新用户最后一次登录时间")
  @PatchMapping("/updateLastloginTime")
  public ResponseModel updateLastloginTime(
          @ApiParam(name = "account", value = "指定的新的用户账号", required = true) String account) {
    try {
      this.userService.updateLastloginTime(account);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("查询当前用户信息，且包括当前用户直接绑定的用户组、组织机构和岗位信息。但是不包括用户角色，因为用户角色需要单独查")
  @GetMapping("/{userId}")
  public ResponseModel findDetailsById(
      @ApiParam(name = "userId", value = "指定的用户编号", required = true) @PathVariable("userId") String userId) {
    try {
      UserVo currentUser = this.userService.findDetailsById(userId);
      return this.buildHttpResultW(currentUser,"positions", "groups", "orgs");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation("查询当前用户信息(按照用户账号查询)，不包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位")
  @GetMapping("/findByAccount")
  public ResponseModel findByAccount(@ApiParam(name = "account", value = "指定的用户账户（肯定是唯一的）") @RequestParam("account") String account) {
    try {
      UserVo currentUser = this.userService.findByAccount(account);
      return this.buildHttpResultW(currentUser, "positions", "groups", "orgs");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation("查询当前已登录的操作者信息(按照用户账号查询)，不包括当前操作者已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位")
  @GetMapping("/findByPrincipal")
  public ResponseModel findByPrincipal(HttpServletRequest request , HttpServletResponse response) {
    try {
      Principal op = this.getPrincipal();
      UserVo currentUser = this.userService.findByAccount(op.getName());
      
      // 包括jsesion信息，也进行返回
      HttpSession session = request.getSession();
      String jsession = session.getId();
      response.addHeader("JSESSIONID", jsession);
      response.addHeader("persistence", jsession);
      response.addHeader("Access-Control-Expose-Headers","JSESSIONID,persistence");
      return this.buildHttpResultW(currentUser, "positions", "groups", "orgs");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  } 
  
  @ApiOperation("预制控件查询，暂时查询当前操作用户基础信息")
  @GetMapping("/findByUserId")
  public ResponseModel findByUserId(@ApiParam(name = "userId", value = "指定的用户id信息") @RequestParam("userId")String userId){
    try {
      UserVo creator = this.userService.findByUserId(userId);
      return this.buildHttpResultW(creator);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("按照条件搜索用户，支持模糊查询")
  @GetMapping("/findByConditions")
  public ResponseModel findByConditions(
      @ApiParam(name = "userVo", value = "用户信息") UserVo userVo,
      @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<UserVo> page = this.userService.findByConditions(userVo, pageable);
      return this.buildHttpResultW(page, "roles");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("查询多个用户信息（只包括用户的基本信息）")
  @GetMapping("/findByIds")
  public ResponseModel findByIds(@RequestParam(value = "id数组", defaultValue = "[]") String[] ids){
    try {
      Set<UserVo> users = this.userService.findByIds(Arrays.asList(ids));
      return this.buildHttpResultW(users);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("根据用户账号与名称模糊查询")
  @GetMapping("/findByAccountLikeOrNameLike")
  public ResponseModel findByAccountLikeOrNameLike(@RequestParam("param")String param){
    try {
      List<UserVo> users = this.userService.findByAccountLikeOrNameLike(param);
      return this.buildHttpResultW(users);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照指定的手机号，查询账户基本信息，该方法的查询不包括任何关联信息，例如角色信息。
   * @param phone 指定的用户手机号
   * @return
   */
  @ApiOperation("按照指定的手机号，查询账户基本信息，该方法的查询不包括任何关联信息，例如角色信息。")
  @GetMapping("/findByPhone")
  public ResponseModel findByPhone(@RequestParam("phone") String phone){
    try {
      UserVo user = this.userService.findByPhone(phone);
      return this.buildHttpResultW(user);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据组织机构ID查询绑定的用户
   * @param orgId
   * @return
   */
  @ApiOperation("根据组织机构ID查询绑定的用户")
  @GetMapping("findByOrgId")
  public ResponseModel findByOrgId(@RequestParam @ApiParam("组织机构ID") String orgId) {
    try {
      Set<UserVo> users = userService.findByOrgId(orgId);
      return buildHttpResultW(users);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据岗位id查询绑定的用户
   * @param positionId
   * @return
   */
  @ApiOperation("根据岗位id查询绑定的用户")
  @GetMapping("findByPositionId")
  public ResponseModel findByPositionId(@RequestParam @ApiParam("岗位编码") String positionId) {
    try {
      Set<UserVo> users = userService.findByPositionId(positionId);
      return buildHttpResultW(users);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation("根据用户id和获取下属的类型，查询指定用户所有的下属（子孙级）")
  @GetMapping("findByAccountAndTypeStruDescendant")
  public ResponseModel findByAccountAndTypeStruDescendant(@ApiParam("用户账号") @RequestParam String account,
                                                          @ApiParam("查询下属用户的方式，1:根据职位查询,0:查询所有情况的") @RequestParam(defaultValue = "0") Integer type){
    try{
      List<UserVo> userVos = this.userService.findByAccountAndTypeStruDescendant(account, type);
      return this.buildHttpResultW(userVos, new String[]{});
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
