package com.bizunited.platform.user.excel.process;

import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsImportBoxProcess;
import com.bizunited.platform.user.common.service.region.AdministrativeRegionService;
import com.bizunited.platform.user.common.vo.AdministrativeRegionVo;
import com.bizunited.platform.user.excel.vo.AdministrativeRegionImportVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * 行政区域导入处理实现
 * @Author: Paul Chan
 * @Date: 2020/10/29 13:51
 */
@Scope("prototype")
@Component("AdministrativeRegionImportProcess")
public class AdministrativeRegionImportProcess implements FormDetailsImportBoxProcess<AdministrativeRegionVo, AdministrativeRegionImportVo> {

  @Autowired
  private AdministrativeRegionService administrativeRegionService;

  /**
   * 暂存导入的行政区域信息，因为当前类是多例模式，所以每次导入的数据是隔离开的
   */
  private Map<String, AdministrativeRegionImportVo> importMap = new HashMap<>();
  /**
   * 已经存在库中的行政区域信息，K：行政区域编码
   */
  private Map<String, AdministrativeRegionVo> dbRegionMap = new HashMap<>();

  @Override
  public AdministrativeRegionVo process(AdministrativeRegionImportVo excelBean, Map<String, Object> params, String templateCode, String templateVersion) {
    try {
      this.validate(excelBean);
    } catch (RuntimeException e) {
      throw new ExcelMigrateException(e.getMessage(), e);
    }
    AdministrativeRegionVo region = new AdministrativeRegionVo();
    region.setRegionName(excelBean.getRegionName());
    region.setRegionCode(excelBean.getRegionCode());
    region.setLongitude(excelBean.getLongitude());
    region.setLatitude(excelBean.getLatitude());
    region.setEmail(excelBean.getEmail());
    String parentCode = excelBean.getParentCode();
    if(StringUtils.isNotBlank(parentCode)) {
      AdministrativeRegionVo parent = new AdministrativeRegionVo();
      parent.setRegionCode(parentCode);
      region.setParent(parent);
    }
    return region;
  }

  @Override
  public void execute(AdministrativeRegionVo region, Map<String, Object> params) {
    AdministrativeRegionVo parent = region.getParent();
    if(parent != null) {
      AdministrativeRegionVo dbParent = dbRegionMap.get(parent.getRegionCode());
      region.setParent(dbParent);
    }
    AdministrativeRegionVo dbRegion = administrativeRegionService.create(region);
    dbRegionMap.put(dbRegion.getRegionCode(), dbRegion);
  }

  /**
   * 导入前数据校验
   * @param excelBean
   */
  private void validate(AdministrativeRegionImportVo excelBean) {
    Validate.notNull(excelBean, "导入数据不能为空！");
    Validate.notBlank(excelBean.getRegionCode(), "编码不能为空！");
    Validate.notBlank(excelBean.getRegionName(), "名称不能为空！");
    Validate.notBlank(excelBean.getLongitude(), "经度不能为空！");
    Validate.notBlank(excelBean.getLatitude(), "纬度不能为空！");
    Validate.isTrue(!importMap.containsKey(excelBean.getRegionCode()), "重复的行政区域：%s", excelBean.getRegionCode());
    String parentCode = excelBean.getParentCode();
    if(StringUtils.isNotBlank(parentCode)) {
      AdministrativeRegionImportVo parent = importMap.get(parentCode);
      if(parent == null) {
        AdministrativeRegionVo dbParent = administrativeRegionService.findByCode(parentCode);
        Validate.notNull(dbParent, "未找到行政区域【%s】的上级区域：%s，请检查上级区域在数据库中或在当前组织表格的前置行中", excelBean.getRegionCode(), parentCode);
        dbRegionMap.put(parentCode, dbParent);
      }
    }
    importMap.put(excelBean.getRegionCode(), excelBean);
  }

}
