package com.bizunited.platform.user.excel.process;

import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsImportBoxProcess;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.excel.vo.OrganizationImportVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * 组织机构导入的处理类，实现的nebula 的导入框架,
 * 采用多例模式，方便实现自己的上下文
 * @Author: Paul Chan
 * @Date: 2020/10/27 14:45
 */
@Scope("prototype")
@Component("OrganizationImportProcess")
public class OrganizationImportProcess implements FormDetailsImportBoxProcess<OrganizationVo, OrganizationImportVo> {

  @Autowired
  private OrganizationService organizationService;

  /**
   * 暂存导入的组织机构信息，因为当前类是多例模式，所以每次导入的数据是隔离开的
   */
  private Map<String, OrganizationImportVo> orgImportMap = new HashMap<>(16);
  /**
   * 已经存在库中的组织机构信息，K：组织机构编码
   */
  private Map<String, OrganizationVo> dbOrgsMap = new HashMap<>(16);

  @Override
  public OrganizationVo process(OrganizationImportVo excelBean, Map<String, Object> params, String templateCode, String templateVersion) {
    try {
      this.validate(excelBean);
    } catch (RuntimeException e) {
      throw new ExcelMigrateException(e.getMessage(), e);
    }
    OrganizationVo org = new OrganizationVo();
    org.setCode(excelBean.getCode());
    org.setOrgName(excelBean.getOrgName());
    org.setDescription(excelBean.getDescription());
    org.setType(excelBean.getType());
    org.setLevel(excelBean.getLevel());
    String parentCode = excelBean.getParentCode();
    if(StringUtils.isNotBlank(parentCode)) {
      OrganizationVo parent = new OrganizationVo();
      parent.setCode(parentCode);
      org.setParent(parent);
    }
    return org;
  }

  @Override
  public void execute(OrganizationVo org, Map<String, Object> params) {
    OrganizationVo parent = org.getParent();
    if(parent != null) {
      OrganizationVo dbParent = dbOrgsMap.get(parent.getCode());
      org.setParent(dbParent);
    }
    OrganizationVo dbOrg = organizationService.create(org);
    dbOrgsMap.put(dbOrg.getCode(), dbOrg);
  }

  /**
   * 校验导入的excel数据
   * @param excelBean
   */
  private void validate(OrganizationImportVo excelBean) {
    Validate.notNull(excelBean, "导入数据不能为空!");
    Validate.notBlank(excelBean.getCode(), "组织编码不能为空!");
    Validate.notBlank(excelBean.getOrgName(), "组织名称不能为空!");
    Validate.notNull(excelBean.getType(), "组织机构类型不能为空");
    Validate.notNull(excelBean.getLevel(), "组织层级不能为空");
    Validate.isTrue(!orgImportMap.containsKey(excelBean.getCode()), "重复的组织机构：%s", excelBean.getCode());
    String parentCode = excelBean.getParentCode();
    if(StringUtils.isNotBlank(parentCode)) {
      OrganizationImportVo importParent = orgImportMap.get(parentCode);
      if(importParent == null) {
        OrganizationVo dbParent = organizationService.findByCode(parentCode);
        Validate.notNull(dbParent, "未找到组织【%s】的上级组织：%s，请检查上级组织在数据库中或在当前组织表格的前置行中", excelBean.getCode(), parentCode);
        dbOrgsMap.put(parentCode, dbParent);
      }
    }
    orgImportMap.put(excelBean.getCode(), excelBean);
  }

}
