package com.bizunited.platform.user.excel.process;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsImportBoxProcess;
import com.bizunited.platform.user.common.enums.UserGenderEnum;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.excel.vo.UserImportExcelVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_PASSWORD;

/**
 * 导入用户数据处理类
 *
 * @Author: hefan
 * @Date: 2020/11/4 21:32
 */
@Service
public class UserImportProcess implements FormDetailsImportBoxProcess<UserVo, UserImportExcelVo> {

  private String encodedPassword;

  @Autowired
  private UserService userService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  @Qualifier("passwordEncoder")
  @Lazy
  private PasswordEncoder passwordEncoder;

  @PostConstruct
  public void init() {
    encodedPassword = this.passwordEncoder.encode(DEFAULT_PASSWORD);
  }

  /**
   * 校验导入的数据，并转为VO
   *
   * @param excelBean
   * @param params
   * @param templateCode
   * @param templateVersion
   * @return
   */
  @Override
  public UserVo process(UserImportExcelVo excelBean, Map<String, Object> params, String templateCode, String templateVersion) {
    try {
      // 校验excelBean
      this.validateExcelBean(excelBean);
      // 把excel中提取的数据拷贝到PO
      return this.copyExcelToEntity(excelBean);
    } catch (Exception e) {
      throw new ExcelMigrateException(e.getMessage());
    }
  }

  /**
   * 保存导入数据
   *
   * @param userVo
   * @param params
   */
  @Override
  public void execute(UserVo userVo, Map<String, Object> params) {
    // 使用默认密码
    userVo.setPassword(encodedPassword);
    // 保存数据
    userService.create(userVo);
  }


  @Override
  public String importProcessName() {
    return "用户导入";
  }

  /**
   * 导入数据拷入VO
   *
   * @param excelBean
   * @return
   */
  private UserVo copyExcelToEntity(UserImportExcelVo excelBean) {
    // 性别是要转数字的 性别.0保密，1男 2女
    this.convertGender(excelBean);
    //同属性名拷贝
    return nebulaToolkitService.copyObjectByWhiteList(excelBean, UserVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 性别字段：文字转数字标识
   *
   * @param excelBean
   */
  private void convertGender(UserImportExcelVo excelBean) {
    String gender = excelBean.getGenderStr();
    UserGenderEnum userGenderEnum = UserGenderEnum.valueOfDesc(gender);
    excelBean.setGender(Objects.nonNull(userGenderEnum) ? userGenderEnum.getGender() : null);
  }

  /**
   * 校验导入数据
   *
   * @param excelBean
   */
  private void validateExcelBean(UserImportExcelVo excelBean) {
    // 1.账号唯一校验
    String account = excelBean.getAccount();
    // 根据account查询是否存在一条数据
    UserVo byAccount = userService.findByAccount(account);
    Validate.isTrue(Objects.isNull(byAccount), "账号：%s，已存在", account);
    // 2.手机号码唯一校验
    String phone = excelBean.getPhone();
    // phone可以为空白
    if (StringUtils.isNotBlank(phone)) {
      // 根据phone查询是否存在一条数据
      UserVo byPhone = userService.findByPhone(phone);
      Validate.isTrue(Objects.isNull(byPhone), "电话：%s，已存在", phone);
    }
  }


}
