package com.bizunited.platform.user.service.feign;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.interceptor.FeignRequestInterceptor;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.service.feign.fallback.OrganizationFeignClientFallback;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.security.Principal;
import java.util.List;

/**
 * 组织机构的feign远程调用
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:16
 */
@FeignClient(qualifier = "OrganizationFeignClient", name = "${user.application.name}", path = "/v1/nebula/orgs", fallback = OrganizationFeignClientFallback.class, configuration = FeignRequestInterceptor.class)
public interface OrganizationFeignClient {

  /**
   * 添加组织机构
   * @param organization 组织机构相关信息
   * @return
   */
  @PostMapping("")
  ResponseModel create(@RequestBody OrganizationVo organization);


  /**
   * 修改组织机构
   * @param organization 组织机构相关信息
   * @return
   */
  @PatchMapping("/update")
  ResponseModel update(@RequestBody OrganizationVo organization);

  /**
   * 修改组织机构状态---(启用、禁用)
   * @param orgId 组织机构id
   * @return
   */
  @PatchMapping("/updateStatus/{orgId}")
  ResponseModel updateStatus(@PathVariable("orgId") String orgId);

  /**
   * 修改组织机构状态---(启用、禁用)
   * @param orgIds 组织机构id
   * @return
   */
  @PatchMapping("/disables/{orgIds}")
  ResponseModel disables(@PathVariable("orgIds") String[] orgIds);

  /**
   * 修改组织机构状态---(启用、禁用)
   * @param orgIds 组织机构id
   * @return
   */
  @PatchMapping("/enables/{orgIds}")
  ResponseModel enables(@PathVariable("orgIds") String[] orgIds);

  /**
   * 对一个指定的currentOrgId进行其父级组织机构parentOrgId的绑定操作（原有currentOrgId的父级信息将会被取消）
   */
  @PatchMapping("/bindParent")
  ResponseModel bindParent(@RequestParam(name = "currentOrgId") String currentOrgId, @RequestParam(name = "parentOrgId") String parentOrgId);

  /**
   * 对一个指定的currentOrgId取消其当前父级关系的绑定信息
   */
  @PatchMapping("/unbindParent")
  ResponseModel unbindParent(@RequestParam(name = "currentOrgId") String currentOrgId);

  /**
   * 该服务方法可以在忽略当前组织机构（currentOrgId）已绑定父级机构的情况下，
   * 重新为当前组织机构绑定一个新的父级机构
   */
  @PatchMapping("/reBindParent")
  ResponseModel reBindParent(@RequestParam(name = "currentOrgId") String currentOrgId, @RequestParam(name = "parentOrgId") String parentOrgId);

  /**
   * 将指定组织机构（currentOrgId）和指定的一个/多个用户数据编号（userId）形成绑定关系——一个用户只能属于一个组织机构
   * @param currentOrgId
   * @param userIds      指定的一个/多个用户数据编号
   */
  @PatchMapping("/bindUsers")
  ResponseModel bindUsers(@RequestParam(name = "currentOrgId") String currentOrgId, @RequestParam(name = "userIds") String[] userIds);

  /**
   * 重新绑定组织机构的用户
   * @param currentOrgId
   * @param userIds
   * @return
   */
  @PatchMapping("/rebindUsers")
  ResponseModel rebindUsers(@RequestParam(name = "currentOrgId") String currentOrgId, @RequestParam(name = "userIds") String[] userIds);

  /**
   * 解除指定用户和指定组织机构的绑定关系
   * @param currentOrgId 指定的组织机构数据库编号
   * @param userId       指定的人员编号
   */
  @PatchMapping("/unbindUsers")
  ResponseModel unbindUsers(@RequestParam(name = "currentOrgId") String currentOrgId, @RequestParam(name = "userId") String[] userId);

  /**
   * 查询目前没有指定父级组织机构的所有组织机构信息，无论其状态是否正常（status无论什么值）
   * @return
   */
  @GetMapping("/findByNullParent")
  ResponseModel findByNullParent();

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，无论其状态是否正常（status无论什么值）
   * @param parentId
   * @return
   */
  @GetMapping("/findByParent")
  ResponseModel findByParent(@RequestParam(name = "parentId") String parentId);

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，并且匹配过滤条件，无论其状态是否正常（status无论什么值）
   * @return
   */
  @GetMapping("/findByParentAndFilterConditions")
  ResponseModel findByParentAndFilterConditions(@RequestParam(name = "parentId") String parentId,
                                                @RequestParam(name = "id") String id,
                                                @RequestParam(name = "code") String code,
                                                @RequestParam(name = "orgName") String orgName);

  /**
   * 查询目前没有指定父级组织机构的所有组织机构信息，并且依据其状态进行查询
   * @param status
   * @return
   */
  @GetMapping("/findByNullParentAndStatus")
  ResponseModel findByNullParentAndStatus(@RequestParam(name = "status") Integer status);

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，并且依据其状态进行查询
   * @param parentId
   * @param status
   * @return
   */
  @GetMapping("/findByParentAndStatus")
  ResponseModel findByParentAndStatus(@RequestParam(name = "parentId") String parentId, @RequestParam(name = "status") Integer status);

  /**
   * 根据组织机构名称，查询这个组织机构以及这个组织（这些组织机构）对应的所有父级、子级结构信息
   * @param orgName 组织机构名称信息
   * @return
   */
  @GetMapping("/findByOrgName")
  ResponseModel findByOrgName(@RequestParam(name = "orgName") String orgName);

  /**
   * 查询指定组织下的详细信息，无论其状态是否正常（status无论什么值）
   * @param id 组织机构id
   * @return
   */
  @GetMapping("/findDetailsById")
  ResponseModel findDetailsById(@RequestParam(name = "id") String id);

  /**
   * 查询指定组织下的详细信息，无论其状态是否正常（status无论什么值）
   * @param currentOrgId
   * @return
   */
  @GetMapping("/findUnbindOrgById")
  ResponseModel findUnbindOrgById(@RequestParam(name = "currentOrgId") String currentOrgId);

  /**
   * 查询所有的组织机构信息，无论其状态是否正常（status无论什么值）
   * @return
   */
  @GetMapping("/findAll")
  ResponseModel findAll();

  /**
   * 预制控件查询，查询用户组织机构相关信息
   * @param userId
   * @return
   */
  @GetMapping("/findOrgByUserId")
  ResponseModel findOrgByUserId(@RequestParam("userId") String userId);

  /**
   * 查询用户组织机构相关信息
   * @param ids
   * @return
   */
  @PostMapping("/findByIds")
  ResponseModel findByIds(@RequestBody List<String> ids);

  /**
   * 组织机构-树结构查询-根据指定组织机构ID，查询这个组织机构以及这个组织机构对应的所有子级结构信息
   *
   * @param id 当前组织机构Id
   * @return
   */
  @GetMapping("/findById")
  ResponseModel findById(@RequestParam(value = "id") String id);

  /**
   * 组织机构-根据组织机构名称实现模糊查询
   *
   * @param orgName 组织机构名称
   * @return
   */
  @GetMapping("/findByOrgNameLike")
  ResponseModel findByOrgNameLike(@RequestParam(name = "orgName") String orgName);

  /**
   * 根据组织编码查询
   *
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  ResponseModel findByCode(@RequestParam(name = "code") String code);

  /**
   * 根据指定的用户信息，查询这个用户所绑定的组织机构信息(一个或者多个)，以及这个组织机构的所有父级组织机构信息
   * 由于这里的构造是无限级别的，所以需要在方法内做递归处理，且直接返回json信息。<p>
   * 注意：这些组织机构的返回信息中，包括了组织机构已经绑定的角色信息。<p>
   * 一定注意该方法和RoleService.findByUserId(String userId)方法在实现逻辑上的区别。
   *
   * @return
   */
  @GetMapping("/findByUserId")
  ResponseModel findByUserId(@RequestParam(name = "userId") String userId);

  /**
   * 解除指定用户和任意组织机构的绑定关系
   *
   * @param userId
   */
  @GetMapping("/unbindUser")
  ResponseModel unbindUser(@RequestParam(name = "userId") String userId);

  /**
   * 获取组织的上级组织
   *
   * @param childId
   * @return
   */
  @GetMapping("/childId")
  ResponseModel findByChild(@RequestParam(name = "childId") String childId);

  /**
   * 获取用户的主组织机构
   * @param userId
   * @return
   */
  @GetMapping("findMainOrgByUserId")
  ResponseModel findMainOrgByUserId(@RequestParam("userId") String userId);

  /**
   * 更新用户的主组织机构
   * @param userId
   * @param orgId
   * @return
   */
  @PatchMapping("updateUserMainOrg")
  ResponseModel updateUserMainOrg(@RequestParam("userId") String userId, @RequestParam("orgId") String orgId);

  /**
   * 根据条件查询树结构
   * 返回的数据，除了返回匹配的数据外，还包含匹配数据的所有上级
   * @param code
   * @param orgName
   * @param tstatus
   * @return
   */
  @GetMapping("findTreeByConditions")
  ResponseModel findTreeByConditions(@RequestParam(value = "id", required = false) String id,
                                     @RequestParam(value = "code", required = false) String code,
                                     @RequestParam(value = "orgName", required = false) String orgName,
                                     @RequestParam(value = "tstatus", required = false) Integer tstatus);

  /**
   * 根据登陆人关联的组织查询
   * @param type
   * @return
   */
  @GetMapping("findByPrincipal")
  ResponseModel findByPrincipal(@RequestParam("type") Integer type, @RequestParam("principal") Principal principal);

  /**
   * 根据登陆人关联的组织和输入层级查询控件
   * @param type
   * @param account
   * @return
   */
  @GetMapping("findByTypeAndLevel")
  ResponseModel findByTypeAndLevel(@RequestParam("type") Integer type, @RequestParam("account") String account, @RequestParam("level") Integer level);

  /**
   * 组织合并，用指定组织id，去替换被合并的组织id（包括，他关联的子组织，职位，客户，终端），然后被合并的组织被停用
   * @param currentOrgId 指定的组织机构数据库编号
   * @param id       指定的被合并组织编号
   */
  @PatchMapping("/mergeOrgById")
  ResponseModel mergeOrgById(@RequestParam(name = "currentOrgId") String currentOrgId, @RequestParam(name = "id") String id);

  /**
   * 绑定指定组织机构的行政区域
   * @param orgId
   * @param regionIds
   * @return
   */
  @PatchMapping("bindRegions")
  ResponseModel bindRegions(@RequestParam("orgId") String orgId, @RequestParam("regionIds") String[] regionIds);

  /**
   * 重新绑定指定组织机构的行政区域
   * @param orgId
   * @param regionIds
   * @return
   */
  @PatchMapping("rebindRegions")
  ResponseModel rebindRegions(@RequestParam("orgId") String orgId, @RequestParam("regionIds") String[] regionIds);

  /**
   * 解绑指定组织机构的行政区域
   * @param orgId
   * @param regionIds
   * @return
   */
  @PatchMapping("unbindRegions")
  ResponseModel unbindRegions(@RequestParam("orgId") String orgId, @RequestParam("regionIds") String[] regionIds);

  /**
   * 组织机构-树结构查询-根据指定组织机构ids或者编码集合(必传任一)，查询这个组织机构以及这个组织机构对应的所有子级结构信息
   * @param ids
   * @param codes
   * @return
   */
  @GetMapping("/findChildrenByIdsOrCodes")
  ResponseModel findChildrenByIdsOrCodes(@RequestParam("ids")  String[] ids, @RequestParam("codes")  String[] codes);

  /**
   * 根据ID查询当前组织及当前组织的所有下级,以一维数组的格式返回数据
   * @param id
   * @return
   */
  @GetMapping("findByIdStruSelfAndDescendant")
  ResponseModel findByIdStruSelfAndDescendant(@RequestParam("id") String id);

  /**
   * 根据层级查询组织机构
   * @param level
   * @return
   */
  @GetMapping("findByLevel")
  ResponseModel findByLevel(@RequestParam("level") Integer level);

}
